package cn.quantgroup.config;

import cn.quantgroup.exceptionhandler.SentryExceptionHandler;
import io.sentry.DefaultSentryClientFactory;
import io.sentry.SentryClient;
import io.sentry.connection.AsyncConnection;
import io.sentry.connection.Connection;
import io.sentry.dsn.Dsn;
import io.sentry.event.helper.ContextBuilderHelper;
import io.sentry.event.helper.HttpEventBuilderHelper;
import lombok.extern.slf4j.Slf4j;

import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * @author ：dongjianhua
 * @date ：Created in 2021/5/14 10:03
 * @description：
 * @modified By：
 * @version:
 */
@Slf4j
public class ThrErrSentryClientFactory extends DefaultSentryClientFactory {

    private SentryExceptionHandler exceptionHandler;

    ThrErrSentryClientFactory(SentryExceptionHandler exceptionHandler) {
        this.exceptionHandler = exceptionHandler;
    }

    @Override
    public SentryClient createSentryClient(Dsn dsn) {
        SentryClient sentryClient = new ThrErrSentryClient(createConnection(dsn), getContextManager(dsn));
        try {
            // `ServletRequestListener` was added in the Servlet 2.4 API, and
            // is used as part of the `HttpEventBuilderHelper`, see:
            // https://tomcat.apache.org/tomcat-5.5-doc/servletapi/
            Class.forName("javax.servlet.ServletRequestListener", false, this.getClass().getClassLoader());
            sentryClient.addBuilderHelper(new HttpEventBuilderHelper());
        } catch (ClassNotFoundException e) {
            log.debug("The current environment doesn't provide access to servlets,"
                    + " or provides an unsupported version.");
        }
        sentryClient.addBuilderHelper(new ContextBuilderHelper(sentryClient));
        return configureSentryClient(sentryClient, dsn);
    }

    /**
     * Encapsulates an already existing connection in an {@link AsyncConnection} and get the async options from the
     * Sentry DSN.
     *
     * @param dsn        Data Source Name of the Sentry server.
     * @param connection Connection to encapsulate in an {@link AsyncConnection}.
     * @return the asynchronous connection.
     */
    @Override
    protected Connection createAsyncConnection(Dsn dsn, Connection connection) {

        int maxThreads = getAsyncThreads(dsn);
        int priority = getAsyncPriority(dsn);

        BlockingDeque<Runnable> queue;
        int queueSize = getAsyncQueueSize(dsn);
        if (queueSize == -1) {
            queue = new LinkedBlockingDeque<>();
        } else {
            queue = new LinkedBlockingDeque<>(queueSize);
        }

        ExecutorService executorService = new ThreadPoolExecutor(
                maxThreads, maxThreads, 0L, TimeUnit.MILLISECONDS, queue,
                new DaemonThreadFactory(priority), getRejectedExecutionHandler(dsn));

        boolean gracefulShutdown = getAsyncGracefulShutdownEnabled(dsn);

        long shutdownTimeout = getAsyncShutdownTimeout(dsn);
        ThrErrAsyncConnectionWarp warp = new ThrErrAsyncConnectionWarp(connection, exceptionHandler);
        return new AsyncConnection(warp, executorService, gracefulShutdown, shutdownTimeout);
    }

    @SuppressWarnings("PMD.AvoidThreadGroup")
    protected static final class DaemonThreadFactory implements ThreadFactory {
        private static final AtomicInteger POOL_NUMBER = new AtomicInteger(1);
        private final ThreadGroup group;
        private final AtomicInteger threadNumber = new AtomicInteger(1);
        private final String namePrefix;
        private final int priority;

        private DaemonThreadFactory(int priority) {
            SecurityManager s = System.getSecurityManager();
            group = (s != null) ? s.getThreadGroup() : Thread.currentThread().getThreadGroup();
            namePrefix = "sentry-pool-" + POOL_NUMBER.getAndIncrement() + "-thread-";
            this.priority = priority;
        }

        @Override
        public Thread newThread(Runnable r) {
            Thread t = new Thread(group, r, namePrefix + threadNumber.getAndIncrement(), 0);
            if (!t.isDaemon()) {
                t.setDaemon(true);
            }
            if (t.getPriority() != priority) {
                t.setPriority(priority);
            }
            return t;
        }
    }
}
