package cn.quantgroup.config;

import io.sentry.SentryClient;
import io.sentry.connection.Connection;
import io.sentry.connection.LockedDownException;
import io.sentry.context.ContextManager;
import io.sentry.event.Event;
import io.sentry.event.helper.ShouldSendEventCallback;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashSet;
import java.util.Set;

/**
 * @author ：dongjianhua
 * @date ：Created in 2021/5/14 10:01
 * @description：
 * @modified By：
 * @version:
 */
@Slf4j
public class ThrErrSentryClient extends SentryClient {
    private static final Logger lockdownLogger = LoggerFactory.getLogger(ThrErrSentryClient.class.getName() + ".lockdown");

    /**
     * The underlying {@link Connection} to use for sending events to Sentry.
     */
    private final Connection connection;

    /**
     * Set of callbacks that are checked before each {@link Event} is sent to Sentry.
     */
    private final Set<ShouldSendEventCallback> shouldSendEventCallbacks = new HashSet<>();

    /**
     * Constructs a {@link SentryClient} instance using the provided connection.
     * <p>
     * Note that the most recently constructed instance is stored statically so it can be used with
     * the static helper methods.
     *
     * @param connection     Underlying {@link Connection} instance to use for sending events
     * @param contextManager {@link ContextManager} instance to use for storing contextual data
     */
    ThrErrSentryClient(Connection connection, ContextManager contextManager) {
        super(connection, contextManager);
        this.connection = connection;
    }

    /**
     * Add a callback that is called before an {@link Event} is sent to Sentry.
     *
     * @param shouldSendEventCallback callback instance
     */
    @Override
    public void addShouldSendEventCallback(ShouldSendEventCallback shouldSendEventCallback) {
        shouldSendEventCallbacks.add(shouldSendEventCallback);
    }

    /**
     * Sends a built {@link Event} to the Sentry server.
     *
     * @param event event to send to Sentry.
     */
    @Override
    public void sendEvent(Event event) {
        for (ShouldSendEventCallback shouldSendEventCallback : shouldSendEventCallbacks) {
            if (!shouldSendEventCallback.shouldSend(event)) {
                log.trace("Not sending Event because of ShouldSendEventCallback: {}", shouldSendEventCallback);
                return;
            }
        }

        try {
            connection.send(event);
        } catch (LockedDownException e) {
            lockdownLogger.warn("The connection to Sentry is currently locked down.", e);
        } catch (Exception e) {
            log.error("An exception occurred while sending the event to Sentry.", e);
        } finally {
            getContext().setLastEventId(event.getId());
        }
    }
}
