package cn.quantgroup.cashloanflowboss.api.order.service;
import cn.quantgroup.cashloanflowboss.spi.clf.entity.CallbackFailRecord;
import cn.quantgroup.cashloanflowboss.spi.clf.model.LoanProgress;
import cn.quantgroup.cashloanflowboss.spi.clf.model.CallbackRouter;
import cn.quantgroup.cashloanflowboss.spi.clf.model.KANoticeType;

import cn.quantgroup.cashloanflowboss.api.channel.entity.ChannelConf;
import cn.quantgroup.cashloanflowboss.api.channel.repository.ChannelConfRepository;
import cn.quantgroup.cashloanflowboss.api.channel.util.ChannelConfUtil;
import cn.quantgroup.cashloanflowboss.api.order.entity.OrderApprove;
import cn.quantgroup.cashloanflowboss.api.order.model.*;
import cn.quantgroup.cashloanflowboss.api.order.repository.OrderApproveRepository;
import cn.quantgroup.cashloanflowboss.api.order.util.OrderUtil;
import cn.quantgroup.cashloanflowboss.core.Application;
import cn.quantgroup.cashloanflowboss.core.base.ServiceResult;
import cn.quantgroup.cashloanflowboss.core.base.Tuple;
import cn.quantgroup.cashloanflowboss.spi.clf.entity.CallbackRecord;
import cn.quantgroup.cashloanflowboss.spi.clf.entity.ClfOrderMapping;
import cn.quantgroup.cashloanflowboss.spi.clf.model.CallbackRecordVoModel;
import cn.quantgroup.cashloanflowboss.spi.clf.repository.ClfOrderMappingRepository;
import cn.quantgroup.cashloanflowboss.spi.clf.service.CLFCenterService;
import cn.quantgroup.cashloanflowboss.spi.clotho.service.ClothoCenterService;
import cn.quantgroup.cashloanflowboss.spi.jolyne.JolyneCenter;
import cn.quantgroup.cashloanflowboss.spi.user.service.XyqbUserService;
import cn.quantgroup.cashloanflowboss.spi.xyqb.entity.Contract;
import cn.quantgroup.cashloanflowboss.spi.xyqb.entity.FundLending;
import cn.quantgroup.cashloanflowboss.spi.xyqb.service.XYQBCenterService;
import cn.quantgroup.cashloanflowboss.utils.JSONTools;
import cn.quantgroup.user.retbean.XUser;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.ConcurrentMap;
import javax.persistence.criteria.Predicate;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * function:
 * date: 2019/8/8
 *
 * @author: suntao
 */

@Slf4j
@Service
public class OrderService {

    @Autowired
    private ChannelConfRepository channelConfRepository;
    @Autowired
    private OrderApproveRepository orderApproveRepository;


    @Autowired
    private XyqbUserService xyqbUserService;
    @Autowired
    private XYQBCenterService xyqbCenterService;
    @Autowired
    private ClothoCenterService clothoCenterService;
    @Autowired
    private CLFCenterService clfCenterService;
    @Autowired
    private JolyneCenter jolyneCenter;





    @Autowired
    private ClfOrderMappingRepository clfOrderMappingRepository;




    public Page<OrderVo> orderList(Long channelId, String channelOrderNo, Integer pageNumber, Integer pageSize) {
        Page<ClfOrderMapping> page = this.clfOrderMappingRepository.findAll((root, criteriaQuery, criteriaBuilder) -> {

            List<Predicate> predicates = new ArrayList<>();

            // 指定渠道号
            if (Objects.nonNull(channelId)) {
                predicates.add(criteriaBuilder.equal(root.get("registeredFrom"), channelId.longValue()));
            }
            if (StringUtils.isNotEmpty(channelOrderNo)) {
                predicates.add(criteriaBuilder.equal(root.get("channelOrderNo"), channelOrderNo));
            }

            // 设置查询条件
            criteriaQuery.where(criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()])));
            // 指定排序
            criteriaQuery.orderBy(criteriaBuilder.desc(root.get("id")));

            return criteriaQuery.getRestriction();

        }, new PageRequest(pageNumber, pageSize));


        return page.map(it -> {
            OrderVo orderVo = new OrderVo();
            orderVo.setId(it.getId());
            orderVo.setChannelId(it.getRegisteredFrom());
            orderVo.setChannelOrderNumber(it.getChannelOrderNo());
            if (!Application.getPrincipal().isChannel()) {
                orderVo.setLoanId(it.getLoanId());
                // 授信申请号，非渠道权限登陆显示admin及以上权限用户
                orderVo.setApplyNo(it.getApplyNo());
            }
            orderVo.setCreatedAt(it.getCreatedAt());
            if (StringUtils.isNotEmpty(it.getApplyNo())) {

                OrderApprove orderApprove = orderApproveRepository.findByCreditNumber(it.getApplyNo());

                ServiceResult<XyqbCurrentOrderStatusServiceResultModel> xyqbCenterServiceXyqbOrderStatus = xyqbCenterService.getXyqbOrderStatus(it.getApplyNo(), it.getLoanId());
                log.info("[xyqbCenterService.getXyqbOrderStatus]xyqbOrderStatus={}", JSONTools.serialize(xyqbCenterServiceXyqbOrderStatus));
                if (xyqbCenterServiceXyqbOrderStatus.isSuccess()) {
                    QueryXyqbOrderStatus currentStatus = xyqbCenterServiceXyqbOrderStatus.getData().getCurrentStatus();

                    Tuple<String, List<OrderVo.OptButton>> currentStatusAndButtons = OrderUtil.getCurrentStatusAndButtons(currentStatus, orderApprove);
                    orderVo.setStatus(currentStatusAndButtons.getKey());
                    orderVo.setOpt(currentStatusAndButtons.getValue());
                    orderVo.setMessage("");
                } else {
                    orderVo.setMessage("订单查询错误");
                }
            } else {
                orderVo.setStatus("授信中");
                orderVo.setOpt(new ArrayList<>());
                orderVo.setMessage("");
            }

            return orderVo;
        });
    }

    /**
     *
     * @param approveVo
     * @return Tuple<Boolean 审批结果 true 审批完成 false 审批操作失败, Boolean 资产计划配置 true 配置没问题 false 配置需要检查>
     */
    public Tuple<Boolean, Boolean> approveOpt(ApproveVo approveVo) {

        if (StringUtils.isEmpty(approveVo.getAmount())) {
            // 如果UI 金额为空 默认4000
            approveVo.setAmount("4000");
        }
        if (approveVo.getPeriod() == null) {
            // 如果UI 期数为空 3期
            approveVo.setPeriod(3);
        }

        ClfOrderMapping orderMapping = clfOrderMappingRepository.findByChannelOrderNoLastOne(approveVo.getChannelOrderNumber());
        if (orderMapping == null) {
            log.info("approveOpt,审批失败，无订单 channelOrderNumber={}", approveVo.getChannelOrderNumber());
            return new Tuple<>(false, false);
        }

        if (Application.getPrincipal().isChannel() && !Application.getPrincipal().isSameChannel(orderMapping.getRegisteredFrom())) {
            log.info("approveOpt,审批失败，不是该渠道订单无法审批 channelOrderNumber={}", approveVo.getChannelOrderNumber());
            return new Tuple<>(false, false);
        }

        XUser xUser = xyqbUserService.findXUserById(orderMapping.getQgUserId());
        if (xUser == null) {
            log.info("approveOpt,审批失败，未找到用户 channelOrderNumber={}", approveVo.getChannelOrderNumber());
            return new Tuple<>(false, false);
        }

        ChannelConf channelConf = channelConfRepository.getByChannelId(orderMapping.getRegisteredFrom());
        if (channelConf == null) {
            log.info("approveOpt,审批失败，boss渠道配置为空 channelOrderNumber={}", approveVo.getChannelOrderNumber());
            return new Tuple<>(false, false);
        }

        // 额度有效期
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(Calendar.DAY_OF_YEAR, 7);


        Integer fundId = ChannelConfUtil.getFundIdByType(approveVo.getFundType(), channelConf);
        // 资方 及 期数额度规则
        String fundFormat = String.format(OrderUtil.financeProductsFormat, approveVo.getAmount(), approveVo.getAmount(),
                approveVo.getPeriod(), fundId);


        OrderApprove orderApprove = new OrderApprove();
        orderApprove.setChannelOrderNumber(approveVo.getChannelOrderNumber());
        orderApprove.setCreditNumber(orderMapping.getApplyNo());
        orderApprove.setChannelId(orderMapping.getRegisteredFrom());
        orderApprove.setFundId(fundId);
        orderApprove.setFundType(approveVo.getFundType());
        orderApprove.setCreateTime(new Date());
        orderApprove.setUpdateTime(new Date());
        orderApproveRepository.save(orderApprove);


        Map<String, Object> data = new HashMap<>(16);
        data.put("code", 0);
        data.put("msg", "success");
        data.put("bizChannel", orderMapping.getRegisteredFrom());
        data.put("uuid", xUser.getUuid());
        data.put("bizNo", orderMapping.getApplyNo());
        data.put("bizType", channelConf.getBizType());
        data.put("auditResult", approveVo.getIsPass());
        data.put("amount", approveVo.getAmount());
        data.put("deadLine", calendar.getTime().getTime());
        data.put("financeProducts", fundFormat);

        // 发起审批
        String approveStringResult = clothoCenterService.approve(data);
        Boolean corpPolicyValidate = false;
        try {
            corpPolicyValidate = xyqbCenterService.validateCorpPolicy(Long.valueOf(fundId), new Date());
        } catch (Exception e) {
            log.error("审批,资方校验异常，channelOrderNumber={}，fundId={}, e={}", orderMapping.getChannelOrderNo(), fundId, ExceptionUtils.getStackTrace(e));
        }
        if ("success".equals(approveStringResult)) {
            log.info("审批申请成功，channelOrderNumber={}", orderMapping.getChannelOrderNo());
            return new Tuple<>(true, corpPolicyValidate);
        } else {
            log.info("审批申请失败，channelOrderNumber={}，requestParam={}", orderMapping.getChannelOrderNo(), JSONTools.serialize(data));
            return new Tuple<>(false, corpPolicyValidate);
        }

    }

    /**
     * 放款
     *
     * @param lendingFormModel
     * @return
     */
    @Transactional
    public boolean lending(LendingFormModel lendingFormModel) {

        boolean result = false;

        OrderApprove orderApprove = this.orderApproveRepository.findByChannelOrderNumber(lendingFormModel.getChannelOrderNumber());

        ClfOrderMapping orderMapping = this.clfOrderMappingRepository.findByChannelOrderNoAndRegisteredFromLastOne(lendingFormModel.getChannelOrderNumber(), lendingFormModel.getChannelId());

        // 更新合同状态
        Contract conscont= xyqbCenterService.findContractByUserId(orderMapping.getQgUserId());
        if (conscont != null) {
            conscont.setGenerateStatus(2);
            xyqbCenterService.saveContract(conscont);
        } else {
            // 合同为空 log表记录问题，UI用户查询
        }

        // 更新待放款时间
        FundLending fundLending = xyqbCenterService.findFundLendingByLoanId(orderMapping.getLoanId());
        if (fundLending != null) {
            LocalDateTime localDateTime = LocalDateTime.now();
            localDateTime.plusMinutes(-5L);
            fundLending.setCreateTime(new Date(localDateTime.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli()));
            xyqbCenterService.saveFundLending(fundLending);
        }

        if (orderApprove.getFundType() == 0) {
            // 如果是非存管
            result = this.xyqbCenterService.payResultNotify(orderMapping.getLoanId(), lendingFormModel.getResult());
        } else {
            // p2p

            this.xyqbCenterService.payResultNotify(orderMapping.getLoanId(), lendingFormModel.getResult());
            Map data = this.xyqbCenterService.queryLendingRecordCount(orderApprove.getFundId(), 0);
            if (data.size() > 0) {
                if (Objects.nonNull(data.get("totalAmount")) && Objects.nonNull(data.get("totalCount"))) {
                    result = this.clothoCenterService.lending(orderApprove.getFundId(), new BigDecimal(String.valueOf(data.get("totalAmount"))), Integer.valueOf(String.valueOf(data.get("totalCount"))));
                }
            }

            result = true;
        }

        return result;

    }

    public boolean cancel(OrderVo orderVo) {
        ClfOrderMapping orderMapping = clfOrderMappingRepository.findByChannelOrderNoLastOne(orderVo.getChannelOrderNumber());
        if (orderMapping == null) {
            log.info("cancel,关单失败，无订单 channelOrderNumber={}", orderVo.getChannelOrderNumber());
            return false;
        }

        XUser xUser = xyqbUserService.findXUserById(orderMapping.getQgUserId());
        if (xUser == null) {
            log.info("cancel,关单失败，未找到用户 channelOrderNumber={}", orderVo.getChannelOrderNumber());
            return false;
        }
        Long userId = xUser.getId();
        ConcurrentMap<Object, Object> data = Maps.newConcurrentMap();
        ArrayList<Object> cancel_list = Lists.newArrayList();
        cancel_list.add("update xyqb.quota_credit set is_active=0 where user_id=" + userId);
        cancel_list.add("update xyqb.quota_account set is_active=0 where user_id=" + userId);
        cancel_list.add("update xyqb.loan_application_history set progress=16 where user_id=" + userId);
        cancel_list.add("update xyqb.loan_application_history set is_active=0 where user_id=" + userId);
        cancel_list.add("delete from apply_quota_record where user_id=" + userId);
        cancel_list.add("delete from user_operation_history where user_id=" + userId);
        data.put("sql", cancel_list);
        String cancel_result = jolyneCenter.cancel(JSONTools.serialize(data));
        return "success".equals(cancel_result);
    }

    public HistoryOrderStatusVoModel getOrderHistoryStatus(String channelOrderNumber) {
        ClfOrderMapping orderMapping = clfOrderMappingRepository.findByChannelOrderNoLastOne(channelOrderNumber);
        HistoryOrderStatusVoModel historyOrderStatusVoModel = new HistoryOrderStatusVoModel();
        if (orderMapping == null) {
            return historyOrderStatusVoModel;
        }
        OrderApprove orderApprove = orderApproveRepository.findByCreditNumber(orderMapping.getApplyNo());

        ServiceResult<XyqbHistoryOrderStatusServiceResultModel> xyqbOrderHistoryStatusResult = xyqbCenterService.getXyqbOrderHistoryStatus(orderMapping.getApplyNo(), orderMapping.getLoanId());
        if (xyqbOrderHistoryStatusResult.isSuccess()) {
            log.info("[getOrderHistoryStatus]查询成功channelOrderNumber={}, xyqbOrderHistoryStatusResult={}", channelOrderNumber, JSONTools.serialize(xyqbOrderHistoryStatusResult));
            XyqbHistoryOrderStatusServiceResultModel data = xyqbOrderHistoryStatusResult.getData();
            List<VoOrderStatus> historyStatusList = new ArrayList<>();
            for (QueryXyqbOrderStatus queryXyqbOrderStatus : data.getHistoryStatus()) {
                VoOrderStatus voOrderStatus = new VoOrderStatus();
                Tuple<String, List<OrderVo.OptButton>> oneStatus = OrderUtil.getCurrentStatusAndButtons(queryXyqbOrderStatus, orderApprove);
                voOrderStatus.setStatus(oneStatus.getKey());
                voOrderStatus.setUpdateTime(new Date(queryXyqbOrderStatus.getUpdateTime()));
                historyStatusList.add(voOrderStatus);
            }
            historyOrderStatusVoModel.setHistoryStatus(historyStatusList);
            return historyOrderStatusVoModel;
        } else {
            log.info("[getOrderHistoryStatus]查询失败channelOrderNumber={}, xyqbOrderHistoryStatusResult={}", channelOrderNumber, JSONTools.serialize(xyqbOrderHistoryStatusResult));
            return historyOrderStatusVoModel;
        }
    }

    /**
     * 贷前关单
     *
     * @param
     * @return
     */
    public boolean cancelPreLoan(String channelOrderNumber) {

        /*Order order = this.orderRepository.findOrderByChannelOrderNumber(channelOrderNumber);

        ClfOrderMapping orderMapping = this.clfOrderMappingRepository.findByChannelOrderNoLastOne(channelOrderNumber);

        if (orderMapping == null) {
            log.info("cancelPreLoan,贷前关单失败，无订单 channelOrderNumber={}", channelOrderNumber);
            return false;
        }*/

        Long id = xyqbCenterService.cancelPreLoan(channelOrderNumber);

        ConcurrentMap<Object, Object> data = Maps.newConcurrentMap();
        data.put("data", id);
        data.put("remark", "系统关单");
        String result = clothoCenterService.cancelPreLoan(data);
        return true;
    }

    /**
     * 贷后关单
     *
     * @param
     * @return
     */
    public boolean cancelAfterLoan(String channelOrderNumber) {

        ClfOrderMapping orderMapping = this.clfOrderMappingRepository.findByChannelOrderNoLastOne(channelOrderNumber);

        if (orderMapping == null) {
            log.info("cancelAfterLoan,贷后关单失败，无订单 channelOrderNumber={}", channelOrderNumber);
            return false;
        }
        if (orderMapping.getLoanId() == null) {
            log.info("cancelAfterLoan,贷后关单失败，loanId为空，channelOrderNumber={}", channelOrderNumber);
            return false;
        }
        ConcurrentMap<Object, Object> data = Maps.newConcurrentMap();
        data.put("loanId", orderMapping.getLoanId());
        data.put("userId", orderMapping.getQgUserId());
        data.put("flowNo", UUID.randomUUID());
        String id = clothoCenterService.cancelAfterLoan(data);

        return true;
    }

    /**
     * 查询订单历史推送记录
     * @param channelOrderNumber
     * @return
     */
    public List<CallbackRecordVoModel> getOrderCallbackRecordList(String channelOrderNumber) {

        List<CallbackRecordVoModel> callbackRecordVoModelList = new ArrayList<>();

        ClfOrderMapping orderMapping = clfOrderMappingRepository.findByChannelOrderNoLastOne(channelOrderNumber);
        if (orderMapping == null) {
            return null;
        }

        List<CallbackRecord> callbackRecordList = clfCenterService.findCallbackRecordByApplyNo(orderMapping.getApplyNo());
        if (CollectionUtils.isNotEmpty(callbackRecordList)) {
            for (CallbackRecord callbackRecord : callbackRecordList) {
                CallbackRecordVoModel callbackRecordVoModel = new CallbackRecordVoModel();
                callbackRecordVoModel.setChannelOrderNo(orderMapping.getChannelOrderNo());
                callbackRecordVoModel.setApplyNo(orderMapping.getApplyNo());
                callbackRecordVoModel.setLoanId(orderMapping.getLoanId());
                callbackRecordVoModel.setTermNo(callbackRecord.getTermNo());
                callbackRecordVoModel.setNoticeType(callbackRecord.getNoticeType());
                callbackRecordVoModel.setCallbackStatus(callbackRecord.getCallbackStatus());
                callbackRecordVoModel.setChannelId(callbackRecord.getChannelId());
                callbackRecordVoModel.setCreatedAt(callbackRecord.getCreatedAt());
                callbackRecordVoModel.setUpdatedAt(callbackRecord.getUpdatedAt());
                callbackRecordVoModel.setTableName("Success");
                callbackRecordVoModelList.add(callbackRecordVoModel);
            }
        }

        List<CallbackFailRecord> callbackFailRecordList = clfCenterService.findCallbackFailRecordByApplyNo(orderMapping.getApplyNo());
        if (CollectionUtils.isNotEmpty(callbackFailRecordList)) {
            for (CallbackFailRecord callbackFailRecord : callbackFailRecordList) {
                CallbackRecordVoModel callbackFailRecordVoModel = new CallbackRecordVoModel();
                callbackFailRecordVoModel.setChannelOrderNo(orderMapping.getChannelOrderNo());
                callbackFailRecordVoModel.setApplyNo(orderMapping.getApplyNo());
                callbackFailRecordVoModel.setLoanId(orderMapping.getLoanId());
                callbackFailRecordVoModel.setTermNo(callbackFailRecord.getTermNo());
                callbackFailRecordVoModel.setNoticeType(callbackFailRecord.getNoticeType());
                callbackFailRecordVoModel.setCallbackStatus(callbackFailRecord.getCallbackStatus());
                callbackFailRecordVoModel.setChannelId(callbackFailRecord.getChannelId());
                callbackFailRecordVoModel.setRetryTimes(callbackFailRecord.getRetryTimes());
                callbackFailRecordVoModel.setFailCode(callbackFailRecord.getFailCode());
                callbackFailRecordVoModel.setCallbackRouter(callbackFailRecord.getCallbackRouter());
                callbackFailRecordVoModel.setIsActive(callbackFailRecord.getIsActive());
                callbackFailRecordVoModel.setExdata(callbackFailRecord.getExdata());
                callbackFailRecordVoModel.setCreatedAt(callbackFailRecord.getCreatedAt());
                callbackFailRecordVoModel.setUpdatedAt(callbackFailRecord.getUpdatedAt());
                callbackFailRecordVoModel.setTableName("Fail");
                callbackRecordVoModelList.add(callbackFailRecordVoModel);
            }
        }
        return callbackRecordVoModelList;
    }
}
