package cn.quantgroup.cashloanflowboss.api.user.controller;

import cn.quantgroup.cashloanflowboss.api.user.entity.User;
import cn.quantgroup.cashloanflowboss.api.user.model.Pagination;
import cn.quantgroup.cashloanflowboss.api.user.model.RegisterUserFormModel;
import cn.quantgroup.cashloanflowboss.api.user.model.UserDetailInfo;
import cn.quantgroup.cashloanflowboss.api.user.model.UserInfoModel;
import cn.quantgroup.cashloanflowboss.api.user.service.UserService;
import cn.quantgroup.cashloanflowboss.api.user.service.UserServiceImpl;
import cn.quantgroup.cashloanflowboss.component.security.annotiation.Security;
import cn.quantgroup.cashloanflowboss.component.validator.constraints.NotEmpty;
import cn.quantgroup.cashloanflowboss.core.base.Result;
import cn.quantgroup.cashloanflowboss.core.base.Tuple;
import cn.quantgroup.cashloanflowboss.spi.user.service.XyqbUserService;

import javax.validation.Valid;

import cn.quantgroup.cashloanflowboss.spi.xyqb.service.XYQBCenterService;
import cn.quantgroup.cashloanflowboss.spi.xyqb.service.XYQBCenterServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * Created by WeiWei on 2019/7/22.
 */
@RestController
@RequestMapping("user")
public class UserController {

    @Autowired
    private UserService userService;
    @Autowired
    private XyqbUserService xyqbUserService;


    /**
     * 注册用户
     *
     * @param registerUserFormModel
     * @return
     */
    @PostMapping()
    public Boolean createUser(@RequestBody @Valid RegisterUserFormModel registerUserFormModel) {
        return this.userService.createUser(registerUserFormModel.getUsername(), registerUserFormModel.getPassword());
    }

    /**
     * 添加用户
     *
     * @param userInfoModel
     * @return
     */
    @Security(authorityId = "User.addUser")
    @PostMapping("saveUserInfo")
    public Result<User> saveUserInfo(@RequestBody UserInfoModel userInfoModel) {
        User info = this.userService.saveUserInfo(userInfoModel);
        return Result.buildSuccess(info);
    }

    /**
     * 获取用户列表（分页）
     *
     * @return
     */
    @GetMapping("/list")
    public Page<User> queryUsers(@RequestParam @Valid Pagination pagination) {
        return this.userService.getUsers(pagination.getPageNumber(), pagination.getPageSize());
    }

    /**
     * 启用用户
     *
     * @param id 用户ID
     * @return
     */
    @PutMapping("/enable")
    @Security(authorityId = "User.enable")
    public Boolean enableUser(@RequestParam @Valid @NotEmpty(message = "无效的用户ID") String id) {
        return this.userService.enableUser(id);
    }

    /**
     * 禁用用户
     *
     * @param id 用户ID
     * @return
     */
    @PutMapping("/disable")
    @Security(authorityId = "User.disable")
    public Boolean disableUser(@RequestParam @Valid @NotEmpty(message = "无效的用户ID") String id) {
        return this.userService.disableUser(id);
    }

    /**
     * 移除用户
     *
     * @param id 用户ID
     * @return
     */
    @DeleteMapping()
    @Security(authorityId = "User.delete")
    public Boolean removeUser(@RequestParam @Valid @NotEmpty(message = "无效的用户ID") String id) {
        return this.userService.removeUser(id);
    }

    /**
     * 获取用户详情
     *
     * @return
     */
    @GetMapping("/detail/info")
    public Result<UserDetailInfo> userDetailInfo() {
        return Result.buildSuccess(userService.getUserDetailInfo());
    }

    /**
     * 清除用户活跃订单
     *
     * @param mobile 用户ID
     * @return
     */
    @PutMapping("/order/clean")
    @Security(authorityId = "User.cleanUserOrder")
    public Result<Boolean> cleanUserOrder(@RequestParam @Valid @NotEmpty(message = "无效的用户手机号") String mobile) {
        Tuple<Boolean, String> result = userService.cleanUserActiveOrder(mobile);
        return Result.buildSuccess(result.getKey(), result.getValue());
    }

    /**
     * 清除用户信息
     *
     * @param mobile 用户ID
     * @return
     */
    @PutMapping("/info/clean")
    @Security(authorityId = "User.cleanUserInfo")
    public Boolean deleteByUserId(@RequestParam @Valid @NotEmpty(message = "无效的用户手机号") String mobile) {
        Tuple<Boolean, String> result = this.xyqbUserService.deleteByUserId(mobile);
        return result.getKey();
    }

}
