package cn.quantgroup.cashloanflowboss.api.user.service;

import cn.quantgroup.cashloanflowboss.api.login.model.Principal;
import cn.quantgroup.cashloanflowboss.api.optlog.model.OptEnumName;
import cn.quantgroup.cashloanflowboss.api.user.dictionary.UserStatus;
import cn.quantgroup.cashloanflowboss.api.user.entity.User;
import cn.quantgroup.cashloanflowboss.api.user.model.QueryUserListModel;
import cn.quantgroup.cashloanflowboss.api.user.model.UserDetailInfo;
import cn.quantgroup.cashloanflowboss.api.user.model.UserInfoModel;
import cn.quantgroup.cashloanflowboss.api.user.repository.UserRepository;
import cn.quantgroup.cashloanflowboss.core.Application;
import cn.quantgroup.cashloanflowboss.core.annotation.opt.OperationAnno;
import cn.quantgroup.cashloanflowboss.core.asserts.Assert;
import cn.quantgroup.cashloanflowboss.core.base.Tuple;
import cn.quantgroup.cashloanflowboss.core.dictionary.ApplicationStatus;
import cn.quantgroup.cashloanflowboss.spi.user.service.XyqbUserService;
import cn.quantgroup.cashloanflowboss.spi.xyqb.service.XYQBCenterService;
import cn.quantgroup.cashloanflowboss.utils.MD5Tools;
import cn.quantgroup.user.retbean.XUser;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import javax.persistence.criteria.Predicate;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Created by WeiWei on 2019/7/22.
 */
@Slf4j
@Service
public class UserServiceImpl implements UserService{

    @Autowired
    private UserRepository userRepository;
    @Autowired
    private XYQBCenterService xyqbCenterService;
    @Autowired
    private XyqbUserService xyqbUserService;



    /**
     * 创建用户
     *
     * @param username 用户名
     * @param password 登录密码（明文）
     * @return
     */
    @Override
    public boolean createUser(String username, String password) {

        User user = new User();
        user.setUsername(username);
        user.setPassword(MD5Tools.md5(password));
        user.setStatus(UserStatus.ENABLED);

        return Objects.nonNull(this.userRepository.save(user));

    }

    /**
     * 获取用户
     *
     * @param username 用户名
     * @return
     */
    @Override
    public User getUser(String username) {
        return this.userRepository.getUserByUsername(username);
    }

    /**
     * 获取用户列表（分页）
     *
     * @param queryUserListModel
     * @return
     */
    @Override
    public Page<User> getUsers(QueryUserListModel queryUserListModel) {

        Page<User> page = this.userRepository.findAll((root, criteriaQuery, criteriaBuilder) -> {
            List<Predicate> predicates = new ArrayList<>();

            if (StringUtils.isNotEmpty(queryUserListModel.getNickname())) {
                predicates.add(criteriaBuilder.equal(root.get("nickname"), queryUserListModel.getNickname()));
            }
            criteriaQuery.where(criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()])));
            // 指定排序
            criteriaQuery.orderBy(criteriaBuilder.desc(root.get("id")));

            return criteriaQuery.getRestriction();

        }, new PageRequest(queryUserListModel.getPageNumber(), queryUserListModel.getPageSize()));

        return page;
    }

    /**
     * 启用用户
     *
     * @param id 用户ID
     * @return
     */
    @Override
    public boolean enableUser(Long id) {

        User user = this.userRepository.findOne(id);

        // 检查用户是否存在
        Assert.isNull(user, ApplicationStatus.INVALID_USER);

        user.setStatus(UserStatus.ENABLED);

        return Objects.nonNull(this.userRepository.save(user));

    }

    /**
     * 禁用用户
     *
     * @param id 用户ID
     * @return
     */
    @Override
    public boolean disableUser(Long id) {

        User user = this.userRepository.findOne(id);

        // 检查用户是否存在
        Assert.isNull(user, ApplicationStatus.INVALID_USER);

        user.setStatus(UserStatus.DISABLED);

        return Objects.nonNull(this.userRepository.save(user));

    }

    /**
     * 更新用户
     *
     * @param user
     * @return
     */
    @Override
    public User updateUser(User user) {
        return this.userRepository.save(user);
    }

    /**
     * 移除用户
     *
     * @param id 用户ID
     * @return
     */
    @Override
    public Boolean removeUser(Long id) {

        try {
            this.userRepository.delete(id);
            return true;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        return false;

    }

    @Override
    public UserDetailInfo getUserDetailInfo() {
        Principal principal = Application.getPrincipal();
        Assert.isNull(principal, ApplicationStatus.INVALID_USER);

        return UserDetailInfo.valueOf(principal);
    }


    @Override
    public User saveUserInfo(UserInfoModel userInfoModel) {
        User user1 = new User();
        BeanUtils.copyProperties(userInfoModel, user1);
        user1.setPassword(MD5Tools.md5(userInfoModel.getUsername()));
        user1.setRoles(userInfoModel.getRoles());
        return userRepository.save(user1);
    }

    @OperationAnno(channelNo = "#this[0]", opt = OptEnumName.USER_ORDER_CLEAN, succSPEL = "#this.key", optDetailSPEL = "#this.value")
    @Override
    public Tuple<Boolean,String> cleanUserActiveOrder(String mobile) {
        XUser user = xyqbUserService.findUserByPhoneNo(mobile);
        if (user == null) {
            log.info("cleanUserOrder,清除用户活跃订单失败，未找到用户 phoneNo={}", mobile);
            return new Tuple<>(false,"清除用户活跃订单失败，未找到用户");
        }
        return xyqbCenterService.cleanUserActiveOrder(user.getId());
    }
}
