package cn.quantgroup.cashloanflowboss.api.user.service;

import cn.quantgroup.cashloanflowboss.api.login.model.Principal;
import cn.quantgroup.cashloanflowboss.api.user.dictionary.UserStatus;
import cn.quantgroup.cashloanflowboss.api.user.entity.User;
import cn.quantgroup.cashloanflowboss.api.user.model.UserDetailInfo;
import cn.quantgroup.cashloanflowboss.api.user.model.UserInfoModel;
import cn.quantgroup.cashloanflowboss.api.user.repository.UserRepository;
import cn.quantgroup.cashloanflowboss.core.Application;
import cn.quantgroup.cashloanflowboss.core.asserts.Assert;
import cn.quantgroup.cashloanflowboss.core.dictionary.ApplicationStatus;
import cn.quantgroup.cashloanflowboss.utils.MD5Tools;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import java.util.HashSet;
import java.util.Objects;

/**
 * Created by WeiWei on 2019/7/22.
 */
@Slf4j
@Service
public class UserServiceImpl implements UserService{

    @Autowired
    private UserRepository userRepository;

    /**
     * 创建用户
     *
     * @param username 用户名
     * @param password 登录密码（明文）
     * @return
     */
    @Override
    public boolean createUser(String username, String password) {

        User user = new User();
        user.setUsername(username);
        user.setPassword(MD5Tools.md5(password));
        user.setStatus(UserStatus.ENABLED);

        return Objects.nonNull(this.userRepository.save(user));

    }

    /**
     * 获取用户
     *
     * @param username 用户名
     * @return
     */
    @Override
    public User getUser(String username) {
        return this.userRepository.getUserByUsername(username);
    }

    /**
     * 获取用户列表（分页）
     *
     * @param pageNumber
     * @param pageSize
     * @return
     */
    @Override
    public Page<User> getUsers(Integer pageNumber, Integer pageSize) {
        return this.userRepository.findAllBy(new PageRequest(pageNumber, pageSize));
    }

    /**
     * 启用用户
     *
     * @param id 用户ID
     * @return
     */
    @Override
    public boolean enableUser(String id) {

        User user = this.userRepository.findOne(id);

        // 检查用户是否存在
        Assert.isNull(user, ApplicationStatus.INVALID_USER);

        user.setStatus(UserStatus.ENABLED);

        return Objects.nonNull(this.userRepository.save(user));

    }

    /**
     * 禁用用户
     *
     * @param id 用户ID
     * @return
     */
    @Override
    public boolean disableUser(String id) {

        User user = this.userRepository.findOne(id);

        // 检查用户是否存在
        Assert.isNull(user, ApplicationStatus.INVALID_USER);

        user.setStatus(UserStatus.DISABLED);

        return Objects.nonNull(this.userRepository.save(user));

    }

    /**
     * 更新用户
     *
     * @param user
     * @return
     */
    @Override
    public User updateUser(User user) {
        return this.userRepository.save(user);
    }

    /**
     * 移除用户
     *
     * @param id 用户ID
     * @return
     */
    @Override
    public Boolean removeUser(String id) {

        try {
            this.userRepository.delete(id);
            return true;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        return false;

    }

    @Override
    public UserDetailInfo getUserDetailInfo() {
        Principal principal = Application.getPrincipal();
        Assert.isNull(principal, ApplicationStatus.INVALID_USER);

        User user = userRepository.findById(principal.getUserId());

        return UserDetailInfo.valueOf(principal, user);
    }


    @Override
    public User saveUserInfo(UserInfoModel userInfoModel) {
        User user1 = new User();
        BeanUtils.copyProperties(userInfoModel, user1);
        user1.setPassword(MD5Tools.md5(userInfoModel.getUsername()));
        user1.setRoles(userInfoModel.getRoles());
        return userRepository.save(user1);
    }
}
