package cn.quantgroup.cashloanflowboss.core.configuration;

import cn.quantgroup.cashloanflowboss.core.asserts.Assert;
import cn.quantgroup.cashloanflowboss.core.base.Result;
import cn.quantgroup.cashloanflowboss.core.dictionary.ApplicationStatus;
import cn.quantgroup.cashloanflowboss.core.exception.ApplicationException;
import com.fasterxml.jackson.core.io.JsonEOFException;
import com.fasterxml.jackson.databind.JsonMappingException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.annotation.Order;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.stereotype.Component;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;

/**
 * 应用系统异常配置
 * <p>
 * Created by WeiWei on 2017/5/2.
 */
@Slf4j
@Order(1)
@Component
@RestControllerAdvice
public class ApplicationExceptionConfiguration {

    /**
     * 封装系统异常
     *
     * @param e
     * @return
     */
    @ExceptionHandler({ApplicationException.class})
    public Result applicationExceptionHandler(ApplicationException e) {
        log.error("程序员开小差了,系统异常信息:{}", e.getMessage());
        return new Result<>(e.getStatus(), null, e.getStatus().getMessage());
    }

    /**
     * 封装数据绑定异常
     *
     * @param e
     * @return
     */
    @ExceptionHandler({MethodArgumentNotValidException.class, BindException.class})
    public Result validExceptionHandler(Throwable e) {

        Result result = new Result(ApplicationStatus.ARGUMENT_VALID_EXCEPTION);

        BindingResult bindingResult = null;

        if (MethodArgumentNotValidException.class.isInstance(e)) {

            MethodArgumentNotValidException validException = (MethodArgumentNotValidException) e;

            bindingResult = validException.getBindingResult();

        } else if (BindException.class.isInstance(e)) {

            BindException bindException = (BindException) e;

            bindingResult = bindException.getBindingResult();

        }

        if (Assert.notEmpty(bindingResult.getAllErrors())) {
            ObjectError error = bindingResult.getAllErrors().get(0);
            result.setMessage(error.getDefaultMessage());
        }

        return result;

    }

    /**
     * 封装JSON格式化异常
     *
     * @param e
     * @return
     */
    @ExceptionHandler({HttpMessageNotReadableException.class, JsonEOFException.class, JsonMappingException.class})
    public Result jsonFormatException(Throwable e) {
        return new Result<>(ApplicationStatus.JSON_DATA_EXCEPTION, null, HttpMessageNotReadableException.class.isInstance(e) ? ((HttpMessageNotReadableException) e).getRootCause().getMessage() : e.getMessage());
    }

    /**
     * 封装未明确的异常
     *
     * @param e
     * @return
     */
    @ExceptionHandler({Throwable.class})
    public Result generalException(Throwable e) {

        Throwable cause = e;

        while ((cause = cause.getCause()) != null) {
            if (ApplicationException.class.isAssignableFrom(cause.getClass())) {
                cause = cause.getCause();
                break;
            }
        }

        String errorMessage = ApplicationException.class.isAssignableFrom(e.getClass()) ? cause.getMessage() : e.getMessage();
        log.error("程序员开小差了,未明确的异常信息:{}", e);

        return new Result<>(ApplicationStatus.INTERNAL_SERVICE_ERROR, ApplicationStatus.INTERNAL_SERVICE_ERROR.getMessage());

    }

}