package cn.quantgroup.xyqb.service.v2;

import cn.quantgroup.sms.MsgParams;
import cn.quantgroup.tech.util.TechEnvironment;
import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.constant.UserConstant;
import cn.quantgroup.xyqb.controller.req.v2.SMSReq;
import cn.quantgroup.xyqb.exception.BizException;
import cn.quantgroup.xyqb.exception.BizExceptionEnum;
import cn.quantgroup.xyqb.model.SMSCodeBean;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.session.XyqbSessionContextHolder;
import cn.quantgroup.xyqb.util.DateUtils;
import cn.quantgroup.xyqb.util.IpUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

@Service
@Slf4j
public class SMSVoiceVerificationCodeStrategy implements VerificationCodeStrategy {
    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> redisTemplate;
    @Autowired
    private ISmsService smsService;

    private static final long EXPIRE_MINUTES = 10;

    /**
     * 手机号语音上限
     */
    private static final Long PHONE_VOICE_MAX_PER_DAY = 5L;

    /**
     * 设备每天上限
     */
    private static final Long DEVICE_MAX_PER_DAY = 20L;

    /**
     * ip上限
     */
    private static final Long IP_MAX_PER_DAY = 5000L;

    @Override
    public Integer getType() {
        return 1;
    }

    @Override
    public SMSCodeBean send(SMSReq smsReq) {
        SMSCodeBean smsCodeBean = new SMSCodeBean();
        String usage = null;
        switch (smsReq.getSceneType()) {
            case 0:
                usage = "4";
                break;
            case 1:
                usage = "6";
                break;
            case 2:
                usage = "5";
                break;
        }

        int codeLength = Constants.SMS_CODE_LEN_4;
        if (smsReq.getCodeLength() != null) {
            codeLength = smsReq.getCodeLength();
        }
        String randomCode = getRandomCode(codeLength);
        if (usage == null) {
            throw new BizException(BizExceptionEnum.ERROR_USAGE);
        }

        if (sendVerificationCode2VoiceNew(smsReq.getPhoneNo(), randomCode, usage)) {
            smsCodeBean.setCode(randomCode);
            return smsCodeBean;
        } else {
            throw new BizException(BizExceptionEnum.ERROR_SEND_SMS);
        }
    }


    /**
     * 新版本语音验证码
     */
    private boolean sendVerificationCode2VoiceNew(String phoneNo, String randomCode, String usage) {
        SessionStruct sessionStruct = XyqbSessionContextHolder.getXSession();

        String clientIp = sessionStruct.getIp();
        // 手机号计数器
        String verificationCountKey = Constants.REDIS_VOICE_CODE_COUNT + usage;
        Long getPhoneVerificationCount = redisTemplate.opsForHash().increment(verificationCountKey, phoneNo, 1);
        redisTemplate.expire(verificationCountKey, DateUtils.getSeconds(), TimeUnit.SECONDS);
        // 设备号计数器
        Long getDeviceVerificationCount = 0L;
        if (StringUtils.isNotBlank(sessionStruct.getScDeviceId())) {
            String verificationDeviceCountKey = Constants.REDIS_VOICE_DEVICE_COUNT + sessionStruct.getScDeviceId();
            getDeviceVerificationCount = redisTemplate.opsForHash().increment(Constants.REDIS_VOICE_DEVICE_COUNT, verificationDeviceCountKey, 1);
            redisTemplate.expire(Constants.REDIS_VOICE_DEVICE_COUNT, DateUtils.getSeconds(), TimeUnit.SECONDS);
        }
        // IP计数器
        Long getIPVerificationCount = 0L;
        if (StringUtils.isNotBlank(clientIp)) {
            String verificationIPCountKey = Constants.REDIS_VOICE_IP_COUNT + clientIp;
            getIPVerificationCount = redisTemplate.opsForHash().increment(Constants.REDIS_VOICE_IP_COUNT, verificationIPCountKey, 1);
            redisTemplate.expire(Constants.REDIS_VOICE_IP_COUNT, DateUtils.getSeconds(), TimeUnit.SECONDS);
        }
        // 手机号上限检查
        if (getPhoneVerificationCount > PHONE_VOICE_MAX_PER_DAY) {
            throw new BizException(BizExceptionEnum.PHONE_VOICE_MAX_PER_DAY);
        }
        // 设备号上限检查
        if (getDeviceVerificationCount > DEVICE_MAX_PER_DAY) {
            throw new BizException(BizExceptionEnum.DEVICE_MAX_PER_DAY);
        }
        // IP上限检查
        if (!IpUtil.whiteOf(clientIp) && getIPVerificationCount > IP_MAX_PER_DAY) {
            throw new BizException(BizExceptionEnum.IP_MAX_PER_DAY);
        }

        String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;

        if (!UserConstant.defaultTenantId.equals(sessionStruct.getTenantId())) {
            key = Constants.REDIS_PREFIX_VERIFICATION_CODE + sessionStruct.getTenantId() + "_" + phoneNo;
        }

        long expire = redisTemplate.getExpire(key, TimeUnit.MINUTES);
        if (expire >= EXPIRE_MINUTES - 1) {
            throw new BizException(BizExceptionEnum.DUPLICATE_MIN);
        }
        String uniqueId = phoneNo + UUID.randomUUID().toString().replaceAll("-", "");
        MsgParams message = new MsgParams(Collections.singletonList(4), phoneNo, "1", "4", Collections.singletonList(randomCode), uniqueId);
        try {
            smsService.getSmsSender().sendMsg(message);
            redisTemplate.opsForValue().set(key, uniqueId + ":" + randomCode, EXPIRE_MINUTES, TimeUnit.MINUTES);
            //删除用户重置密码，多次错误逻辑
            deleteRetSendCode(phoneNo);
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 删除用户重置密码时短信验证错误
     *
     * @param phoneNo 手机号码
     */
    private void deleteRetSendCode(String phoneNo) {
        String verificationCountKey = Constants.REDIS_VERIFICATION_COUNT + phoneNo;
        redisTemplate.opsForHash().delete(verificationCountKey, Constants.REDIS_VERIFICATION_COUNT);
    }

    private final static String RANDOM_CHARS = "0123456789";

    private String getRandomCode(int count) {
        if (TechEnvironment.isPro()) {
            return RandomStringUtils.random(count, RANDOM_CHARS);
        }
        if (count == 6) {
            return "000000";
        }
        return Constants.SUCCESS_CODE;
    }
}
