package cn.quantgroup.xyqb.service.user.impl;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserDetail;
import cn.quantgroup.xyqb.model.Gender;
import cn.quantgroup.xyqb.model.IdCardInfo;
import cn.quantgroup.xyqb.repository.IUserDetailRepository;
import cn.quantgroup.xyqb.repository.IUserRepository;
import cn.quantgroup.xyqb.service.auth.IIdCardService;
import cn.quantgroup.xyqb.service.user.IUserDetailService;
import cn.quantgroup.xyqb.service.user.vo.UserDetailVO;
import cn.quantgroup.xyqb.util.ValidationUtil;
import com.google.common.collect.Maps;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

/**
 * Created by 11 on 2016/12/29.
 */
@Service
public class UserDetailServiceImpl implements IUserDetailService {
  private static final Logger LOGGER = LoggerFactory.getLogger(UserDetailServiceImpl.class);
  @Autowired
  private IUserDetailRepository userDetailRepository;
  @Autowired
  private IUserRepository userRepository;
  @Autowired
  private IIdCardService idCardService;

  @Override
  public UserDetail findByUserId(Long userId) {
    return userDetailRepository.findByUserId(userId);
  }

  @Override
  public UserDetail saveUserDetail(UserDetail userDetail) throws DataIntegrityViolationException {
    return userDetailRepository.save(userDetail);
  }

  @Override
  public UserDetail findByPhoneNo(String phoneNo) {
    return userDetailRepository.findByPhoneNo(phoneNo);
  }

  @Override
  public void updateUserQQ(Long userId, String qq) {
    userDetailRepository.updateUserQQ(qq, userId);
  }

  @Override
  public void updateUserEmail(Long userId, String email) {
    userDetailRepository.updateUserEmail(email, userId);
  }

  @Override
  public List<UserDetailVO> searchUserDetailList(String name, String phoneNo, String idNo) {
    List<UserDetail> details = userDetailRepository.findAll(getSpecification(name, phoneNo, idNo));
    Map<Long, User> userMap = Maps.newHashMap();
    if (!CollectionUtils.isEmpty(details)) {
      List<Long> userIds = details.stream().map(d -> d.getUserId()).collect(Collectors.toList());
      List<User> users = userRepository.findAll((root, query, cb) -> {
        query.where(root.get("id").in(userIds));
        return query.getRestriction();
      });
      userMap = users.stream().collect(Collectors.toMap(User::getId, o -> o));
    }
    Map<Long, User> finalUserMap = userMap;
    List<UserDetailVO> userDetailVOS = details.stream().map(o -> fromUserDetailAndUserMap(o, finalUserMap)).collect(Collectors.toList());
    return userDetailVOS;
  }

  @Override
  public List<UserDetail> findByUserIdIn(List<Long> userIds) {

    return userDetailRepository.findAll((root, query, cb) -> {
      query.where(root.get("userId").in(userIds));
      return query.getRestriction();
    });

  }

  @Override
  public List<UserDetail> findByPhoneNos(List<String> phoneNos) {
    return userDetailRepository.findAll((root, query, cb) -> {
      query.where(root.get(Constants.PHONE_NO).in(phoneNos));
      return query.getRestriction();
    });
  }

  @Override
  public List<UserDetail> findByIdnos(List<String> idnos) {
    return userDetailRepository.findAll((root, query, cb) -> {
      query.where(root.get("idNo").in(idnos));
      return query.getRestriction();
    });
  }

  private Specification<UserDetail> getSpecification(String name, String phoneNo, String idNo) {
    List<Predicate> list = new ArrayList<>();
    Specification<UserDetail> specification = new Specification<UserDetail>() {
      @Override
      public Predicate toPredicate(Root<UserDetail> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder criteriaBuilder) {
        if (!StringUtils.isEmpty(name)) {
          list.add(criteriaBuilder.equal(root.get("name").as(String.class), name));
        }
        if (!StringUtils.isEmpty(phoneNo)) {
          list.add(criteriaBuilder.equal(root.get(Constants.PHONE_NO).as(String.class), phoneNo));
        }
        if (!StringUtils.isEmpty(idNo)) {
          list.add(criteriaBuilder.equal(root.get("idNo").as(String.class), idNo));
        }
        Predicate[] p = new Predicate[list.size()];
        return criteriaBuilder.and(list.toArray(p));
      }
    };
    return specification;
  }

  private UserDetailVO fromUserDetailAndUserMap(UserDetail userDetail, Map<Long, User> userMap) {
    UserDetailVO userDetailVO = UserDetailVO.fromUserDetail(userDetail);
    User user = userMap.get(userDetail.getUserId());
    if (user != null) {
      userDetailVO.setEnable(user.getEnable());
    }
    return userDetailVO;
  }

  private Specification<UserDetail> valueInSpecification(List<Long> userId, List<String> phoneNo, List<String> idNo) {
    List<Predicate> list = new ArrayList<>();
    Specification<UserDetail> specification = new Specification<UserDetail>() {
      @Override
      public Predicate toPredicate(Root<UserDetail> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder criteriaBuilder) {
        if (userId!=null&&userId.size()>0) {
          criteriaQuery.where(root.get("userId").in(userId));
          list.add(criteriaQuery.getRestriction());
        }
        if (phoneNo!=null&&phoneNo.size()>0) {
          criteriaQuery.where(root.get(Constants.PHONE_NO).in(phoneNo));
          list.add(criteriaQuery.getRestriction());
        }
        if (idNo!=null&&idNo.size()>0) {
          criteriaQuery.where(root.get("idNo").in(idNo));
          list.add(criteriaQuery.getRestriction());
        }
        Predicate[] p = new Predicate[list.size()];
        return criteriaBuilder.and(list.toArray(p));
      }
    };
    return specification;
  }

  private PageRequest buildPageRequest(int pageNumber, int pagzSize, String sortType) {
    Sort sort = null;
    if ("auto".equals(sortType)) {
      sort = new Sort(Sort.Direction.ASC, "userId");
    }

    return new PageRequest(pageNumber - 1, pagzSize, sort);
  }

  @Override
  public Page<UserDetail> getUserDetailsPage(List<Long> userId, List<String> phoneNos, List<String> idNos, int pageNumber, int pageSize,
                                             String sortType) {
    PageRequest pageRequest = buildPageRequest(pageNumber, pageSize, sortType);
    Specification<UserDetail> spec = valueInSpecification(userId,phoneNos,idNos);
    return userDetailRepository.findAll(spec, pageRequest);
  }

  @Override
  public void fixedIdNoAndGender() {
    List<UserDetail> details=userDetailRepository.selectUserDetailsBy();
    if(!CollectionUtils.isEmpty(details)){
      for (UserDetail detail:details){
        try {
          if(!idCardService.isIdCardValid(detail.getIdNo())){
            LOGGER.error("修复用户老数据身份证号错误",detail.getUserId());
            continue;
          }
          Gender gender=idCardService.getIdCardInfo(detail.getIdNo()).getGender();
          if(null!=gender){
            userDetailRepository.updateGender(gender.ordinal(),detail.getUserId());
          }

        } catch (ParseException e) {
          LOGGER.error("修复用户老数据身份证号错误",detail.getUserId());
         continue;
        }
      }
      LOGGER.info("修复用户老数据身份证号完成");
    }
  }

  @Override
  public int updateIdCard(String name, String idNo, String phoneNo) {
    UserDetail userDetail = userDetailRepository.findByPhoneNo(phoneNo);
    if (null != userDetail) {
      if (!StringUtils.isBlank(idNo)) {
        try {
          IdCardInfo idCardInfo = idCardService.getIdCardInfo(idNo);
          return userDetailRepository.updateIdNoByPhoneNo(idCardInfo.getIdNo(), Optional.ofNullable(idCardInfo.getGender()).orElse(Gender.UNKNOWN).ordinal(), phoneNo);
        } catch (ParseException e) {
          LOGGER.error("用户的身份证错误，phoneNo:{},e:{}", phoneNo, e);
          return 0;
        }
      }
      if (!StringUtils.isBlank(name) && ValidationUtil.validateChinese(name)) {
        return userDetailRepository.updateNameByPhoneNo(name, phoneNo);
      }
    }
    return 0;
  }
}
