package cn.quantgroup.xyqb.service.register.handler.impl;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.exception.PushUserToLkbException;
import cn.quantgroup.xyqb.model.UserRegisterParam;
import cn.quantgroup.xyqb.service.register.handler.AbstractUserRegisterHandler;
import cn.quantgroup.xyqb.service.user.ILkbUserService;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.util.JsonUtil;
import cn.quantgroup.xyqb.util.PasswordUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import java.sql.Timestamp;
import java.util.Objects;
import java.util.UUID;

/**
 * Created by liqing on 2017/12/4 0004.
 * 注册 - 基本信息
 * 失败影响注册流程
 */
@Component("baseUserRegisterHandler")
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
@Slf4j
public class BaseUserRegisterHandler extends AbstractUserRegisterHandler {

    @Autowired
    private ILkbUserService lkbUserService;
    @Autowired
    private IUserService userService;

    @Override
    public User handleRegister(UserRegisterParam userRegisterParam) {
        String uuid = UUID.randomUUID().toString();
        // 保存user信息
        User user = saveUser(uuid, userRegisterParam);
        // user保存成功后同步用户信息到Lkb
        pushUserToLkb(uuid, userRegisterParam);
        userRegisterParam.setUser(user);
        if (successor != null) {
            return successor.handleRegister(userRegisterParam);
        }
        return userRegisterParam.getUser();
    }

    /**
     * 保存用户user信息
     *
     * @param uuid
     * @param userRegisterParam
     * @return
     */
    private User saveUser(String uuid, UserRegisterParam userRegisterParam) {
        User user = new User();
        user.setUuid(uuid);
        user.setEnable(true);
        user.setPhoneNo(userRegisterParam.getPhoneNo());
        Long registerFrom = userRegisterParam.getRegisterFrom();
        if (Objects.isNull(registerFrom)) {
            registerFrom = 1L;
        }
        Long channelId = userRegisterParam.getChannelId();
        if (Objects.equals(channelId, 222L)) {
            user.setRegisteredFrom(channelId);
        } else {
            user.setRegisteredFrom(registerFrom);
        }
        String password = StringUtils.defaultString(userRegisterParam.getPassword(), "");
        // 如果需要生成随机密码
        if (userRegisterParam.isGenerateRandomPwd()) {
            password = PasswordUtil.generateRandomPwd(Constants.RANDOM_PWD_LEN);
        }
        user.setPassword(PasswordUtil.MD5(password.toLowerCase() + Constants.PASSWORD_SALT));
        Timestamp currentTime = new Timestamp(System.currentTimeMillis());
        user.setUpdatedAt(currentTime);
        user.setCreatedAt(currentTime);
        userService.saveUser(user);
        return user;
    }

    /**
     * 同步用户信息到Lkb，失败影响注册流程
     *
     * @param uuid
     * @param userRegisterParam
     */
    private void pushUserToLkb(String uuid, UserRegisterParam userRegisterParam) {
        log.info("[pushUserToLkb]同步用户信息到Lkb，phoneNo : {}", userRegisterParam.getPhoneNo());
        boolean pushResult = lkbUserService.pushUser(uuid, userRegisterParam.getPhoneNo(),
                userRegisterParam.getName(), userRegisterParam.getIdNo());
        if (!pushResult) {
            log.error("[userRegisterHandler][baseUserRegisterHandler]同步用户至Lkb出错,userRegisterParam:{}", JsonUtil.toJson(userRegisterParam));
            throw new PushUserToLkbException("同步用户至Lkb出错");
        }
    }

}
