package cn.quantgroup.xyqb.service.auth.impl;

import cn.quantgroup.xyqb.entity.UserAuthorized;
import cn.quantgroup.xyqb.model.AuthPattern;
import cn.quantgroup.xyqb.model.UserAuthorizedParam;
import cn.quantgroup.xyqb.repository.IUserAuthorizedRepository;
import cn.quantgroup.xyqb.repository.IUserRepository;
import cn.quantgroup.xyqb.service.auth.IIdCardService;
import cn.quantgroup.xyqb.service.auth.IUserAuthorizedService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.sql.Timestamp;
import java.text.ParseException;
import java.util.Objects;

/**
 * @author xufei on 2018/1/5.
 */
@Slf4j
@Service
public class UserAuthorizedServiceImpl implements IUserAuthorizedService {
    @Resource
    private IIdCardService iIdCardService;

    @Resource
    private IUserAuthorizedRepository userAuthorizedRepository;

    @Resource
    private IUserRepository userRepository;

//    @Cacheable(value = "user_authorized_idno_cache", key = "#idNo", unless = "#result == false", cacheManager = "cacheManager")
    @Override
    public Boolean hasUserAuthorized(String idNo) {
        try {
            if (StringUtils.isBlank(idNo) || !iIdCardService.isIdCardValid(idNo)) {
                return Boolean.FALSE;
            }
        } catch (ParseException e) {
            log.error("[hasUserAuthorized]参数异常e:{}", e);
            return Boolean.FALSE;
        }

        UserAuthorized userAuthorized = userAuthorizedRepository.findByIdNo(idNo);
        log.info("[hasUserAuthorized]查询实名账户userAuthorized:{}", userAuthorized);
        return null != userAuthorized;
    }

    @Override
    public Boolean checkUserAuthorizedParam(UserAuthorizedParam userAuthorizedParam) throws Exception {
        if (null == userAuthorizedParam) {
            return Boolean.TRUE;
        }
        String userUuid = userAuthorizedParam.getUserUuid();
        if (StringUtils.isBlank(userUuid)) {
            return Boolean.TRUE;
        }
        String name = userAuthorizedParam.getName();
        String authPattern = userAuthorizedParam.getAuthPattern();
        String idNo = userAuthorizedParam.getIdNo();
        return (StringUtils.isBlank(name) || StringUtils.isBlank(authPattern) || !iIdCardService.isIdCardValid(idNo));
    }

    @Override
    public UserAuthorized createUserAuthorized(UserAuthorizedParam userAuthorizedParam) {
        // 数据检查,以及检查userUuid是否存在
        if (Objects.isNull(userAuthorizedParam) || this.hasUserAuthorized(userAuthorizedParam.getIdNo())
                || Objects.isNull(userRepository.findByUuid(userAuthorizedParam.getUserUuid()))) {
            return null;
        }
        AuthPattern authPatternEnum = AuthPattern.valueOf(userAuthorizedParam.getAuthPattern());

        UserAuthorized userAuthorized = new UserAuthorized();
        userAuthorized.setAuthPattern(authPatternEnum);
        userAuthorized.setAvailable(Boolean.TRUE);
        userAuthorized.setName(userAuthorizedParam.getName());
        userAuthorized.setIdNo(userAuthorizedParam.getIdNo());
        userAuthorized.setUserUuid(userAuthorizedParam.getUserUuid());

        try {
            userAuthorized = userAuthorizedRepository.save(userAuthorized);
            log.info("[createUserAuthorized]创建实名账户成功,userAuthorized:{}", userAuthorized);
            return userAuthorized;
        } catch (Exception e) {
            log.error("[createUserAuthorized]创建实名账户异常e:{}", e);
            return null;
        }
    }

//    @Cacheable(value = "user_authorized_uuid_2_id_cache", key = "#userUuid", unless = "#result == null", cacheManager = "cacheManager")
    @Override
    public String getUserAuthorizedId(String userUuid) {
        if (StringUtils.isBlank(userUuid)) {
            return null;
        }

        UserAuthorized userAuthorized = userAuthorizedRepository.findByUserUuid(userUuid);
        log.info("[getUserAuthorizedId]获取实名账户的id,userUuid:{},userAuthorized:{}", userUuid, userAuthorized);
        if (null == userAuthorized) {
            return null;
        }
        return userAuthorized.getId();
    }
}
