package cn.quantgroup.xyqb.controller.external.motan;

import cn.quantgroup.motan.bean.*;
import cn.quantgroup.motan.retbean.*;
import cn.quantgroup.motan.service.UserMotanService;
import cn.quantgroup.motan.vo.UserSysResult;
import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.entity.*;
import cn.quantgroup.xyqb.entity.enumerate.*;
import cn.quantgroup.xyqb.model.IdCardInfo;
import cn.quantgroup.xyqb.model.IdType;
import cn.quantgroup.xyqb.model.LoginProperties;
import cn.quantgroup.xyqb.model.UserRet;
import cn.quantgroup.xyqb.model.session.LoginInfo;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.repository.IUserRepository;
import cn.quantgroup.xyqb.service.auth.IIdCardService;
import cn.quantgroup.xyqb.service.merchant.IMerchantService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.service.user.*;
import cn.quantgroup.xyqb.service.wechat.IWechatService;
import cn.quantgroup.xyqb.util.PasswordUtil;
import cn.quantgroup.xyqb.util.ValidationUtil;
import com.alibaba.fastjson.JSON;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.weibo.api.motan.config.springsupport.annotation.MotanService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.sql.Timestamp;
import java.text.ParseException;
import java.util.List;
import java.util.Random;
import java.util.stream.Collectors;

/**
 * Created by 11 on 2017/2/27.
 */
@Slf4j
@MotanService(export = "userMotan:8002", registry = "registryConfig")
public class MotanUserServiceImpl implements UserMotanService {
  private static final ObjectMapper MAPPER = new ObjectMapper();
  private final static Random random = new Random();

  static {
    MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
  }

  @Autowired
  private IIdCardService idCardService;
  @Autowired
  private IUserService userService;
  @Autowired
  private IUserDetailService userDetailService;
  @Autowired
  private IUserRepository userRepository;
  @Autowired
  private ISmsService smsService;
  @Autowired
  private ILkbUserService lkbUserService;
  @Autowired
  private IMerchantService merchantService;
  @Autowired
  private ISessionService sessionService;
  @Autowired
  private IContactService contactService;
  @Autowired
  private IAddressService addressService;
  @Autowired
  private IUserExtInfoService userExtInfoService;
  @Autowired
  private IWechatService wechatService;

  @Override
  public UserSysResult<XUser> findUserByPhoneNo(String phoneNo) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return returnValue(null, "手机号格式不正确");
    }
    User user = userService.findByPhoneInDb(phoneNo);
    return returnValue(fromUser(user), "");
  }

  @Override
  public UserSysResult<XUser> findUserByUuid(String uuid) {
    if (StringUtils.isBlank(uuid)) {
      return returnValue(null, "参数不能为空");
    }
    User user = userService.findByUuidInDb(uuid);
    return returnValue(fromUser(user), "");
  }

  @Override
  public UserSysResult<XUser> saveUser(UserBean userBean) {
    if (null == userBean || !ValidationUtil.validatePhoneNo(userBean.getPhoneNo())) {
      return returnValue(null, userBean == null ? "要保存的用户对象不能为null" : "手机号不正确");
    }
    Timestamp now = new Timestamp(System.currentTimeMillis());
    User user = userService.findByPhoneWithCache(userBean.getPhoneNo());
    if (user == null) {
      user = new User();
      user.setCreatedAt(now);
      user.setPhoneNo(userBean.getPhoneNo());
      user.setEnable(true);
      String password = userBean.getPassword();
      if (StringUtils.isBlank(password)) {
        password = String.valueOf(random.nextInt(899999) + 100000);
      }
      user.setPassword(PasswordUtil.MD5(password + Constants.PASSWORD_SALT));
    } else {
      if (StringUtils.isNotBlank(user.getPassword())) {
        user.setPassword(PasswordUtil.MD5(userBean.getPassword() + Constants.PASSWORD_SALT));
      }
    }
    user.setUpdatedAt(now);
    user.setUuid(userBean.getUuid());
    user.setRegisteredFrom(userBean.getRegisteredFrom());
    User saveUser = userService.saveUser(user);
    return returnValue(fromUser(saveUser), "");
  }

  @Override
  public UserSysResult<XUserDetail> findUserDetailByUserId(Long userId) {
    if (userId == null || userId.longValue() < 1) {
      return returnValue(null, "参数必须是正整数");
    }
    UserDetail userDetail = userDetailService.findByUserId(userId);
    XUserDetail xUserDetail = fromUserDetail(userDetail);
    return returnValue(xUserDetail, "");
  }

  private XUserDetail fromUserDetail(UserDetail userDetail) {
    if (userDetail == null) {
      return null;
    }
    XUserDetail xUserDetail = new XUserDetail();
    xUserDetail.setId(userDetail.getId());
    xUserDetail.setUserId(userDetail.getUserId());
    xUserDetail.setPhoneNo(userDetail.getPhoneNo());
    xUserDetail.setName(userDetail.getName());
    xUserDetail.setIdNo(userDetail.getIdNo());
    xUserDetail.setIdType(cn.quantgroup.motan.enums.IdType.valueOf(userDetail.getIdType().getName()));
    xUserDetail.setGender(cn.quantgroup.motan.enums.Gender.valueOf(userDetail.getGender().getName()));
    xUserDetail.setEmail(userDetail.getEmail());
    xUserDetail.setQq(userDetail.getQq());
    xUserDetail.setId(userDetail.getId());
    xUserDetail.setCreatedAt(userDetail.getCreatedAt());
    xUserDetail.setUpdatedAt(userDetail.getUpdatedAt());
    xUserDetail.setIsAuthenticated(userDetail.getIsAuthenticated());
    return xUserDetail;
  }

  private XUser fromUser(User user) {
    if (user == null) {
      return null;
    }
    XUser xUser = new XUser();
    xUser.setId(user.getId());
    xUser.setPassword(user.getPassword());
    xUser.setRegisteredFrom(user.getRegisteredFrom());
    xUser.setUuid(user.getUuid());
    xUser.setEnable(user.getEnable());
    xUser.setPhoneNo(user.getPhoneNo());
    xUser.setUpdatedAt(user.getUpdatedAt());
    xUser.setCreatedAt(user.getCreatedAt());
    return xUser;
  }

  @Override
  public UserSysResult<XUser> findUserByUserId(Long id) {
    if (id == null || id.longValue() < 1) {
      return returnValue(null, "参数必须是正整数");
    }
    User user = userService.findById(id);
    return returnValue(fromUser(user), "");
  }


  @Override
  public UserSysResult<XUserDetail> findUserDetailByPhone(String phoneNo) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return returnValue(null, "手机号格式不正确");
    }
    UserDetail userDetail = userDetailService.findByPhoneNo(phoneNo);
    return returnValue(fromUserDetail(userDetail), "");
  }

  @Override
  public UserSysResult<XUserDetail> updateUserDetail(UserDetailUpdateBean userDetailUpdateBean) {
    if (null == userDetailUpdateBean) {
      return returnValue(null, "要保存的用户详情对象不能为null");
    }
    UserDetail userDetail = new UserDetail();
    if (null != userDetailUpdateBean.getId() && userDetailUpdateBean.getId() > 0L) {
      userDetail.setId(userDetailUpdateBean.getId());
    }
    userDetail.setUserId(userDetailUpdateBean.getUserId());
    userDetail.setName(userDetailUpdateBean.getName());
    userDetail.setPhoneNo(userDetailUpdateBean.getPhoneNo());
    userDetail.setIdNo(userDetailUpdateBean.getIdNo());
    Timestamp time = new Timestamp(System.currentTimeMillis());
    userDetail.setCreatedAt(time);
    userDetail.setUpdatedAt(time);
    userDetail.setIdType(IdType.ID_CARD);
    try {
      userDetail.setGender(idCardService.getIdCardInfo(userDetailUpdateBean.getIdNo()).getGender());
    } catch (ParseException e) {
      log.error("根据身份证获取性别出错,userDetailUpdateBean:{}", JSON.toJSONString(userDetailUpdateBean), e);
      return returnValue(null, "根据身份证获取性别出错.");
    }
    userDetail.setEmail(userDetailUpdateBean.getEmail());
    userDetail = userDetailService.saveUserDetail(userDetail);
    if (userDetail != null) {
      XUserDetail xUserDetail = convertObject(JSON.toJSONString(userDetail), new TypeReference<XUserDetail>() {
      });
      return returnValue(xUserDetail, "");
    }
    return returnValue(null, "保存失败");
  }

  @Override
  public UserSysResult<XUserDetail> saveUserDetail(UserDetailSaveBean userDetailSaveBean) {
    return null;
  }

  @Override
  public UserSysResult<XLoginInfo> appLoginAndFetchLoginInfo(AppLoginParam appLoginParam, String requestIp) {
    if (!ValidationUtil.validatePhoneNo(appLoginParam.getPhoneNo())) {
      return returnValue(null, "账户或者密码错误");
    }

    log.info("第三方用户登录, loginFrom:{}, requestIp:{}", appLoginParam.getRegisterFrom(), requestIp);
    User user = userService.findByPhoneInDb(appLoginParam.getPhoneNo());
    if (user == null) {
      user = register(appLoginParam.getRegisterFrom(), appLoginParam.getPhoneNo(), appLoginParam.getIdNo(), appLoginParam.getName());
    }
    if (user == null) {
      return returnValue(null, "账户或者密码错误");
    }
    Merchant merchant = merchantService.findMerchantByName(appLoginParam.getKey());
    if (merchant == null) {
      return returnValue(null, "无效的商户");
    }
    LoginProperties loginProperties = new LoginProperties();
    loginProperties.setChannelId(appLoginParam.getChannelId());
    loginProperties.setMerchantName(merchant.getName());
    loginProperties.setCreatedFrom(appLoginParam.getRegisterFrom());
    loginProperties.setAppChannel(appLoginParam.getAppChannel());
    SessionStruct sessionStruct = sessionService.createSessionAndPersist(user, loginProperties);

    LoginInfo loginInfo = new LoginInfo();
    loginInfo.setUser(UserRet.getUserRet(user));
    loginInfo.setToken(sessionStruct.getSid());
    LoginInfo.LoginContext context = new LoginInfo.LoginContext();
    context.setChannelId(sessionStruct.getValues().getLoginProperties().getChannelId());
    context.setCreatedFrom(sessionStruct.getValues().getLoginProperties().getCreatedFrom());
    context.setAppChannel(appLoginParam.getAppChannel());
    loginInfo.setLoginContext(context);

    log.info("第三方用户获取信息登录成功, loginFrom:{}, phoneNo:{},appChannel:{}", appLoginParam.getRegisterFrom(), appLoginParam.getPhoneNo(), appLoginParam.getAppChannel());
    XLoginInfo xLoginInfo = convertObject(JSON.toJSONString(loginInfo), new TypeReference<XLoginInfo>() {
    });
    return returnValue(xLoginInfo, "");
  }

  private <T> T convertObject(String json, TypeReference typeReference) {
    try {
      if (StringUtils.isEmpty(json)) {
        return null;
      }
      return MAPPER.readValue(json, typeReference);
    } catch (Exception ex) {
      log.error("用户信息转换出错,json:{}", json, ex);
      return null;
    }

  }

  @Override
  public void updateUserQQ(String qq, Long userId) {
    if (StringUtils.isEmpty(qq) || userId == null || userId == 0L) {
      log.error("QQ或用户id非法,不允许修改:qq{}, userid:{}", qq, userId);
      return;
    }
    userDetailService.updateUserQQ(userId, qq);
  }

  @Override
  public UserSysResult<List<XContact>> findContactsByUserId(Long userId) {
    List<Contact> contacts = contactService.findByUserId(userId);
    List<XContact> xContacts = convertObject(JSON.toJSONString(contacts), new TypeReference<List<XContact>>() {
    });
    return returnValue(xContacts, "");
  }

  @Override
  public UserSysResult<List<XContact>> saveContacts(Long userId, List<XContact> contacts) {
    if (null == userId || userId == 0L) {
      return returnValue(null, "保存联系人列表失败,用户id为空");
    }
    if (CollectionUtils.isEmpty(contacts)) {
      return returnValue(null, "保存联系人列表失败,列表为空");
    }
    //做一次类型转换,
    String jsonContacts = JSON.toJSONString(contacts);
    List<Contact> contactList = null;
    try {
      contactList = MAPPER.readValue(jsonContacts, new TypeReference<List<Contact>>() {
      });
    } catch (Exception e) {
      log.error("联系人列表转换错误", e);
      return returnValue(null, "联系人转换错误");
    }
    Timestamp now = new Timestamp(System.currentTimeMillis());
    for (Contact c : contactList) {
      c.setId(null);
      c.setUserId(userId);
      c.setRelation(c.getRelation() == null ? Relation.OTHER : c.getRelation());
      c.setCreatedAt(now);
      c.setUpdateAt(now);
    }
    List<Contact> result = contactService.save(contactList);
    List<XContact> xResult = convertObject(JSON.toJSONString(result), new TypeReference<List<XContact>>() {
    });
    return returnValue(xResult, "");
  }

  @Override
  public UserSysResult<XAddress> findAddressByUserId(Long userId) {
    UserSysResult<XAddress> userSysResult = returnValue(null, null);
    Address address = addressService.findByUserId(userId);
    if (null == address) {
      return userSysResult;
    }
    XAddress xAddress = convertObject(JSON.toJSONString(address), new TypeReference<XAddress>() {
    });
    return returnValue(xAddress, "");
  }

  @Override
  public UserSysResult<XAddress> saveAddress(Long userId, Long provinceCode, String province, Long cityCode, String city,
                                             Long districtCode, String district, String address) {
    UserSysResult<XAddress> userSysResult = returnValue(null, null);
    if (null == userId || null == provinceCode || null == cityCode) {
      return userSysResult;
    }
    Address addressObj = addressService.findByUserId(userId);
    if (addressObj == null) {
      addressObj = new Address();
      Timestamp now = new Timestamp(System.currentTimeMillis());
      addressObj.setCreatedAt(now);
      addressObj.setUpdateAt(now);
    }
    addressObj.setUserId(userId);
    addressObj.setProvinceCode(provinceCode);
    addressObj.setCityCode(cityCode);
    addressObj.setCity(city);
    addressObj.setDistrictCode(districtCode);
    addressObj.setDistrict(district);
    addressObj.setAddress(address);
    addressObj.setProvince(province);
    addressObj = addressService.save(addressObj);
    XAddress xAddress = convertObject(JSON.toJSONString(addressObj), new TypeReference<XAddress>() {
    });
    return returnValue(xAddress, "");
  }

  @Override
  public UserSysResult<XUserExtInfo> updateUserExtInfo(
      Long userId, cn.quantgroup.motan.enums.IncomeEnum incomeEnum, cn.quantgroup.motan.enums.IncomeRangeEnum incomeRangeEnum,
      cn.quantgroup.motan.enums.OccupationEnum occupationEnum, cn.quantgroup.motan.enums.EducationEnum educationEnum, Boolean hasCar,
      Boolean hasSocialSecurity, Boolean hasHouse, Boolean hasCreditCard, cn.quantgroup.motan.enums.MaritalStatus maritalStatus) {
    UserSysResult<XUserExtInfo> userSysResult = returnValue(null, null);
    if (null == userId || 0L == userId) {
      return returnValue(null, "用户id为空");
    }
    UserExtInfo info = userExtInfoService.findByUserId(userId);
    if (info == null) {
      Timestamp now = new Timestamp(System.currentTimeMillis());
      info = new UserExtInfo();
      info.setUserId(userId);
      info.setCreatedAt(now);
      info.setUpdateAt(now);
    }
    if (incomeEnum != null) {
      info.setIncomeEnum(IncomeEnum.valueOf(incomeEnum.name()));
    }
    if (incomeRangeEnum != null) {
      info.setIncomeRangeEnum(IncomeRangeEnum.valueOf(incomeRangeEnum.name()));
    }
    if (occupationEnum != null) {
      info.setOccupationEnum(OccupationEnum.valueOf(occupationEnum.name()));
    }
    if (educationEnum != null) {
      info.setEducationEnum(EducationEnum.valueOf(educationEnum.name()));
    }
    if (hasCar != null) {
      info.setHasCar(hasCar);
    }
    if (hasCreditCard != null) {
      info.setHasCreditCard(hasCreditCard);
    }
    if (hasHouse != null) {
      info.setHasHouse(hasHouse);
    }
    if (maritalStatus != null) {
      info.setMarryStatus(MaritalStatus.valueOf(maritalStatus.name()));
    }
    info = userExtInfoService.save(info);
    XUserExtInfo userExtInfo = convertObject(JSON.toJSONString(info), new TypeReference<XUserExtInfo>() {
    });
    return returnValue(userExtInfo, "");
  }

  @Override
  public UserSysResult<String> queryOpenIdByXyqbUserId(Long userId) {
    if (userId == null || userId < 1) {
      return returnValue(null, "用户id不能为空");
    }
    WechatUserInfo wechatUserInfo = wechatService.queryOpenIdByUserId(userId);
    if (wechatUserInfo == null) {
      return returnValue(null, "wechat信息为空");
    }
    return returnValue(wechatUserInfo.getOpenId(), "success");
  }

  @Override
  public UserSysResult<Boolean> disableUser(Long userId) {
    return null;
  }

  @Override
  public UserSysResult<String> queryOpenIdByPhoneNo(String phoneNo) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return returnValue(null, "手机号格式不正确");
    }
    WechatUserInfo wechatUserInfo = wechatService.findWechatUserInfoByPhoneNo(phoneNo);
    String openId = null == wechatUserInfo ? null : wechatUserInfo.getOpenId();
    return returnValue(openId, null);
  }

  @Override
  public UserSysResult<UserRegisterTrace> findRegisterTraceByPhoneNo(String phoneNo) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return returnValue(null, "手机号格式不正确");
    }
    return null;
  }

  @Override
  public UserSysResult<XUserSpouse> findUserSpouseByUserId(Long userId) {
    return null;
  }

  @Override
  public UserSysResult<XUserExtInfo> findUserExtInfoByUserId(Long userId) {
    return null;
  }

  @Override
  public UserSysResult<List<XUserDetail>> queryUserDetailBySpecification(String name, String phoneNo, String idNo) {
    List<UserDetail> userDetails = userDetailService.searchUserDetailList(name, phoneNo, idNo);
    if (CollectionUtils.isEmpty(userDetails)) {
      return null;
    }
    List<XUserDetail> xUserDetails = userDetails.stream().map(user -> fromUserDetail(user)).collect(Collectors.toList());
    return returnValue(xUserDetails, null);
  }

  @Override
  public UserSysResult<String> loginWithUuid(String uuid) {
    return null;
  }

  @Override
  public UserSysResult<Boolean> userImportCheck(String phoneNo, String registerFrom) {
    return null;
  }

  @Override
  public UserSysResult<XUserSpouse> saveUserSpouse(XUserSpouse userSpouse) {
    return null;
  }

  @Override
  public UserSysResult<AppLoginResultBean> appLogin(AppLoginParam param) {
    return null;
  }

  /**
   * 封装返回结果.
   *
   * @param t
   * @param message
   * @param <T>
   * @return
   */
  private <T> UserSysResult<T> returnValue(T t, String message) {
    UserSysResult<T> userSysResult = new UserSysResult();
    userSysResult.setMsg(message);
    /**
     * 无返回结果时,返回的code和BusinessCode均是0001
     */
    if (null == t) {
      userSysResult.setBusinessCode("0001");
      userSysResult.setCode("0001");
    } else {
      userSysResult.setBusinessCode("0000");
      userSysResult.setCode("0000");
    }
    userSysResult.setData(t);
    return userSysResult;
  }

  /**
   * 注册新用户
   *
   * @return
   */
  private User register(Long registerFrom, String phoneNo, String idNo, String name) {

    String randomCode = String.valueOf(random.nextInt(899999) + 100000);

    String uuid = lkbUserService.registerApp(phoneNo, randomCode);
    Timestamp currentTime = new Timestamp(System.currentTimeMillis());

    log.info("第三方登录用户，保存 User");

    User user = new User();
    user.setRegisteredFrom(registerFrom);
    user.setUuid(uuid);
    user.setEnable(true);
    user.setPhoneNo(phoneNo);
    user.setPassword(PasswordUtil.MD5(randomCode + Constants.PASSWORD_SALT));
    user.setUpdatedAt(currentTime);
    user.setCreatedAt(currentTime);
    userRepository.saveAndFlush(user);

    if (registerFrom != 645L) {
      smsService.sendAfterRegister(phoneNo);
      log.info("第三方登录用户注册成功, registerFrom:{}, phoneNo:{},idNo:{},name:{} 并且已发送短信通知", registerFrom, phoneNo, idNo, name);
    }

    if (StringUtils.isNotEmpty(idNo) && StringUtils.isNotEmpty(name)) {
      IdCardInfo cardInfo;
      try {
        cardInfo = idCardService.getIdCardInfoWithExceptions(idNo);

        UserDetail userDetail = new UserDetail();
        userDetail.setIdNo(cardInfo.getIdNo());
        userDetail.setPhoneNo(phoneNo);
        userDetail.setName(name);
        userDetail.setIdType(IdType.ID_CARD);
        userDetail.setCreatedAt(currentTime);
        userDetail.setUpdatedAt(currentTime);
        userDetail.setUserId(user.getId());
        userDetail.setGender(cardInfo.getGender());
        userDetailService.saveUserDetail(userDetail);

        lkbUserService.userUpdate(user.getUuid(), name, idNo);
      } catch (Exception e) {
        log.error("保存 UserDetail 出现异常", e);
      }
    }

    return user;
  }
}
