package cn.quantgroup.xyqb.controller.external.user;

import cn.quantgroup.user.enums.Relation;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.logcaller.LogHttpCaller;
import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.entity.*;
import cn.quantgroup.xyqb.entity.enumerate.*;
import cn.quantgroup.xyqb.model.*;
import cn.quantgroup.xyqb.service.api.IUserApiService;
import cn.quantgroup.xyqb.service.auth.IIdCardService;
import cn.quantgroup.xyqb.service.http.IHttpService;
import cn.quantgroup.xyqb.service.register.IUserRegisterService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.user.*;
import cn.quantgroup.xyqb.service.wechat.IWechatService;
import cn.quantgroup.xyqb.util.*;
import cn.quantgroup.xyqb.util.encrypt.MD5Util;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.sql.Timestamp;
import java.text.ParseException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Created by Miraculous on 2016/12/19.
 */
@RestController
@RequestMapping("/innerapi")
public class InnerController implements IBaseController {

  private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(InnerController.class);

  @Autowired
  private IUserService userService;
  @Autowired
  private IUserDetailService userDetailService;
  @Autowired
  private IIdCardService idCardService;
  @Autowired
  private IUserExtInfoService userExtInfoService;
  @Autowired
  private IContactService contactService;
  @Autowired
  private IAddressService addressService;
  @Autowired
  private IWechatService wechatService;
  @Autowired
  private IUserSpouseService userSpouseService;
  @Autowired
  private ISessionService sessionService;
  @Autowired
  private IUserApiService userApiService;

  @Autowired
  private IHttpService httpService;
  @Autowired
  private IUserRegisterService userRegisterService;

  @Value("${xyqb.user.query.url}")
  private String queryUrl;

  @Value("${xyqb.auth.url}")
  private String hanguguanUrl;

  private static final ObjectMapper MAPPER = new ObjectMapper();

  static {
    MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    MAPPER.configure(SerializationFeature.WRITE_ENUMS_USING_INDEX, true);
  }
  private static final char[] PWD_BASE = {
          'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k',
          'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w',
          'x', 'y', 'z', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9'
  };

  @RequestMapping("/user/search/phoneNo")
  @LogHttpCaller
  public JsonResult findByPhoneNo(String phoneNo) {
    User user = userService.findByPhoneInDb(phoneNo);
    if (user == null) {
      return JsonResult.buildErrorStateResult("", null);
    }

    UserRet userRet = new UserRet(user);

    return JsonResult.buildSuccessResult("", userRet);
  }

  @RequestMapping("/user/search/uuid")
  @LogHttpCaller
  public JsonResult findByUuid(String uuid) {
    User user = userService.findByUuidInDb(uuid);
    if (user == null) {
      return JsonResult.buildErrorStateResult("", null);
    }
    UserRet userRet = new UserRet(user);
    return JsonResult.buildSuccessResult("", userRet);
  }

  @RequestMapping("/userInfo/search/uuid")
  @LogHttpCaller
  public JsonResult findUserInfoByUuid(@RequestParam(value = "uuid") String uuid) {
    LOGGER.info("需要查询的用户uuidid, uuid:" + uuid);

    if (StringUtils.isBlank(uuid)) {
      return JsonResult.buildSuccessResult(null, null);
    }
    User user = userService.findByUuidInDb(uuid);
    if(null!=user){
      if(!user.getEnable()){
        return JsonResult.buildSuccessResult("", null);
      }
      UserDetail userDetail = userDetailService.findByUserId(user.getId());
      UserInfo info = new UserInfo(user,userDetail);
      return JsonResult.buildSuccessResult("", info);
    }
    return JsonResult.buildSuccessResult("", null);
  }
  @RequestMapping("/userInfo/search/phone")
  @LogHttpCaller
  public JsonResult findUserInfoByPhone(@RequestParam(value = "phone") String phone) {
    LOGGER.info("需要查询的用户phone, phone:" + phone);

    if (StringUtils.isBlank(phone)) {
      return JsonResult.buildSuccessResult(null, null);
    }
    User user = userService.findByPhoneInDb(phone);
    if(null!=user){
      if(!user.getEnable()){
        return JsonResult.buildSuccessResult("", null);
      }
      UserDetail userDetail = userDetailService.findByUserId(user.getId());
      UserInfo info = new UserInfo(user,userDetail);
      return JsonResult.buildSuccessResult("", info);
    }
    return JsonResult.buildSuccessResult("", null);
  }


  @RequestMapping("/user/getPhoneByUserIds")
  @LogHttpCaller
  public JsonResult findByIds(@RequestParam(value = "userIds") String userIdsString) {
    LOGGER.info("批量查询用户的手机号列表, userIdsString:" + userIdsString);
    if (StringUtils.isEmpty(userIdsString)) {
      return JsonResult.buildErrorStateResult(null, null);
    }
    List<Long> userIds = JSONObject.parseObject(userIdsString, new TypeReference<List<Long>>() {
    });
    if (!CollectionUtils.isEmpty(userIds) && userIds.size() <= 500) {
      Map<Long, String> userIdAndPhoneMap = userService.findPhoneByIdsInDb(userIds);
      return JsonResult.buildSuccessResult("", userIdAndPhoneMap);
    } else {
      return JsonResult.buildErrorStateResult("批量查询每次最多进行500条用户信息的查询", null);
    }
  }

  @RequestMapping("/user/save")
  public JsonResult saveUser(
      String phoneNo, Long registeredFrom, Long createdAt, Long updatedAt,
      String password, String uuid) {
    LOGGER.info("保存用户，phoneNo：{},registeredFrom;{},uuid:{}",phoneNo,registeredFrom,uuid);
    //参数验证
    if (StringUtils.isBlank(phoneNo)) {
      return JsonResult.buildErrorStateResult("用户手机号不能为空.", null);
    }
    if (registeredFrom == null) {
      registeredFrom = 0L;
    }
    if (StringUtils.isBlank(password)) {
      password= genRandomPwd();
      password = PasswordUtil.MD5(password.toLowerCase() + Constants.PASSWORD_SALT);
    }
    if (StringUtils.isBlank(uuid)) {
      return JsonResult.buildErrorStateResult("用户uuid为空.", null);
    }
    if (createdAt == 0L || updatedAt == 0L) {
      createdAt = System.currentTimeMillis();
      updatedAt = System.currentTimeMillis();
    }
    User user = userService.findByPhoneWithCache(phoneNo);
    if (user == null) {
      user = new User();
    } else if (!user.getEnable()) {
      LOGGER.info("用户已经注销，phoneNo：{},",phoneNo);
      return JsonResult.buildErrorStateResult("用户已经注销", null);
    }
    if (org.apache.commons.lang.StringUtils.isEmpty(user.getUuid())) {
      user.setUuid(uuid);
    }
    user.setPhoneNo(phoneNo);
    user.setCreatedAt(new Timestamp(createdAt));
    user.setUpdatedAt(new Timestamp(updatedAt));
    user.setEnable(true);
    user.setRegisteredFrom(registeredFrom);
    user.setUuid(uuid);
    user.setPassword(password);
    user = userService.saveUser(user);
    UserRet userRet = null;
    if (user != null) {
      userRet = MqUtils.sendRegisterMessage(registeredFrom, null,user);
    }
    return JsonResult.buildSuccessResult(null, userRet);
  }


  /**
   * 保存用户详细信息
   */
  @RequestMapping("/user_detail/save")
  public JsonResult saveUserDetail(Long userId, String phoneNo, String name, String idNo,
      String email, Long id) {
    LOGGER.info("保存用户详细信息，phoneNo：{},userId;{},name:{}，idNo：{},email;{}",phoneNo,userId,name,idNo,email);
    //参数验证
    if (userId == null || userId == 0L) {
      return JsonResult.buildErrorStateResult("用户id为空.", null);
    }
    if (StringUtils.isBlank(phoneNo)) {
      return JsonResult.buildErrorStateResult("用户手机号为空.", null);
    }
    if (StringUtils.isBlank(name)) {
      return JsonResult.buildErrorStateResult("用户姓名为空.", null);
    }
    if (StringUtils.isBlank(idNo)) {
      return JsonResult.buildErrorStateResult("用户身份证为空.", null);
    }
    if (!ValidationUtil.validateChinese(name)) {
      return JsonResult.buildErrorStateResult("用户姓名错误", null);
    }
    IdCardInfo info = null;
    try {
      info = idCardService.getIdCardInfo(idNo);
    } catch (ParseException ex) {
      LOGGER.error("用户身份证号错误, idNo: {}", idNo);
      return JsonResult.buildErrorStateResult("身份证号码错误", null);
    }
    if (Objects.isNull(info) || !info.isValid()) {
      return JsonResult.buildErrorStateResult("身份证号码错误", null);
    }
    UserDetail userDetail=userDetailService.findByUserId(userId);
    Timestamp time = new Timestamp(System.currentTimeMillis());
    if(null==userDetail){
      userDetail = new UserDetail();
      userDetail.setCreatedAt(time);
    }
    if (id != null && id > 0) {
      userDetail.setId(id);
    }
    userDetail.setUserId(userId);
    userDetail.setName(name);
    userDetail.setPhoneNo(phoneNo);
    userDetail.setIdNo(idNo);
    userDetail.setUpdatedAt(time);
    userDetail.setIdType(IdType.ID_CARD);
    userDetail.setGender(info.getGender());
    userDetail.setEmail(email);
    userDetail = userDetailService.saveUserDetail(userDetail);
    if (userDetail != null) {
      return JsonResult.buildSuccessResult(null, UserDetailRet.getUserDetail(userDetail));
    }
    return JsonResult.buildErrorStateResult("", null);
  }

  /**
   * 根据用户id查询用户的详细信息
   */
  @RequestMapping("/user_detail/search/userId")
  @LogHttpCaller
  public JsonResult findUserDetailByUserId(Long userId) {
    UserDetail userDetail = userDetailService.findByUserId(userId);
    if (userDetail != null) {
      return JsonResult.buildSuccessResult(null, UserDetailRet.getUserDetail(userDetail));
    }
    return JsonResult.buildErrorStateResult("", null);
  }

  @RequestMapping("/user/search/userId")
  @LogHttpCaller
  public JsonResult findUserByUserId(Long userId) {
    User user = userService.findById(userId);
    if (user != null) {
      return JsonResult.buildSuccessResult(null, new UserRet(user));
    }
    return JsonResult.buildErrorStateResult("", null);
  }

  @RequestMapping("/user_detail/search/phone")
  @LogHttpCaller
  public JsonResult findUserDetailByPhone(String phoneNo) {
    UserDetail userDetail = userDetailService.findByPhoneNo(phoneNo);
    if (userDetail != null) {
      return JsonResult.buildSuccessResult(null, UserDetailRet.getUserDetail(userDetail));
    }
    return JsonResult.buildErrorStateResult("", null);
  }

  /**
   * 根据phone查找用户完整信息
   * @param phoneNo
   * @return
   */
  @RequestMapping("/user_full_info/search/phone")
  @LogHttpCaller
  public JsonResult findUserFullInfoByPhone(String phoneNo) {
    if (StringUtils.isBlank(phoneNo)) {
      LOGGER.warn("[findUserFullInfoByPhone]phoneNo为空");
      return JsonResult.buildErrorStateResult(null, null);
    }
    Optional<UserFullInfo> userFullInfoOptional = userApiService.getUserFullInfoByPhone(phoneNo);
    if (!userFullInfoOptional.isPresent()) {
      LOGGER.warn("[findUserFullInfoByPhone]没有用户信息，phoneNo={}", MD5Util.build(phoneNo));
      return JsonResult.buildErrorStateResult(null, null);
    }
    return JsonResult.buildSuccessResult(null, userFullInfoOptional.get());
  }

  /**
   * 根据uuid查找用户完整信息
   * @param uuid
   * @return
   */
  @RequestMapping("/user_full_info/search/uuid")
  @LogHttpCaller
  public JsonResult findUserFullInfoByUUuid(String uuid) {
    if (StringUtils.isBlank(uuid)) {
      LOGGER.warn("[findUserFullInfoByUUuid]uuid为空");
      return JsonResult.buildErrorStateResult(null, null);
    }
    Optional<UserFullInfo> userFullInfoOptional = userApiService.getUserFullInfoByUuid(uuid);
    if (!userFullInfoOptional.isPresent()) {
      LOGGER.warn("[findUserFullInfoByUUuid]没有用户信息，uuid={}", uuid);
      return JsonResult.buildErrorStateResult(null, null);
    }
    return JsonResult.buildSuccessResult(null, userFullInfoOptional.get());
  }

  @RequestMapping("/user_detail/update/qq")
  public JsonResult updateUserQQ(String qq, Long userId) {
    if (StringUtils.isEmpty(qq) || userId == null || userId == 0L) {
      return JsonResult.buildErrorStateResult("参数校验失败,qq或用户id为空", null);
    }
    userDetailService.updateUserQQ(userId, qq);
    return JsonResult.buildSuccessResult(null, null);
  }

  @RequestMapping("/contact/search/user_id")
  @LogHttpCaller
  public JsonResult findContactsByUserId(Long userId) {
    if (null == userId) {
      return JsonResult.buildErrorStateResult(null, null);
    }
    List<Contact> contacts = contactService.findByUserId(userId, true);
    if (null == contacts || contacts.size() == 0) {
      return JsonResult.buildErrorStateResult(null, Collections.emptyList());
    }
    return JsonResult.buildSuccessResult(null, ContactRet.contacts2ContactRets(contacts));
  }

  @RequestMapping("/contact/save/contacts")
  public JsonResult save2Contact(Long userId, @RequestParam(value = "contacts") String contactsStr) {
    LOGGER.info("保存用户联系人:userId:{}, contacts:{}",userId, contactsStr);
    if (StringUtils.isEmpty(contactsStr)) {
      return JsonResult.buildErrorStateResult(null, null);
    }
    if (userId == null) {
      LOGGER.info("用户ID不能为空");
      return JsonResult.buildErrorStateResult(null, null);
    }
    List<Contact> contacts = JSONObject.parseObject(contactsStr, new TypeReference<List<Contact>>(){});
    if (CollectionUtils.isEmpty(contacts)) {
      LOGGER.info("联系人不能空");
      return JsonResult.buildErrorStateResult(null, null);
    }
    for(Contact contact : contacts){
      if (!contact.valid()) {
        LOGGER.info("用户手机号或姓名错误, phoneNo:{},name:{}",  contact.getPhoneNo(),  contact.getName());
        return JsonResult.buildErrorStateResult(null, null);
      }
    }
    List<Contact> result = contactService.save(userId, contacts);
    return JsonResult.buildSuccessResult(null, ContactRet.contacts2ContactRets(result));
  }

  @RequestMapping("/contact/update/contact")
  public JsonResult updateContact(@RequestParam Long contactId, @RequestParam(required = false) String name,
                                  @RequestParam(required = false) String phoneNo,
                                  @RequestParam(required = false) Relation relation, String key,
                                  @RequestParam String reason,HttpServletRequest request) {
    if (!"@qwsdedad131323213w!".equals(key) || contactId == null) {
      return JsonResult.buildErrorStateResult("参数不合法", null);
    }

    if (StringUtils.isEmpty(name) && StringUtils.isEmpty(phoneNo) ){
      return JsonResult.buildErrorStateResult("修改联系人修改条件不能都为空", null);
    }
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      LOGGER.info("用户手机号错误, phoneNo:{}",  phoneNo);
      return JsonResult.buildErrorStateResult("用户手机号错误", null);
    }
    if (!ValidationUtil.validateChinese(name)) {
      LOGGER.info("用户姓名错误, name:{}",  name);
      return JsonResult.buildErrorStateResult("用户姓名错误", null);
    }
    if(StringUtils.isEmpty(reason)){
      return JsonResult.buildErrorStateResult("修改原因不能为空", null);
    }
    Contact contact = contactService.findById(contactId);
    if (null == contact) {
      return JsonResult.buildErrorStateResult("修改联系人不存在", null);
    }
    contact = contactService.saveContact(name, phoneNo, relation, contact);
    LOGGER.info("修改后联系人信息:{},修改原因：{},操作ip:{}",contact,reason,IPUtil.getRemoteIP(request));

    return JsonResult.buildSuccessResult("修改联系人成功", contact);
  }

  @RequestMapping("/address/search/user_id")
  public JsonResult findAddressByUserId(Long userId) {
    if (userId == null) {
      return JsonResult.buildErrorStateResult(null, null);
    }
    Address address = addressService.findByUserId(userId);
    if (address == null) {
      return JsonResult.buildErrorStateResult(null, null);
    }
    return JsonResult.buildSuccessResult(null, AddressRet.address2AddressRet(address));
  }

  @RequestMapping("/address/save")
  public JsonResult saveAddress(
      Long userId, Long provinceCode, Long cityCode, String city,
      Long districtCode, String district, String address, String province) {
    LOGGER.info("保存地址详情:city:{},province:{},district:{}, address:{},userId:{}" , city, province, district, address, userId);
    if (userId == null || provinceCode == null || cityCode == null || StringUtils.isBlank(address)) {
      LOGGER.info("错误保存地址详情，其中参数不能为空:city:{},province:{},district:{}, address:{},userId:{}" , city, province, district, address, userId);
      return JsonResult.buildErrorStateResult("参数不能为空", null);
    }
    Address addressObj = addressService.findByUserId(userId);
    if (addressObj == null) {
      addressObj = new Address();
      Timestamp now = new Timestamp(System.currentTimeMillis());
      addressObj.setCreatedAt(now);
      addressObj.setUpdateAt(now);
    }
    addressObj.setUserId(userId);
    addressObj.setProvinceCode(provinceCode);
    addressObj.setCityCode(cityCode);
    addressObj.setCity(city);
    addressObj.setDistrictCode(districtCode);
    addressObj.setDistrict(district);
    addressObj.setAddress(address);
    addressObj.setProvince(province);
    addressObj = addressService.save(addressObj);
    LOGGER.info("保存后地址详情:{}",addressObj);
    return JsonResult.buildSuccessResult(null, AddressRet.address2AddressRet(addressObj));
  }

  @RequestMapping("/user_ext_info/update")
  @LogHttpCaller
  public JsonResult updateMarryStatus(
      Long userId, IncomeEnum incomeEnum, IncomeRangeEnum incomeRangeEnum,
      OccupationEnum occupationEnum, EducationEnum educationEnum, Boolean hasCar,
      Boolean hasSocialSecurity, Boolean hasHouse, Boolean hasCreditCard, MaritalStatus maritalStatus) {
    if (null == userId) {
      return JsonResult.buildErrorStateResult("用户ID不能为空", null);
    }
    UserExtInfo info = userExtInfoService.findByUserId(userId);
    Timestamp now = new Timestamp(System.currentTimeMillis());
    if (info == null) {
      info = new UserExtInfo();
      info.setUserId(userId);
      info.setCreatedAt(now);
    }
    info.setUpdateAt(now);
    if (incomeEnum != null) {
      info.setIncomeEnum(cn.quantgroup.user.enums.IncomeEnum.valueOf(incomeEnum.name()));
    }
    if (incomeRangeEnum != null) {
      info.setIncomeRangeEnum(cn.quantgroup.user.enums.IncomeRangeEnum.valueOf(incomeRangeEnum.name()));
    }
    if (occupationEnum != null) {
      info.setOccupationEnum(cn.quantgroup.user.enums.OccupationEnum.valueOf(occupationEnum.name()));
    }
    if (educationEnum != null) {
      info.setEducationEnum(cn.quantgroup.user.enums.EducationEnum.valueOf(educationEnum.name()));
    }
    if (hasCar != null) {
      info.setHasCar(hasCar);
    }
    if (hasCreditCard != null) {
      info.setHasCreditCard(hasCreditCard);
    }

    if (hasSocialSecurity != null) {
      info.setHasSocialSecurity(hasSocialSecurity);
    }


    if (hasHouse != null) {
      info.setHasHouse(hasHouse);
    }
    if (maritalStatus != null) {
      info.setMarryStatus(cn.quantgroup.user.enums.MaritalStatus.valueOf(maritalStatus.name()));
    }
    info = userExtInfoService.save(info);
    return JsonResult.buildSuccessResult(null, UserExtInfoRet.getUserExtInfoRet(info));
  }

  @LogHttpCaller
  @RequestMapping("/user_detail/search_list")
  public JsonResult searchUserDetailList(String name, String phoneNo, String idNo) {

    LOGGER.info("searchUserDetailList ,param.name:{},phone:{},idNo:{},ip:{}", name, phoneNo, idNo, getIp());
    if (StringUtils.isBlank(name) && StringUtils.isBlank(phoneNo) && StringUtils.isBlank(idNo)) {
      return JsonResult.buildErrorStateResult("至少必须满足一个条件不为空", null);
    }

    /*List<UserDetailVO> userDetails = userDetailService.searchUserDetailList(name, phoneNo, idNo);
    return JsonResult.buildSuccessResult("success", userDetails);*/

    ImmutableMap.Builder<String, String> builder = ImmutableMap.builder();
    if (StringUtils.isNotBlank(name)) {
      builder.put("name", name);
    }
    if (StringUtils.isNotBlank(phoneNo)) {
      builder.put("phoneNo", phoneNo);
    }
    if (StringUtils.isNotBlank(idNo)) {
      builder.put("idNo", idNo);
    }
    Map<String, String> param = builder.build();
    String s = httpService.get(queryUrl.concat("/innerapi/user_detail/search_list"), param);
    return JSON.parseObject(s, JsonResult.class);
  }

  @RequestMapping("/user_ext_info/search/user_id")
  @LogHttpCaller
  public JsonResult searchUserExtInfoByUserId(Long userId) {
    if (userId == null) {
      return JsonResult.buildErrorStateResult("userId不能为空", null);
    }
    UserExtInfo userExtInfo = userExtInfoService.findByUserId(userId);
    if (userExtInfo == null) {
      return JsonResult.buildErrorStateResult(null, null);
    }
    return JsonResult.buildSuccessResult(null, UserExtInfoRet.getUserExtInfoRet(userExtInfo));
  }

  @RequestMapping("/user/query/openId")
  @LogHttpCaller
  public JsonResult queryOpenIdByUserId(Long userId) {
    if (userId == null) {
      return JsonResult.buildErrorStateResult("userId不能为空", null);
    }
    WechatUserInfo wechatUserInfo = wechatService.queryOpenIdByUserId(userId);
    if (wechatUserInfo == null) {
      return JsonResult.buildErrorStateResult(null, null);
    }
    return JsonResult.buildSuccessResult("success", wechatUserInfo.getOpenId());
  }

  @LogHttpCaller
  @RequestMapping("/user-association/search/phone")
  public JsonResult findUserAssociationByPhone(String phoneNo) {
    UserDetail userDetail = userDetailService.findByPhoneNo(phoneNo);
    UserAssociation bean = new UserAssociation();

    if (null != userDetail) {
      Address address = addressService.findByUserId(userDetail.getUserId());
      List<Contact> contacts = contactService.findByUserId(userDetail.getUserId(), true);
      UserExtInfo extInfo = userExtInfoService.findByUserId(userDetail.getUserId());
      bean.setUserId(userDetail.getUserId());
      bean.setPhoneNo(userDetail.getPhoneNo());
      bean.setName(userDetail.getName());
      bean.setIdNo(userDetail.getIdNo());
      bean.setQq(userDetail.getQq());
      bean.setGender(Optional.ofNullable(userDetail.getGender()).orElse(cn.quantgroup.xyqb.model.Gender.UNKNOWN).getName());

      if (null != address) {
        bean.setCity(address.getCity());
        bean.setDistrict(address.getDistrict());
        bean.setAddress(address.getAddress());
        bean.setProvince(address.getProvince());
      }
      if (null != extInfo) {
        if(null==extInfo.getEducationEnum()){
          bean.setEducationEnum(cn.quantgroup.user.enums.EducationEnum.UNKNOWN.getName());
        }else{
          bean.setEducationEnum(extInfo.getEducationEnum().getName());
        }
        if(null==extInfo.getOccupationEnum()){
          bean.setOccupationEnum(cn.quantgroup.user.enums.OccupationEnum.UNKNOWN.getName());
        }else{
          bean.setOccupationEnum(extInfo.getOccupationEnum().getName());
        }

        if(null==extInfo.getMarryStatus()){
          bean.setMarryStatus(cn.quantgroup.user.enums.MaritalStatus.UNKNOWN.getDescription());
        }else{
          bean.setMarryStatus(extInfo.getMarryStatus().getDescription());
        }

      }
      if (org.apache.commons.collections.CollectionUtils.isNotEmpty(contacts)) {
        List<XContactInfo> userContacts = contacts2ContactRets(contacts);
        bean.setContractResults(userContacts);
      }
    }

    return JsonResult.buildSuccessResult("", bean);
  }

  @RequestMapping("/user-association/search/uid")
  @LogHttpCaller
  public JsonResult findUserAssociationByUid(Long uid) {
    UserDetail userDetail = userDetailService.findByUserId(uid);
    UserAssociation bean = new UserAssociation();

    if (null != userDetail) {
      Address address = addressService.findByUserId(userDetail.getUserId());
      List<Contact> contacts = contactService.findByUserId(userDetail.getUserId(), true);
      UserExtInfo extInfo = userExtInfoService.findByUserId(userDetail.getUserId());
      bean.setUserId(userDetail.getUserId());
      bean.setPhoneNo(userDetail.getPhoneNo());
      bean.setName(userDetail.getName());
      bean.setIdNo(userDetail.getIdNo());
      bean.setQq(userDetail.getQq());
      bean.setGender(Optional.ofNullable(userDetail.getGender()).orElse(cn.quantgroup.xyqb.model.Gender.UNKNOWN).getName());

      if (null != address) {
        bean.setCity(address.getCity());
        bean.setDistrict(address.getDistrict());
        bean.setAddress(address.getAddress());
        bean.setProvince(address.getProvince());
      }
      if (null != extInfo) {
        if(null==extInfo.getEducationEnum()){
          bean.setEducationEnum(cn.quantgroup.user.enums.EducationEnum.UNKNOWN.getName());
        }else{
          bean.setEducationEnum(extInfo.getEducationEnum().getName());
        }
        if(null==extInfo.getOccupationEnum()){
          bean.setOccupationEnum(cn.quantgroup.user.enums.OccupationEnum.UNKNOWN.getName());
        }else{
          bean.setOccupationEnum(extInfo.getOccupationEnum().getName());
        }

        if(null==extInfo.getMarryStatus()){
          bean.setMarryStatus(cn.quantgroup.user.enums.MaritalStatus.UNKNOWN.getDescription());
        }else{
          bean.setMarryStatus(extInfo.getMarryStatus().getDescription());
        }
      }
      if (org.apache.commons.collections.CollectionUtils.isNotEmpty(contacts)) {
        List<XContactInfo> userContacts = contacts2ContactRets(contacts);
        bean.setContractResults(userContacts);
      }
    }

    return JsonResult.buildSuccessResult("", bean);
  }

  public static XContactInfo contact2ContactRet(Contact c) {
    if (c == null) {
      return null;
    }
    XContactInfo ret = new XContactInfo();
    ret.setId(c.getId());
    ret.setUserId(c.getUserId());
    ret.setName(c.getName());
    ret.setPhoneNo(c.getPhoneNo());
    ret.setRelation(c.getRelation().getDescription());
    return ret;
  }

  public static List<XContactInfo> contacts2ContactRets(List<Contact> cs) {
    if (org.apache.commons.collections.CollectionUtils.isEmpty(cs)) {
      return Collections.emptyList();
    }
    List<XContactInfo> contactRets = new ArrayList<>();
    for (Contact c : cs) {
      contactRets.add(contact2ContactRet(c));
    }
    return contactRets;
  }
  @RequestMapping("/user/wechat/phone_no")
  @LogHttpCaller
  public JsonResult queryOpenIdByPhoneNo(String phoneNo) {
    if (StringUtils.isBlank(phoneNo)) {
      return JsonResult.buildErrorStateResult("手机号错误", null);
    }
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return JsonResult.buildErrorStateResult("手机号格式错误", null);
    }
    WechatUserInfo wechatUserInfo = wechatService.findWechatUserInfoByPhoneNo(phoneNo);
    return JsonResult.buildSuccessResult(null, null == wechatUserInfo ? null : wechatUserInfo.getOpenId());
  }

  @RequestMapping("/user/disable")
  public JsonResult disableUser(Long userId) {
    if (null == userId || 0L == userId) {
      return JsonResult.buildErrorStateResult("userId不能为空", null);
    }
    User user = userService.findById(userId);
    if (null == user) {
      return JsonResult.buildErrorStateResult("未查询到该用户,用户id:" + userId, null);
    }
    user.setEnable(false);
    user.setUpdatedAt(new Timestamp(System.currentTimeMillis()));
    LOGGER.info("注销用户的信息,userId={}", userId);
    user = userService.saveUser(user);
    if (!user.getEnable()) {
      sessionService.deleteByUserId(userId);
      sessionService.deleteUserCatch(user);
    }
    return JsonResult.buildSuccessResult("用户已禁用.", user.getEnable() == false);
  }

  /**
   * 清除用户缓存信息
   * @param phoneNo
   * @return
   */
  @RequestMapping("/flush/cache")
  public JsonResult flushCache(String phoneNo) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return JsonResult.buildErrorStateResult("手机号格式错误", null);
    }
    User user = userService.findByPhoneInDb(phoneNo);
    if (null == user) {
      return JsonResult.buildErrorStateResult("未查询到该用户,用户phoneNo:" + phoneNo, null);
    }
    sessionService.deleteByUserId(user.getId());
    sessionService.deleteUserCatch(user);
    LOGGER.info("删除用户缓存信息，phoneNo:{},userId:{},", user.getPhoneNo(), user.getId());
    return JsonResult.buildSuccessResult("用户缓存信息已经删除", null);
  }
  @RequestMapping("/user/spouse/save")
  public JsonResult saveSpouse(Long userId, MaritalStatus status, String spousePhone, String spouseName) {
    if (userId == null || userId == 0) {
      return JsonResult.buildErrorStateResult("用户不能为空", null);
    }
    if (status == MaritalStatus.MARRIED) {
      if (!ValidationUtil.validatePhoneNo(spousePhone)) {
        return JsonResult.buildErrorStateResult("手机号格式错误", null);
      }
      if (!ValidationUtil.validateChinese(spouseName)) {
        return JsonResult.buildErrorStateResult("配偶姓名错误", null);
      }
    }
    UserSpouse userSpouse = userSpouseService.findByUserId(userId);
    Timestamp timestamp = new Timestamp(System.currentTimeMillis());
    if (userSpouse == null) {
      userSpouse = new UserSpouse(userId);
      userSpouse.setCreatedAt(timestamp);
    }
    userSpouse.setSpouseName(status == MaritalStatus.MARRIED ? spouseName : "");
    userSpouse.setSpousePhone(status == MaritalStatus.MARRIED ? spousePhone : "");
    userSpouse.setStatus(cn.quantgroup.user.enums.MaritalStatus.valueOf(status.name()));
    userSpouse.setUpdateAt(timestamp);
    userSpouse = userSpouseService.save(userSpouse);
    return JsonResult.buildSuccessResult(null, UserSpouseRet.getUserSpouseRet(userSpouse));
  }

  @RequestMapping("/user/spouse/findByUserId")
  @LogHttpCaller
  public JsonResult querySpouse(Long userId) {
    if (userId == null || userId == 0) {
      return JsonResult.buildErrorStateResult("用户不能为空", null);
    }
    UserSpouse userSpouse = userSpouseService.findByUserId(userId);
    if (userSpouse == null || !userSpouse.valid()) {
      userSpouse = new UserSpouse(userId);
      userSpouse.setStatus(cn.quantgroup.user.enums.MaritalStatus.UNKNOWN);
    }
    return JsonResult.buildSuccessResult(null, UserSpouseRet.getUserSpouseRet(userSpouse));
  }

  @RequestMapping("/user/findByPhones")
  public JsonResult getUserIdByPhones(@RequestParam("userPhones") String userPhones) {
    if (StringUtils.isBlank(userPhones)) {
      return JsonResult.buildErrorStateResult("传入用户手机号不可为空", null);
    }
    List<String> phones = JSONObject.parseObject(userPhones, new TypeReference<List<String>>() {
    });
    if (org.apache.commons.collections.CollectionUtils.isNotEmpty(phones)) {
      if (!(phones.size() > 500)) {
        List<User> users = userService.findByPhones(phones);
        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(users)) {
          return JsonResult.buildSuccessResult(null, users.stream().collect(Collectors.toMap(User::getPhoneNo, User::getId)));
        } else {
          return JsonResult.buildSuccessResult(null, null);
        }
      }
      return JsonResult.buildErrorStateResult("单次批量查询不可超过500个手机号", null);
    }
    return JsonResult.buildErrorStateResult("传入用户手机号不可为空", null);
  }

  /**
   * 手机号批量查询uuid
   * @param userPhones
   * @return
   */
  @RequestMapping("/uuid/findByPhones")
  public JsonResult getUuidsByPhones(@RequestParam("userPhones") String userPhones) {

    return JsonResult.buildErrorStateResult("主库不提供查询", null);
  }
  /**
   * 重置密码接口，供内部人员使用（例如绝影）
   */
  @RequestMapping("/user/password/reset/{key}/{phone}")
  public JsonResult resetPassword(@PathVariable("key") String key, @PathVariable("phone") String phone) {
    LOGGER.info("密码重置请求,phone:[{}]", phone);
    if (StringUtils.isBlank(key)) {
      return JsonResult.buildErrorStateResult("用户密码重置失败.", null);
    }
    if (!"ecf75c1f-2ccb-4661-8e4b-2874c0f45a2b".equalsIgnoreCase(key)) {
      LOGGER.error("密码重置失败,key错误!@!,phone:[{}]", phone);
      return JsonResult.buildErrorStateResult("用户密码重置失败.", null);
    }
    try {
      userService.resetPassword(phone, "123456"); //默认重置的密码是123456
      return JsonResult.buildSuccessResult("用户密码已重置.", "null");
    } catch (Exception e) {
      LOGGER.error("密码重置失败,phone[{}],err:[{}]", phone, e);
    }
    return JsonResult.buildErrorStateResult("用户密码重置失败.", null);
  }

  /**
   * 保存用户信息，地址信息，联系人信息
   */
  @RequestMapping("/user/save_multi")
  @LogHttpCaller
  public JsonResult saveMulti(
      String registeredFrom,
      String channelId,
      String phoneNo,
      String name,
      String idNo,
      String provinceCode,
      String province,
      String cityCode,
      String city,
      String districtCode,
      String district,
      String address,
      String contacts) {
    LOGGER.info("[/innerapi/user/save_multi]保存用户、联系人、地址信息:registeredFrom:[{}], phoneNo[{}]，name[{}]，idNo[{}]，provinceCode[{}]，cityCode[{}]，districtCode[{}]，" +
        "address[{}]，contacts[{}]", registeredFrom, phoneNo, name, idNo, provinceCode, cityCode, districtCode, address, contacts);

    if (!NumberUtils.isDigits(registeredFrom)) {
      return JsonResult.buildErrorStateResult("注册渠道异常.", null);
    }

    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return JsonResult.buildErrorStateResult("手机号异常.", null);
    }
    if (StringUtils.isBlank(name)) {
      return JsonResult.buildErrorStateResult("用户名异常.", null);
    }
    boolean isIdCard = true;
    try {
      isIdCard = idCardService.isIdCardValid(idNo);
    } catch (ParseException e) {
      isIdCard = false;
    }
    if (!isIdCard) {
      return JsonResult.buildErrorStateResult("用户身份证号异常.", null);
    }
    if (!NumberUtils.isDigits(provinceCode)) {
      return JsonResult.buildErrorStateResult("省份编号异常.", null);
    }
    if (StringUtils.isBlank(province)) {
      return JsonResult.buildErrorStateResult("省份异常.", null);
    }
    if (!NumberUtils.isDigits(cityCode)) {
      return JsonResult.buildErrorStateResult("城市编号异常.", null);
    }
    if (StringUtils.isBlank(city)) {
      return JsonResult.buildErrorStateResult("城市异常.", null);
    }
    if (!NumberUtils.isDigits(districtCode)) {
      return JsonResult.buildErrorStateResult("区县编号异常.", null);
    }
    if (StringUtils.isBlank(district)) {
      return JsonResult.buildErrorStateResult("区县异常.", null);
    }
    if (StringUtils.isBlank(address)) {
      return JsonResult.buildErrorStateResult("详细地址异常.", null);
    }

    channelId = MoreObjects.firstNonNull(channelId, "-1");
    // 验证用户是否已存在
    User user = userService.findByPhoneInDb(phoneNo);
    if (null != user) {
      return JsonResult.buildErrorStateResult("用户已存在，手机号被占用", null);
    }

    user = userRegisterService.register(Long.valueOf(registeredFrom), Long.valueOf(channelId), phoneNo, name, idNo, provinceCode, province, cityCode, city, districtCode, district, address, contacts);

    UserRet userRet = new UserRet(user);
    return JsonResult.buildSuccessResult(null, userRet);
  }

  @RequestMapping("/user/register")
  public JsonResult register(String phoneNo, String password, Long channelId) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return JsonResult.buildErrorStateResult("手机号不正确", null);
    }
    LOGGER.info("[/innerapi/user/register]channelId={},ip={}", channelId, getIp());
    User user = userService.findByPhoneInDb(phoneNo);
    if (user == null) {
      user = userRegisterService.register(phoneNo, StringUtils.defaultString(password, ""), channelId);
    }
    if (user == null) {
      return JsonResult.buildErrorStateResult("注册失败", null);
    }
    return JsonResult.buildSuccessResult(null, new UserRet(user));
  }

  //根据日期时间段查询新注册用户信息并返回

  @RequestMapping("/contract/queryRegisterUsers")
  @LogHttpCaller
  public JsonResult findRegisterUserByTime(String beginTime,String endTime) {
    if (null == beginTime||endTime==null) {
      return JsonResult.buildErrorStateResult(null, null);
    }
    List<User> userList=userService.findRegisterUserByTime(beginTime,endTime);
    return JsonResult.buildSuccessResult(null, userList);
  }

  @RequestMapping("/forbiddenUserWeChat")
  public JsonResult forbiddenUserWeChat(@RequestParam Long userId, @RequestParam String reason, @RequestParam String content) {
    if (StringUtils.isBlank(reason) || null == userId || userId <= 0) {
      return JsonResult.buildErrorStateResult("参数有误", null);
    }
    //密文
    if (null != content) {
      content = content.replaceAll(" ", "+");
    }
    String str = AESUtil.decryptAfterBase64Decode(content, Constants.AES_KEY);
    if (null == str || !str.equals(userId.toString())) {
      return JsonResult.buildErrorStateResult("解密有误", null);
    }
    int affectedRows=wechatService.forbiddenUserWeChat(userId);
    LOGGER.info("用户微信禁用,userId:{},禁用的原因reason:{},受影响的行数affectedRows:{}",userId,reason,affectedRows);
    return JsonResult.buildSuccessResult("用户禁用微信成功", affectedRows);
  }

  @RequestMapping("/updateIdCard")
  public JsonResult updateIdCard(@RequestParam(required = false)String name, @RequestParam(required = false)String idNo,
                                 @RequestParam String phoneNo, @RequestParam String reason, @RequestParam String content) {
    int x = 1 / 0;
    //密文
    if (null != content) {
      content = content.replaceAll(" ", "+");
    }
    String str = AESUtil.decryptAfterBase64Decode(content, Constants.AES_KEY);
    if (null == str || !str.equals(phoneNo)) {
      return JsonResult.buildErrorStateResult("解密有误", null);
    }
    if(StringUtils.isBlank(name) && StringUtils.isBlank(idNo)){
      return JsonResult.buildErrorStateResult("不能全为空", null);
    }
    if ( StringUtils.isBlank(phoneNo) || StringUtils.isBlank(reason)) {
      return JsonResult.buildErrorStateResult("参数有误", null);
    }
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      LOGGER.info("手机号有误, phoneNo:{}", phoneNo);
      return JsonResult.buildErrorStateResult("用户手机号错误", null);
    }
    int affectedRows=userDetailService.updateIdCard(name, idNo, phoneNo);
    LOGGER.info("更新用户的信息,name;{},idNo:{},phoneNo:{},操作的理由reason:{},受影响的行数affectedRows:{}", name, idNo, phoneNo, reason,affectedRows);
    ImmutableMap.Builder<String, String> builder = ImmutableMap.builder();
    builder.put("phoneNo", phoneNo);
    builder.put("name", name);
    builder.put("idNo", idNo);
    builder.put("reason", reason);
    builder.put("content", content);
    LOGGER.info("去函谷关修改用户身份证号或姓名");
    Map<String, String> param = builder.build();
    httpService.get(hanguguanUrl.concat("/innerapi/updateIdCard"), param);
    return JsonResult.buildSuccessResult("修改用户身份证号或姓名成功", affectedRows);
  }

  @RequestMapping("/forbiddenUserOrNot")
  public JsonResult forbiddenUserOrNot(@RequestParam String phoneNo, @RequestParam Boolean enable,
                                       @RequestParam String reason, @RequestParam String content) {
    if (StringUtils.isBlank(phoneNo) || StringUtils.isBlank(reason)) {
      return JsonResult.buildErrorStateResult("参数有误", null);
    }
    //密文
    if (null != content) {
      content = content.replaceAll(" ", "+");
    }
    String str = AESUtil.decryptAfterBase64Decode(content, Constants.AES_KEY);
    if (null == str || !str.equals(phoneNo)) {
      return JsonResult.buildErrorStateResult("解密有误", null);
    }
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      LOGGER.info("用户手机号错误, phoneNo:{}", phoneNo);
      return JsonResult.buildErrorStateResult("用户手机号错误", null);
    }

    int affectedRows=userService.forbiddenUser(enable, phoneNo);
    LOGGER.info("禁用或者重启用户的理由,reason:{},手机号phoneNo:{},受影响的行数affectedRows:{}", reason, phoneNo,affectedRows);
    ImmutableMap.Builder<String, String> builder = ImmutableMap.builder();
    builder.put("phoneNo", phoneNo);
    builder.put("enable", String.valueOf(enable));
    builder.put("reason", reason);
    builder.put("content", content);
    LOGGER.info("去函谷关禁用或者重启用户");
    Map<String, String> param = builder.build();
    httpService.get(hanguguanUrl.concat("/innerapi/forbiddenUserOrNot"), param);
    return JsonResult.buildSuccessResult("用户禁用或重启成功成功", affectedRows);
  }
  private String genRandomPwd() {
    int pwdMax = PWD_BASE.length;
    // 生成的随机数
    int i;
    // 生成的密码的长度
    int count = 0;
    StringBuffer pwd = new StringBuffer();
    Random r = new Random();
    while (count < 15) {
      // 生成的数最大为36-1
      i = Math.abs(r.nextInt(pwdMax));
      if (i >= 0 && i < PWD_BASE.length) {
        pwd.append(PWD_BASE[i]);
        count++;
      }
    }
    return pwd.toString();
  }

  @RequestMapping("/home")
  String home() {
    int x = 1 / 0;

    return "Hello World!";
  }
}