package cn.quantgroup.xyqb.filter;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.exception.BizException;
import cn.quantgroup.xyqb.exception.BizExceptionEnum;
import cn.quantgroup.xyqb.model.OauthResult;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.remote.StmsRemoteService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.session.XyqbSessionContextHolder;
import cn.quantgroup.xyqb.util.StringUtils;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Enumeration;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * stms权限认证拦截器
 */
@Slf4j
public class InnerInterceptor implements HandlerInterceptor {
    private final ISessionService sessionService;
    private final StmsRemoteService stmsRemoteService;

    public InnerInterceptor(ISessionService sessionService, StmsRemoteService stmsRemoteService) {
        this.sessionService = sessionService;
        this.stmsRemoteService = stmsRemoteService;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {

        //所有开放出去的外部接口，都需要验证租户id和注册来源
        String tenantId = request.getHeader(Constants.X_AUTH_TENANT);
        String registeredFrom = request.getHeader(Constants.X_AUTH_FROM);
        String stmsToken = request.getHeader(Constants.X_AUTH_TOKEN);
        if (StringUtils.isEmpty(tenantId)) {
            throw new BizException(BizExceptionEnum.UN_EXIT_TENANT_ID);
        }

        if (StringUtils.isEmpty(stmsToken)) {
            throw new BizException(BizExceptionEnum.UN_EXIT_STMS_TOKEN);
        }

        SessionStruct sessionStruct = XyqbSessionContextHolder.getXSessionFromRedis(stmsToken, Integer.valueOf(tenantId));

        if (sessionStruct == null) {
            OauthResult oauthResult = stmsRemoteService.checkToken(stmsToken);
            if (oauthResult != null && 2000 == oauthResult.getCode()) {
                LinkedHashMap<String, Object> linkedHashMap = (LinkedHashMap<String, Object>) oauthResult.getData();
                String userId = String.valueOf(linkedHashMap.get("id"));
                String userName = String.valueOf(linkedHashMap.get("name"));
                sessionStruct = XyqbSessionContextHolder.initSTMSSession(stmsToken, userId, userName);
                sessionService.persistSession(sessionStruct.getSid(), sessionStruct.getValues(), sessionStruct.getTenantId());
            } else {
                throw new BizException(BizExceptionEnum.UN_PERMISSION_STMS);
            }


        } else {
            //session续期
            sessionService.persistSession(sessionStruct.getSid(), sessionStruct.getValues(), sessionStruct.getTenantId());
        }

        //校验接口权限
        OauthResult permissionResult = stmsRemoteService.checkPermission(stmsToken,
                request.getRequestURI(), tenantId);
        if (permissionResult != null && 2000 == permissionResult.getCode()) {
            return true;
        } else {
            throw new BizException(BizExceptionEnum.UN_VALID_STMS_TOKEN);

        }
    }

    @Override
    public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler, ModelAndView modelAndView) throws Exception {

    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
        XyqbSessionContextHolder.releaseSession();
    }
}
