package cn.quantgroup.xyqb.aspect.captcha;


import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.util.ValidationUtil;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.nio.charset.Charset;
import java.util.*;

/**
 * 密码错误限次的校验
 * @author 任文超
 * @version 1.0.0
 * @since 2017-11-23
 */
@Aspect
@Component
public class PasswordErrorFiniteValidateAdvisor {

  private static final Logger LOGGER = LoggerFactory.getLogger(PasswordErrorFiniteValidateAdvisor.class);
  private static final String SUPER_CAPTCHA_ID = UUID.nameUUIDFromBytes("__QG_APPCLIENT_AGENT__".getBytes(Charset.forName("UTF-8"))).toString();
  private static final String SUPER_CAPTCHA = "__SUPERQG__";
  /**
   * 图形验证码错误计数器生命周期，与图形验证码生命周期保持一致
   * 参照点：cn.quantgroup.xyqb.config.captcha.RedisCaptchaStore{
   *   DEFAULT_EXPIRED_IN = 120L;
   *   DEFAULT_EXPIRED_TIMEUNIT = TimeUnit.SECONDS;
   * }
    */
  private static final Long SECONDS = 2 * 60L;

  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> redisTemplate;

  /**
   * 密码错误限次切面
   */
  @Pointcut("@annotation(cn.quantgroup.xyqb.aspect.captcha.PasswordErrorFiniteValidator)")
  private void passwordErrorFiniteValidate() {
  }

  /**
   * 在受保护的接口方法执行前, 执行密码错误限次校验
   *
   * @param pjp
   * @return
   * @throws Throwable
   */
  @Around("passwordErrorFiniteValidate()")
  private Object doFiniteValidate(ProceedingJoinPoint pjp) throws Throwable {
    HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
    // 客户端IP
    String clientIp = getIp(request);
    Long countErrorByIp = getIpv4Count(clientIp);
    // 5分钟内不超过3次
    if(countErrorByIp > Constants.Image_Need_Count){
      return JsonResult.buildErrorStateResult("登录失败", null);
    }
    return pjp.proceed();
  }

  // 获取该ipv4密码错误计数器
  private Long getIpv4Count(String ipv4) {
    if (ValidationUtil.validateIpv4(ipv4) && ValidationUtil.validateLocalIpv4(ipv4)) {
      return 0L;
    }
    String key = getIpKey(ipv4);
    String countString = redisTemplate.opsForValue().get(key);
    if (StringUtils.isBlank(countString)) {
      return 0L;
    }
    return Long.valueOf(countString);
  }

  private final static String getIpKey(String ipv4){
    return Constants.REDIS_PASSWORD_ERROR_COUNT_FOR_IPV4 + ipv4;
  }

  /**
   * 客户端IP解析
   *
   * @param request 当前请求，其首部行必须包含形如【SingleToken 13461067662:0123456789abcdef】的UTF-8编码的Base64加密参数
   * @return 客户端IP 或 null
   */
  private String getIp(HttpServletRequest request) {
    Objects.requireNonNull(request, "无效请求");
    String ip = request.getHeader("x-real-ip");
    if (StringUtils.isBlank(ip)) {
      ip = request.getRemoteAddr();
    }
    //过滤反向代理的ip
    String[] stemps = ip.split(",");
    if (stemps.length >= 1) {
      //得到第一个IP，即客户端真实IP
      ip = stemps[0];
    }
    ip = ip.trim();
    if (ip.length() > 23) {
      ip = ip.substring(0, 23);
    }
    return ip;
  }

}
