package cn.quantgroup.xyqb.aspect.logcaller;

import cn.quantgroup.xyqb.util.IPUtil;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.Objects;

/**
 * 调用者记录
 * Created by Administrator on 2017/5/15.
 */
@Aspect
@Component
@Order(value = Ordered.HIGHEST_PRECEDENCE)
public class LogCallHttpAspect {

    private static final Logger LOGGER = LoggerFactory.getLogger(LogCallHttpAspect.class);

    @Pointcut(value = "execution(public * cn.quantgroup.xyqb.controller..*.*(..)) " +
            "&& !execution(* cn.quantgroup.xyqb.controller.ExceptionHandlingController.*(..))")
    private void logHttpCaller() {
    }

    @Around("logHttpCaller()")
    public Object record(ProceedingJoinPoint pjp) throws Throwable {
        Object result = pjp.proceed();
        ServletRequestAttributes attrs = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        if (Objects.nonNull(attrs)) {
            try {
                HttpServletRequest request = attrs.getRequest();
                String remoteIP = IPUtil.getRemoteIP(request);
                this.asyncRecordIt(pjp, result, remoteIP);
            } catch (Exception e) {
                LOGGER.error("打印http请求日志出错", e);
            }
        }
        return result;
    }

    /**
     * 异步记录日志
     *
     * @param pjp
     * @param result
     * @param remoteIP
     */
    private void asyncRecordIt(ProceedingJoinPoint pjp, Object result, String remoteIP) {
        Object[] args = pjp.getArgs();
        String methodName = pjp.getSignature().getName();
        String targetName = pjp.getTarget().getClass().getName();
        LOGGER.info("http api is called,from IP:[{}],method :[{}],args:[{}]", remoteIP, targetName.concat(".").concat(methodName), args);
    }

}
