package cn.quantgroup.xyqb.service.v2;

import cn.quantgroup.xyqb.controller.req.v2.LoginReq;
import cn.quantgroup.xyqb.controller.req.v2.UserInfoConvertReq;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserInfoEntity;
import cn.quantgroup.xyqb.entity.UserTag;
import cn.quantgroup.xyqb.entity.WechatUserInfo;
import cn.quantgroup.xyqb.event.UserLoginEvent;
import cn.quantgroup.xyqb.exception.BizException;
import cn.quantgroup.xyqb.exception.BizExceptionEnum;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.LoginBean;
import cn.quantgroup.xyqb.model.ShopMapRsp;
import cn.quantgroup.xyqb.model.UserInfo;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.model.v2.login.BaseLoginParam;
import cn.quantgroup.xyqb.repository.IUserInfoRepository;
import cn.quantgroup.xyqb.repository.IUserRepository;
import cn.quantgroup.xyqb.repository.IWeChatUserRepository;
import cn.quantgroup.xyqb.session.XyqbSessionContextHolder;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestBody;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service
@Slf4j
public class LoginContext {
    private final ApplicationEventPublisher applicationEventPublisher;

    private final IUserRepository userRepository;

    private final IUserInfoRepository userInfoRepository;
    private final IWeChatUserRepository weChatUserRepository;

    private final Map<Integer, LoginStrategy> interloginStrategyMap = new HashMap<>();

    public LoginContext(List<LoginStrategy> loginStrategies, ApplicationEventPublisher applicationEventPublisher, IUserRepository userRepository,
                        IUserInfoRepository userInfoRepository,IWeChatUserRepository weChatUserRepository) {
        this.applicationEventPublisher = applicationEventPublisher;
        this.userRepository = userRepository;
        this.userInfoRepository = userInfoRepository;
        this.weChatUserRepository = weChatUserRepository;
        loginStrategies.forEach(i -> interloginStrategyMap.put(i.getType(), i));
    }

    public LoginBean login(LoginReq loginReq) {
        if (loginReq.getType() != null && loginReq.getType() == 9000) {
            throw new BizException(BizExceptionEnum.ERROR_INTERNAL_LOGIN);
        }
        return this.interLogin(loginReq);
    }


    /**
     * 内部统一登录
     *
     * @param loginReq LoginReq
     * @return LoginBean
     */
    public LoginBean interLogin(LoginReq loginReq) {

        //第一步、校验参数
        LoginStrategy loginStrategy = interloginStrategyMap.get(loginReq.getType());
        BaseLoginParam loginParam = loginStrategy.checkParam(loginReq);


        //第二步、执行登录操作
        LoginBean loginBean = loginStrategy.login(loginParam);


        //第三步、发送登录成功消息
        SessionStruct sessionStruct = XyqbSessionContextHolder.getXSession();
        Long registeredFrom = sessionStruct.getRegisteredFrom();
        if (registeredFrom == null && sessionStruct.getValues() != null && sessionStruct.getValues().getLoginProperties() != null) {
            registeredFrom = sessionStruct.getValues().getLoginProperties().getCreatedFrom();
        }

        UserTag userTag = UserTag.builder().userId(loginBean.getUserId()).registeredFrom(registeredFrom).tenantId(sessionStruct.getTenantId()).build();
        String geetestId = null;
        if (StringUtils.isNotEmpty(sessionStruct.getGeetestId())) {
            geetestId = sessionStruct.getGeetestId();
        }

        UserLoginEvent userLoginEvent = new UserLoginEvent(this, userTag, geetestId, loginBean.getRegister());
        applicationEventPublisher.publishEvent(userLoginEvent);


        //第四步，补充相关信息
        if (loginBean.getUserId() != null) {
            User user = userRepository.findByIdAndTenantId(loginBean.getUserId(), loginBean.getTenantId());

            if (loginReq.isAttachInfo()) {
                UserInfoEntity userInfoEntity = userInfoRepository.findByUserIdAndTenantId(loginBean.getUserId(), loginBean.getTenantId());
                loginBean.setInfo(new UserInfo(user, userInfoEntity));
            }
        }


        return loginBean;
    }

    /**
     * 消费地图历史数据清洗转换
     */
    public List<ShopMapRsp> shopMapDataConvert(UserInfoConvertReq req){
        List<UserInfoConvertReq.UserConvertInfo> userConvertInfoList = req.getUserConvertInfoList();
        if (CollectionUtils.isEmpty(userConvertInfoList) || userConvertInfoList.get(0).getUser() == null) {
            throw new BizException(BizExceptionEnum.ERROR_PARAM);
        }

        //phoneNo: User
        Map<String,User> phoneUserMap = Maps.newHashMap();

        //phoneNo： UserInfoEntity
        Map<String,UserInfoEntity> phoneUserEntityMap = Maps.newHashMap();

        //phoneNo： WechatUserInfo
        Map<String, WechatUserInfo> phoneWechatUserMap = Maps.newHashMap();
        List<String> phoneNoList = Lists.newArrayList();
        for (UserInfoConvertReq.UserConvertInfo convertInfo : userConvertInfoList) {
            phoneUserMap.put(convertInfo.getUser().getPhoneNo(), convertInfo.getUser());
            phoneUserEntityMap.put(convertInfo.getUser().getPhoneNo(), convertInfo.getUserInfoEntity());
            phoneWechatUserMap.put(convertInfo.getUser().getPhoneNo(), convertInfo.getWechatUserInfo());
            phoneNoList.add(convertInfo.getUser().getPhoneNo());
        }

        //用户注册
        List<User> registerUserList = registerUser(phoneNoList, phoneUserMap, req.getTenantId());
        List<Long> userIdList = registerUserList.stream().map(e -> e.getId()).collect(Collectors.toList());

        //用户扩展信息
        Map<Long, UserInfoEntity> userIdEntityMapQuery = Maps.newHashMap();
        List<UserInfoEntity> userInfoEntityListQuery = userInfoRepository.findByTenantIdAndUserIdIn(req.getTenantId(), userIdList);
        if (CollectionUtils.isNotEmpty(userInfoEntityListQuery)) {
            userIdEntityMapQuery = userInfoEntityListQuery.stream().collect(Collectors.toMap(UserInfoEntity::getUserId, Function.identity(), (k1, k2) -> k1));
        }


        //用户微信信息
        List<String> openIdList = phoneWechatUserMap.values().stream().map(e -> e.getOpenId()).collect(Collectors.toList());
        List<WechatUserInfo> wechatUserInfoListQuery = weChatUserRepository.findByOpenIdInAndAppIdAndTenantId(openIdList, req.getAppId(), req.getTenantId());

        Map<Long, WechatUserInfo> userIdWechatMapQuery = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(wechatUserInfoListQuery)) {
            userIdWechatMapQuery = wechatUserInfoListQuery.stream().collect(Collectors.toMap(WechatUserInfo::getUserId, Function.identity(), (k1, k2) -> k1));
        }

        for (User user : registerUserList) {
            UserInfoEntity userInfoEntity = phoneUserEntityMap.get(user.getPhoneNo());
            if (userInfoEntity != null && userIdEntityMapQuery.get(user.getId()) == null) {
                userInfoEntity.setUserId(user.getId());
                userInfoEntity.setTenantId(req.getTenantId());
            } else if (userInfoEntity != null && userIdEntityMapQuery.get(user.getId()) != null) {
                log.info("用户扩展信息已存在，userId:{},用户信息：{}", user.getId(), JSONObject.toJSONString(userInfoEntity));
            }


            //微信头像昵称正常存储
            WechatUserInfo wechatUserInfo = phoneWechatUserMap.get(user.getPhoneNo());
            if (wechatUserInfo != null && userIdWechatMapQuery.get(user.getId()) == null) {
                wechatUserInfo.setUserId(user.getId());
                wechatUserInfo.setTenantId(req.getTenantId());
            } else if (wechatUserInfo != null && userIdWechatMapQuery.get(user.getId()) == null) {
                log.info("用户微信信息已存在，userId:{},用户信息：{}", user.getId(), JSONObject.toJSONString(wechatUserInfo));
            }
        }

        Collection<WechatUserInfo> wechatUserInfos = phoneWechatUserMap.values();
        if (CollectionUtils.isNotEmpty(wechatUserInfos)) {
            weChatUserRepository.save(wechatUserInfos);
        }

        Collection<UserInfoEntity> entities = phoneUserEntityMap.values();
        if (CollectionUtils.isNotEmpty(entities)) {
            userInfoRepository.save(entities);
        }
        List<ShopMapRsp> shopMapRsps = new ArrayList<>();
        for (User user : registerUserList) {
            ShopMapRsp shopMapRsp = new ShopMapRsp();
            shopMapRsp.setId(user.getId());
            shopMapRsp.setPhoneNo(user.getPhoneNo());
            shopMapRsp.setUuid(user.getUuid());
            shopMapRsps.add(shopMapRsp);
        }

        return shopMapRsps;

        //手机号是否存在--注册 登录
        //微信是否存在-  若存在，和手机号是否匹配相同
        //头像昵称

        //用户存在--更新头像昵称 和微信信息

        //用户不存在--

    }


    private List<User> registerUser(List<String> phoneNoList,Map<String,User> phoneUserMap,Integer tenantId) {
        //查询已注册用户
        List<User> registeredUserList = userRepository.findByPhoneNoInAndTenantId(phoneNoList, tenantId);
        registeredUserList = CollectionUtils.isEmpty(registeredUserList) ? Lists.newArrayList() : registeredUserList;
        List<String> registeredPhoneList = registeredUserList.stream().map(e -> e.getPhoneNo()).collect(Collectors.toList());
        //TODO 可能冻结，或者注销

        //待注册
        List<String> needRegisterPhoneList = phoneNoList.stream().filter(e -> !registeredPhoneList.contains(e)).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(needRegisterPhoneList)) {
            List<User> pendRegisterUserList = Lists.newArrayList();
            for (String phone : needRegisterPhoneList) {

                //TODO 可能会存在已经注销的，又洗进来了
                String uuid = UUID.randomUUID().toString();

                User user = phoneUserMap.get(phone);
                user.setEncryptedPhoneNo(phone);
                user.setEnable(true);
                user.setTenantId(tenantId);
                user.setUuid(uuid);
                //清洗数据，忽略密码
                user.setPassword(null);
                user.setCipherPassword(null);
                pendRegisterUserList.add(user);
            }

            List<User> saveList = userRepository.save(pendRegisterUserList);
            registeredUserList.addAll(saveList);
        }

        return registeredUserList;
    }
}
