package cn.quantgroup.xyqb.service.user.impl;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserDetail;
import cn.quantgroup.xyqb.entity.UserHashMapping;
import cn.quantgroup.xyqb.event.PhoneNoUpdateEvent;
import cn.quantgroup.xyqb.model.UserInfo;
import cn.quantgroup.xyqb.repository.IUserHashMappingRepository;
import cn.quantgroup.xyqb.repository.IUserRepository;
import cn.quantgroup.xyqb.service.user.IUserDetailService;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.util.HashUtil;
import cn.quantgroup.xyqb.util.PasswordUtil;
import cn.quantgroup.xyqb.util.ValidationUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.cache.annotation.Caching;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.sql.Timestamp;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * Created by Miraculous on 15/7/5.
 */
@Service
@Slf4j
public class UserServiceImpl implements IUserService {

    @Autowired
    private RedisTemplate<String, String> stringRedisTemplate;
    @Autowired
    private IUserRepository userRepository;

    @Autowired
    private IUserHashMappingRepository userHashMappingRepository;

    @Autowired
    private IUserDetailService userDetailService;

    @Autowired
    private ApplicationEventPublisher applicationEventPublisher;


    @Override
    @Cacheable(value = "usercache", key = "'xyqbuser' + #phone", unless = "#result == null", cacheManager = "cacheManager")
    public User findByPhoneInDb(String phone) {
        return userRepository.findByPhoneNo(phone);
    }

    @Override
    public Map<Long, String> findPhoneByIdsInDb(List<Long> userIds) {
        if (CollectionUtils.isEmpty(userIds)) {
            return Maps.newHashMap();
        }
        Map<Long, String> userIdAndPhoneMap = Maps.newHashMap();
        List<User> users = userRepository.findByIdIn(userIds);
        users.forEach(user -> userIdAndPhoneMap.put(user.getId(), user.getPhoneNo()));
        return userIdAndPhoneMap;
    }

    @Override
    public User findByUuidInDb(String uuid) {
        return userRepository.findByUuid(uuid);
    }

    @Override
    public String findUuid(String phoneNo, String idNo) {
        // 优先按手机号查
        if (ValidationUtil.validatePhoneNo(phoneNo)) {
            return userRepository.findUuidByPhoneNo(phoneNo);
        }
        // 按身份证号查
        if (StringUtils.isNotBlank(idNo)) {
            return userRepository.findUuidByIdNo(idNo);
        }
        return null;
    }

    @Override
    @CacheEvict(value = "usercache", key = "'xyqbuser' + #user.phoneNo", cacheManager = "cacheManager")
    public User saveUser(User user) {
        return userRepository.saveAndFlush(user);
    }

    @Override
    public User findById(Long userId) {
        return userRepository.findById(userId);
    }

    @Override
    public List<User> findByPhones(List<String> phones) {
        return userRepository.findAll((root, query, cb) -> {
            query.where(root.get(Constants.PHONE_NO).as(String.class).in(phones));
            return query.getRestriction();
        });
    }

    @Override
    @Cacheable(value = "usercache", key = "'xyqbuser' + #phone", unless = "#result == null", cacheManager = "cacheManager")
    public User findByPhoneWithCache(String phone) {
        return userRepository.findByPhoneNo(phone);
    }

    @Override
    @Cacheable(value = "usercache", key = "'xyqbuser' + #uuid", unless = "#result == null", cacheManager = "cacheManager")
    public User findByUuidWithCache(String uuid) {
        return userRepository.findByUuid(uuid);
    }

    @Override
    public boolean exist(String phoneNo) {
        return userRepository.findByPhoneNo(phoneNo) != null;
    }

    /**
     * 修改用户密码
     *
     * @param phoneNo
     * @param password
     * @return
     * @date 2017-02-15 修改用户修改密码时，更新updatedAt时间
     */
    @Override
    @CacheEvict(value = "usercache", key = "'xyqbuser' + #phoneNo", cacheManager = "cacheManager")
    public boolean resetPassword(String phoneNo, String password) {
        User user = userRepository.findByPhoneNo(phoneNo);
        if (user == null) {
            throw new RuntimeException("用户[" + phoneNo + "]不存在");
        }
        user.setUpdatedAt(new Timestamp(System.currentTimeMillis()));
        user.setPassword(PasswordUtil.MD5WithSalt(password));
        user = userRepository.save(user);
        stringRedisTemplate.expire("usercache:xyqbuser" + phoneNo, 1L, TimeUnit.MILLISECONDS);
        return PasswordUtil.validatePassword(password, user.getPassword());
    }

    @Override
    public List<User> findRegisterUserByTime(String beginTime, String endTime) {
        return userRepository.findRegisterUserByTime(beginTime, endTime);
    }

    @Override
    @CacheEvict(value = "usercache", key = "'xyqbuser' + #phoneNo", cacheManager = "cacheManager")
    public int forbiddenUser(Boolean enable, String phoneNo) {
        return userRepository.forbiddenUser(enable, phoneNo);
    }


    @Override
    public List<UserInfo> findUserInfosByPhones(List<String> phones) {

        List<User> users = findByPhones(phones);
        if (CollectionUtils.isEmpty(phones)) {
            return Collections.emptyList();
        }

        List<UserDetail> userDetails = userDetailService.findByPhones(phones);

        if (!CollectionUtils.isEmpty(users)) {
            Map<Long, User> userMap = Maps.newHashMapWithExpectedSize(users.size());
            users.forEach(user -> {
                userMap.put(user.getId(), user);
            });

            if (!CollectionUtils.isEmpty(userDetails)) {
                List<UserInfo> userInfos = Lists.newArrayListWithExpectedSize(userDetails.size());
                userDetails.forEach(userDetail -> {
                    UserInfo userInfo = new UserInfo(userMap.get(userDetail.getUserId()), userDetail);
                    userInfos.add(userInfo);
                });
                return userInfos;
            }

        }
        return Collections.emptyList();
    }

    @Override
    public User findByMd5(Integer type, String md5Value) {
        md5Value = md5Value.toLowerCase();
        long value = HashUtil.crc32(md5Value);
        UserHashMapping userHashMapping;
        if (type == 1) {
            userHashMapping = userHashMappingRepository.findByPhoneNoMd5ShortAndPhoneNoMd5(value, md5Value);
        } else {
            userHashMapping = userHashMappingRepository.findByIdNoMd5ShortAndIdNoMd5(value, md5Value);
        }
        if (userHashMapping == null) {
            return null;
        }
        Long userId = userHashMapping.getUserId();
        return userRepository.findById(userId);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public User modifyPhoneNo(String oldPhoneNo, String newPhoneNo) {
        //1. 判断新手机号是否存在
        User newPhoneUser = userRepository.findByPhoneNo(newPhoneNo);
        if (Objects.nonNull(newPhoneUser)) {
            //新手机号已存在
            return null;
        }
        User oldPhoneUser = userRepository.findByPhoneNo(oldPhoneNo);
        if (Objects.isNull(oldPhoneUser)) {
            //这不是扯了.旧手机号不存在.
            return null;
        }
        //2. 执行修改
        //2.1 修改 user 表
        oldPhoneUser.setPhoneNo(newPhoneNo);
        User user = userRepository.saveAndFlush(oldPhoneUser);

        //2.2 修改 user_detail 表
        UserDetail oldPhoneUserDetail = userDetailService.findByPhoneNo(oldPhoneNo);
        oldPhoneUserDetail.setPhoneNo(newPhoneNo);
        userDetailService.saveUserDetail(oldPhoneUserDetail);

        //3. 发送事件
        PhoneNoUpdateEvent phoneNoUpdateEvent = new PhoneNoUpdateEvent(this, user, oldPhoneNo);
        applicationEventPublisher.publishEvent(phoneNoUpdateEvent);
        return oldPhoneUser;
    }

    @Override
    @Caching(evict = {
            @CacheEvict(value = "usercache", key = "'xyqbuser' + #phoneNo", cacheManager = "cacheManager"),
            @CacheEvict(value = "usercache", key = "'xyqbuser' + #uuid", cacheManager = "cacheManager")
    })
    public void userCacheEvict(String uuid, String phoneNo) {
        log.info("清理用户缓存成功,uuid:{},phoneNo:{}", uuid, phoneNo);
    }

}
