package cn.quantgroup.xyqb.service.wechat.impl;

import cn.quantgroup.xyqb.entity.WechatUserInfo;
import cn.quantgroup.xyqb.exception.WechatRelateUserException;
import cn.quantgroup.xyqb.model.webchat.AccessTokenResponse;
import cn.quantgroup.xyqb.repository.IWeChatUserRepository;
import cn.quantgroup.xyqb.service.http.IHttpService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.wechat.IWechatService;
import cn.quantgroup.xyqb.util.ValidationUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.cache.annotation.Caching;
import org.springframework.data.domain.Example;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * Created by Miraculous on 2017/1/19.
 */
@Slf4j
@Service
public class WechatServiceImpl implements IWechatService {
  private static final String WECHAT_TOKEN_KEY_PREFIX = "wechat:token:";
  private static final String WECHAT_USERINFO_KEY_PREFIX = "wechat:userinfo:";
  @Value("${wechat.appid}")
  private String appId;
  @Value("${wechat.secret}")
  private String secret;
  private String accessTokenUrl;
  private String refreshTokenUrl = "https://api.weixin.qq.com/sns/oauth2/refresh_token?appid=%s&grant_type=refresh_token&refresh_token=%s";
  @Resource
  private IHttpService httpService;
  @Resource
  private ISessionService sessionService;
  @Resource
  private IWeChatUserRepository weChatUserRepository;
  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> redisTemplate;

  @PostConstruct
  private void init() {
    accessTokenUrl = String.format("https://api.weixin.qq.com/sns/oauth2/access_token?appid=%s&secret=%s&grant_type=authorization_code&code=", appId, secret) + "%s";

    refreshTokenUrl = String.format("https://api.weixin.qq.com/sns/oauth2/refresh_token?appid=%s&grant_type=refresh_token&refresh_token=", appId) + "%s";
  }

  @Override
  public AccessTokenResponse getToken(String code) {
    if (StringUtils.isEmpty(code)) {
      return null;
    }
    String key = WECHAT_TOKEN_KEY_PREFIX + code;

    String resultStr = redisTemplate.opsForValue().get(key);
    if (StringUtils.isEmpty(resultStr)) {
      String response = getTokenFromWechatServer(code);
      if (StringUtils.isEmpty(response)) {
        return null;
      }
      try {
        AccessTokenResponse accessTokenResponse = JSONObject.parseObject(response,
            AccessTokenResponse.class);
        if (accessTokenResponse == null) {
          return null;
        }
        accessTokenResponse.setInitialTime(System.currentTimeMillis() - 10000);
        redisTemplate.opsForValue().set(key, JSONObject.toJSONString(accessTokenResponse), accessTokenResponse.getExpiresIn() + 1000, TimeUnit.SECONDS);
        return accessTokenResponse;
      } catch (Exception ex) {
        return null;
      }
    } else {
      try {
        AccessTokenResponse response = JSONObject.parseObject(resultStr, AccessTokenResponse.class);
        // 刷新
        if (response.getInitialTime() + response.getExpiresIn() * 1000 > System.currentTimeMillis()) {
          String refreshTokenStr = refreshToken(response.getRefreshToken());
          response = JSONObject.parseObject(refreshTokenStr, AccessTokenResponse.class);
          if (response == null) {
            return null;
          }
          response.setInitialTime(System.currentTimeMillis() - 10000);
        }
        redisTemplate.opsForValue().set(key, JSONObject.toJSONString(response), response.getExpiresIn() + 1000, TimeUnit.SECONDS);
        return response;
      } catch (Exception ex) {
        return null;
      }
    }
  }

  @Override
  public WechatUserInfo getWechatUserInfoFromWechatServer(String token, String openId) {
    if (StringUtils.isEmpty(token) || StringUtils.isEmpty(openId)) {
      return null;
    }
    String accessUserInfoUrl = "https://api.weixin.qq.com/sns/userinfo?access_token=%s&openid=%s&lang=zh_CN";
    String url = String.format(accessUserInfoUrl, token, openId);
    String result = httpService.get(url);
    return JSONObject.parseObject(result, WechatUserInfo.class);
  }

  @Override
  @Cacheable(value = "WechatUserInfo", key = "'openId:' + #openId", unless = "#result == null", cacheManager = "cacheManager")
  public WechatUserInfo findWechatUserInfoFromDb(String openId) {
    return weChatUserRepository.findByOpenId(openId);
  }

  @Override
  @CacheEvict(value = "WechatUserInfo", key = "'openId:' + #userInfo.openId", cacheManager = "cacheManager")
  @Transactional
  public WechatUserInfo saveWechatUserInfo(WechatUserInfo userInfo) {
    if(Objects.isNull(userInfo) || Objects.isNull(userInfo.getOpenId())){
      return null;
    }
    long count = weChatUserRepository.countByOpenId(userInfo.getOpenId());
    if(count > 0){
      //注意，这里会抛异常（5000/total）,WeChatController中已捕获处理
      return weChatUserRepository.findByOpenId(userInfo.getOpenId());
    }
    if (null == userInfo.getPhoneNo()) {
      userInfo.setPhoneNo("");
    }
    return weChatUserRepository.save(userInfo);
  }

  @Override
  @CacheEvict(value = "WechatUserInfo", key = "'openId:' + #openId", cacheManager = "cacheManager")
  @Transactional(rollbackFor = Exception.class)
  public int relateUser(Long userId, String phoneNo, String openId) {
    if(Objects.isNull(userId) || Objects.isNull(openId) || StringUtils.isBlank(openId)){
      log.error("微信关联失败：非法入参：[service]:userId:{},phoneNo:{},openId:{}", userId, phoneNo, openId);
      return 0;
    }
    /* 如果当前openId已关联其他用户，则解绑成功后要注销其登录session */
    WechatUserInfo wechatUserInfo = weChatUserRepository.findByUserId(userId);
    // 强制解除关联
    int dissociate = weChatUserRepository.dissociateUser(openId, userId);
    if(dissociate < 1){
      log.error("微信关联失败：解绑条数<1：[service]:userId:{},phoneNo:{},openId:{}", userId, phoneNo, openId);
      throw new WechatRelateUserException("微信关联失败");
    }
    int relate = weChatUserRepository.relateUser(userId, Optional.ofNullable(phoneNo).orElse(""), openId);
    if(relate < 1){
      log.error("微信关联失败：绑定条数<1：[service]:userId:{},phoneNo:{},openId:{}", userId, phoneNo, openId);
      throw new WechatRelateUserException("微信关联失败");
    }else if(Objects.nonNull(wechatUserInfo) && Objects.nonNull(wechatUserInfo.getUserId())){
      // 如果当前openId已关联其他用户，则解绑成功后要注销其登录session
      sessionService.deleteByUserId(wechatUserInfo.getUserId());
    }
    log.info("微信关联成功：[service]:userId:{},phoneNo:{},openId:{},dissociate:{},relate:{}", userId, phoneNo, openId,dissociate,relate);
    return relate;
  }

  @Override
  public WechatUserInfo findWechatUserInfoByPhoneNo(String phoneNo) {
    if(!ValidationUtil.validatePhoneNo(phoneNo)){
      return null;
    }
    return weChatUserRepository.findByPhoneNo(phoneNo);
  }

  private String getTokenFromWechatServer(String code) {
    if (StringUtils.isEmpty(code)) {
      return null;
    }
    String finalAccessTokenUrl = String.format(accessTokenUrl, code);
    return httpService.get(finalAccessTokenUrl);
  }

  private String refreshToken(String refreshToken) {
    if (StringUtils.isEmpty(refreshToken)) {
      return null;
    }
    return httpService.get(String.format(refreshTokenUrl, refreshToken));
  }

  /**
   * 按userId查微信信息
   * 此处不应加缓存，否则解绑时不好清除
   * @param userId - 用户标识
   * @return
   */
  @Override
  public WechatUserInfo queryByUserId(Long userId) {
    return weChatUserRepository.findByUserId(userId);
  }


  @Override
  public int forbiddenUserWeChat(Long userId) {
    return weChatUserRepository.removeByUserId(userId);
  }
}
