package cn.quantgroup.xyqb.service.sms.impl;

import cn.quantgroup.sms.SendAndForgetMsg;
import cn.quantgroup.sms.SmsSender;
import cn.quantgroup.xyqb.model.sms.SmsConfig;
import cn.quantgroup.xyqb.model.sms.SmsResult;
import cn.quantgroup.xyqb.service.sms.ISmsSender;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;


import java.util.Collections;

import static cn.quantgroup.xyqb.model.sms.SmsConfig.SMS_INDEX_KEY;

/**
 * @author mengfan.feng
 * @time 2015-07-25 18:47
 */
@Service
public class SmsServiceImpl implements ISmsService {

    private static final Logger LOGGER = LoggerFactory.getLogger(SmsServiceImpl.class);

    @Value("${sms.is.debug}")
    private boolean isDebug;

    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> redisTemplate;

    @Autowired
    private GuoDuSms guoDuSms;

    @Autowired
    private ChuangLanSms chuangLanSms;

    @Autowired
    private ZhangZhongSms zhangZhongSms;

    @Autowired
    private MengWangSms mengWangSms;

    @Autowired
    private ManDaoSms manDaoSms;

    private SmsSender smsSender;

    private static final ISmsSender[] SMS_SENDERS = new ISmsSender[5];

    @PostConstruct
    private void init() {
        smsSender = new SmsSender();
        SMS_SENDERS[0] = guoDuSms;
        SMS_SENDERS[1] = chuangLanSms;
        SMS_SENDERS[2] = zhangZhongSms;
        SMS_SENDERS[3] = mengWangSms;
        SMS_SENDERS[4] = manDaoSms;

        if (isDebug) {
            for (SmsConfig.SMS sms : SmsConfig.SMS.values()) {
                sms.setEnabled(false);
            }
            redisTemplate.opsForValue().set(SMS_INDEX_KEY, "-1");
            return;
        }

        if (StringUtils.isEmpty(redisTemplate.opsForValue().get(SMS_INDEX_KEY))) {
            redisTemplate.opsForValue().set(SMS_INDEX_KEY, "0");
        }
    }

    @Override
    public SmsSender getSmsSender() {
        return this.smsSender;
    }

    @Override
    public SmsResult send(String phone, String content) {
        if (StringUtils.isBlank(phone) || StringUtils.isBlank(content)) {
            return SmsResult.error("参数错误");
        }

        /********** Debug模式 ***********/
        int smsIndex = Integer.parseInt(redisTemplate.opsForValue().get(SMS_INDEX_KEY));
        if (isDebug && smsIndex < 0) {
            return SmsResult.success();
        }

        SmsResult result = SmsResult.error("发送失败");

        /********** 如果发送失败，选择其他短信平台重新发送 ***********/
        for (int i = 0; i < SMS_SENDERS.length; i++) {
            if (smsIndex >= SMS_SENDERS.length) {
                smsIndex = 0;
            }

            try {
                if (!SmsConfig.SMS.values()[smsIndex].isEnabled()) {
                    LOGGER.info("短信平台已关闭, smsIndex:{}", smsIndex);
                    smsIndex++;
                    continue;
                }
                result = SMS_SENDERS[smsIndex].send(phone, content);
            } catch (Exception e) {
                LOGGER.error("短信发送出现异常", e);
                result.setMsg("系统异常");
            }

            if (result.isRet()) {
                LOGGER.info("短信发送成功, smsIndex:{}, phone:{}, content:{}", smsIndex, phone, content);
                break;
            } else {
                LOGGER.warn("短信发送失败, smsIndex:{}, phone:{}, content:{}, reason:{}", smsIndex, phone, content, result.getMsg());
                smsIndex++;
            }
        }

        redisTemplate.opsForValue().set(SMS_INDEX_KEY, smsIndex + "");

        return result;
    }

    @Override
    public void sendAfterRegister(String phoneNo) {
        try {
            smsSender.sendAndForget(new SendAndForgetMsg(Collections.emptyList(), "24", "1", phoneNo));
            LOGGER.info("注册完成，发送短信, phoneNo:{}", phoneNo);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
