package cn.quantgroup.xyqb.aspect.captcha;


import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.service.config.IConfigurationService;
import cn.quantgroup.xyqb.thirdparty.jcaptcha.AbstractManageableImageCaptchaService;
import com.octo.captcha.service.CaptchaServiceException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.nio.charset.Charset;
import java.util.Date;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

/**
 * 类名称：CaptchaValidateAdvisor
 * 类描述：
 *
 * @author 李宁
 * @version 1.0.0
 *          创建时间：15/11/17 14:49
 *          修改人：
 *          修改时间：15/11/17 14:49
 *          修改备注：
 */
@Aspect
@Component
public class CaptchaValidateAdvisor {

    private static final Logger LOGGER = LoggerFactory.getLogger(CaptchaValidateAdvisor.class);
    private static final String SUPER_CAPTCHA_ID = UUID.nameUUIDFromBytes("__QG_APPCLIENT_AGENT__".getBytes(Charset.forName("UTF-8"))).toString();
    private static final String SUPER_CAPTCHA = "__SUPERQG__";

    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> redisTemplate;

    @Autowired
    @Qualifier("customCaptchaService")
    private AbstractManageableImageCaptchaService imageCaptchaService;

    @Autowired
    private IConfigurationService configurationService;

    /**
     * 自动化测试忽略验证码
     */
    @Value("${xyqb.auth.captcha.autotest.enable:false}")
    private boolean autoTestCaptchaEnabled;

    /**
     * 图形验证码切面
     */
    @Pointcut("@annotation(cn.quantgroup.xyqb.aspect.captcha.CaptchaValidator)")
    private void needCaptchaValidate() {
    }

    /**
     * 客户端 Ip限制切面
     */
    @Pointcut("@annotation(cn.quantgroup.xyqb.aspect.captcha.CaptchaRestriction)")
    private void checkCaptchaRestriction() {
    }

    /**
     * 根据 Ip 限制客户端 每分钟获取的图形二维码数量
     *
     * @param proceedingJoinPoint
     * @return
     * @throws Throwable
     */
    @Around("checkCaptchaRestriction()")
    private Object preventCaptchaRequest(ProceedingJoinPoint proceedingJoinPoint) throws Throwable {
        Object[] args = proceedingJoinPoint.getArgs();
        String remoteIp = getClientIp(args);

        if (remoteIp == null || clientIpInSafeList(remoteIp)) {
            return proceedingJoinPoint.proceed();
        }

        String limitKey = "ipcaptcha:" + DateFormatUtils.format(new Date(), "yyyyMMddHHmm") + StringUtils.remove(StringUtils.remove(remoteIp, '.'), ':');
        Long count = redisTemplate.opsForValue().increment(limitKey, 1);
        // 1, 2, 3... <= 50
        if (count == 1) {
            redisTemplate.expire(limitKey, 1, TimeUnit.MINUTES);
        }

        Long captchaPerIpPerMin = getCaptchaPerIpPerMin();
        if (count <= captchaPerIpPerMin) {
            try {
                Object proceed = proceedingJoinPoint.proceed();
                if (proceed != null && proceed.getClass() == JsonResult.class) {
                    JsonResult jsonResult = (JsonResult) proceed;
                    String code = jsonResult.getCode();
                    // 如果生成验证码失败, 不计数
                    if (!"0000".equals(code)) {
                        redisTemplate.opsForValue().increment(limitKey, -1L);
                    }
                }

                return proceed;
            } catch (Exception e) {
                // 发生异常, 不计数
                redisTemplate.opsForValue().increment(limitKey, -1L);
                throw e;
            }
        }

        LOGGER.error("获取验证码失败, 客户端[{}] 每分钟请求验证码超限制 : [{}]", remoteIp, captchaPerIpPerMin);
        return JsonResult.buildErrorStateResult("您获取验证码较频繁，请稍候再试!", null);
    }

    private boolean clientIpInSafeList(String remoteIp) {
        String value = configurationService.getValue(Constants.CONFIG_CAPTCHA_WHITEIP_LIST);
        return !StringUtils.isBlank(value) && value.contains(remoteIp);
    }

    private String getClientIp(Object[] args) {
        if (args.length < 2) {
            return null;
        }
        String ip = String.valueOf(args[1]);
        return StringUtils.isNotBlank(ip) ? ip : null;
    }

    /**
     * 在受图形验证码保护的接口方法执行前, 执行图形验证码校验
     *
     * @param pjp
     * @return
     * @throws Throwable
     */
    @Around("needCaptchaValidate()")
    private Object doCapchaValidate(ProceedingJoinPoint pjp) throws Throwable {
        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        String registerFrom = Optional.ofNullable(request.getParameter("registerFrom")).orElse("");
        String captchaId = Optional.ofNullable(request.getParameter("captchaId")).orElse("");
        Object captchaValue = request.getParameter("captchaValue");

        if (shouldSkipCaptchaValidate(registerFrom, captchaId, captchaValue)) {
            LOGGER.info("使用超级图形验证码校验, registerFrom={}, clientIp={}", registerFrom, request.getRemoteAddr());
            return pjp.proceed();
        }

        JsonResult result = JsonResult.buildSuccessResult("图形验证码错误, 请重新输入", "");
        result.setBusinessCode("0002");
        if (captchaValue != null) {
            String captcha = String.valueOf(captchaValue);
            // 忽略用户输入的大小写
            captcha = StringUtils.lowerCase(captcha);
            // 验证码校验
            Boolean validCaptcha = false;
            try {
                validCaptcha = imageCaptchaService.validateResponseForID(Constants.IMAGE_CAPTCHA_KEY + captchaId, captcha);
            } catch (CaptchaServiceException ex) {
                LOGGER.error("验证码校验异常, {}, {}", ex.getMessage(), ex);
            }

            if (validCaptcha) {
                return pjp.proceed();
            }
        }

        return result;
    }

    private boolean shouldSkipCaptchaValidate(String registerFrom, String captchaId, Object captchaValue) {
        boolean superCaptchaEnabled = Boolean.valueOf(configurationService.getValue(Constants.CONFIG_CAPTCHA_MAGIC_CODE_ENABLED));

        // 如果启用了超级验证码功能, 检查超级验证码, 超级验证码区分大小写
        if (superCaptchaEnabled && autoTestCaptchaEnabled) {
            return true;
        }

        return superCaptchaEnabled
                && StringUtils.equals(SUPER_CAPTCHA_ID, String.valueOf(captchaId))
                && StringUtils.equals(SUPER_CAPTCHA, String.valueOf(captchaValue));
    }

    private Long getCaptchaPerIpPerMin() {
        return Long.valueOf(configurationService.getValue(Constants.CONFIG_CAPTCHA_PERIP_PERMIN));
    }

}
