package cn.quantgroup.xyqb.controller.external.captcha;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.captcha.CaptchaValidator;
import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.thirdparty.jcaptcha.AbstractManageableImageCaptchaService;
import com.octo.captcha.service.CaptchaServiceException;
import java.nio.charset.Charset;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.imageio.ImageIO;
import javax.servlet.http.HttpServletRequest;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import sun.management.counter.LongCounter;

/**
 * 类名称：ImgCaptchaController
 * 类描述：图形验证码控制器
 *
 * @author 李宁
 * @version 1.0.0 创建时间：15/11/17 11:49 修改人： 修改时间：15/11/17 11:49 修改备注：
 */
@RestController
@RequestMapping("/api")
public class ImageCaptchaController implements IBaseController {

  private static final Logger LOGGER = LoggerFactory.getLogger(ImageCaptchaController.class);
  private static final String SUPER_CAPTCHA_ID = UUID.nameUUIDFromBytes("__QG_APPCLIENT_AGENT__".getBytes(Charset.forName("UTF-8"))).toString();
  private static final String SUPER_CAPTCHA = "__SUPERQG__";

  private static final String IMAGE_FORMAT_PNG = "png";
  private static final String IMG_BASE64_PATTREN = "data:image/" + IMAGE_FORMAT_PNG + ";base64,%s";
  private static final String IMAGE_IP_COUNT = "image:ip";
  private static final Long  FIVE_MIN = 24 * 5L;
  @Autowired
  @Qualifier("customCaptchaService")
  private AbstractManageableImageCaptchaService imageCaptchaService;
  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> redisTemplate;

  @ModelAttribute("clientIp")
  public String initClientIp() {
    return getIp();
  }

  /**
   * 自动化测试忽略验证码
   */
  @Value("${xyqb.auth.captcha.autotest.enable:false}")
  private boolean autoTestCaptchaEnabled;

  /**
   * 获取验证码
   * 默认匹配 GET /captcha, 提供4位数字和字母混合图片验证码
   */
  @RequestMapping(value = "/captcha")
  public JsonResult fetchCaptcha(HttpServletRequest request, @ModelAttribute("clientIp") String clientIp) {
    String imageId = UUID.randomUUID().toString();
    BufferedImage challenge = imageCaptchaService.getImageChallengeForID(Constants.IMAGE_CAPTCHA_KEY + imageId, request.getLocale());
    ByteArrayOutputStream jpegOutputStream = new ByteArrayOutputStream();
    try {
      boolean write = ImageIO.write(challenge, IMAGE_FORMAT_PNG, jpegOutputStream);
    } catch (IOException e) {
      e.printStackTrace();
      return JsonResult.buildErrorStateResult("", "fail");
    }

    String imageBase64 = Base64.encodeBase64String(jpegOutputStream.toByteArray());

    Map<String, String> data = new HashMap<>();
    data.put("imageId", imageId);
    data.put("image", String.format(IMG_BASE64_PATTREN, imageBase64));
    return JsonResult.buildSuccessResult("", data);
  }

  /**
   * 新版本获取验证码
   * 默认匹配 GET /captcha, 提供4位数字和字母混合图片验证码
   */
  @RequestMapping(value = "/captcha_new")
  public JsonResult fetchCaptchaNew(HttpServletRequest request, @ModelAttribute("clientIp") String clientIp) {
    if (StringUtils.isBlank(clientIp)) {
      return JsonResult.buildErrorStateResult("", "fail");
    }
   Long count= countIP(clientIp);
    if(count>5){
      String imageId = UUID.randomUUID().toString();
      BufferedImage challenge = imageCaptchaService.getImageChallengeForID(Constants.IMAGE_CAPTCHA_KEY + imageId, request.getLocale());
      ByteArrayOutputStream jpegOutputStream = new ByteArrayOutputStream();
      try {
        boolean write = ImageIO.write(challenge, IMAGE_FORMAT_PNG, jpegOutputStream);
      } catch (IOException e) {
        e.printStackTrace();
        return JsonResult.buildErrorStateResult("", "fail");
      }

      String imageBase64 = Base64.encodeBase64String(jpegOutputStream.toByteArray());

      Map<String, String> data = new HashMap<>();
      data.put("imageId", imageId);
      data.put("image", String.format(IMG_BASE64_PATTREN, imageBase64));
      return JsonResult.buildSuccessResult("", data);
    }


    return JsonResult.buildSuccessResult("", "success");
  }

  private Long countIP(@ModelAttribute("clientIp") String clientIp) {
    Long count=1L;
    String countString=redisTemplate.opsForValue().get(IMAGE_IP_COUNT+ clientIp);
    if(StringUtils.isBlank(countString)){
      redisTemplate.opsForValue().set(IMAGE_IP_COUNT+ clientIp, String.valueOf(count),
          FIVE_MIN, TimeUnit.SECONDS);
    }else{
      count= Long.valueOf(countString)+1L;
      redisTemplate.opsForValue().set(IMAGE_IP_COUNT+ clientIp, String.valueOf(count),
          FIVE_MIN, TimeUnit.SECONDS);
    }
  }

  /**
   * 图片验证码验证
   */
  @CaptchaValidator
  @RequestMapping("/verification_image_code")
  public JsonResult verificationImageCode(HttpServletRequest request, @RequestParam String captchaId, @RequestParam Object captchaValue) {
    Boolean validCaptcha = false;
    String registerFrom = Optional.ofNullable(request.getParameter("registerFrom")).orElse("");
    if (shouldSkipCaptchaValidate(registerFrom, captchaId, captchaValue)) {
      LOGGER.info("使用超级图形验证码校验, registerFrom={}, clientIp={}", registerFrom, request.getRemoteAddr());
      validCaptcha = true;
      return JsonResult.buildSuccessResult("", validCaptcha);
    }

    JsonResult result = JsonResult.buildSuccessResult("图形验证码错误, 请重新输入", "");
    result.setBusinessCode("0002");
    if (captchaValue != null) {
      String captcha = String.valueOf(captchaValue);
      // 忽略用户输入的大小写
      captcha = StringUtils.lowerCase(captcha);
      // 验证码校验

      try {
        validCaptcha = imageCaptchaService.validateResponseForID(Constants.IMAGE_CAPTCHA_KEY + captchaId, captcha);
      } catch (CaptchaServiceException ex) {
        LOGGER.error("验证码校验异常, {}, {}", ex.getMessage(), ex);
      }
    }
    return JsonResult.buildSuccessResult("", validCaptcha);
  }

  private boolean shouldSkipCaptchaValidate(String registerFrom, String captchaId, Object captchaValue) {

    // 如果启用了超级验证码功能, 检查超级验证码, 超级验证码区分大小写
    if (autoTestCaptchaEnabled) {
      return true;
    }

    return StringUtils.equals(SUPER_CAPTCHA_ID, String.valueOf(captchaId)) && StringUtils.equals(SUPER_CAPTCHA, String.valueOf(captchaValue));
  }
}
