package cn.quantgroup.xyqb.controller.internal.user;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.captcha.CaptchaFineteValidator;
import cn.quantgroup.xyqb.aspect.captcha.CaptchaValidator;
import cn.quantgroup.xyqb.aspect.logcaller.LogHttpCaller;
import cn.quantgroup.xyqb.aspect.token.OneTimeTokenValidator;
import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.entity.Merchant;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserDetail;
import cn.quantgroup.xyqb.entity.WechatUserInfo;
import cn.quantgroup.xyqb.exception.PasswordErrorLimitException;
import cn.quantgroup.xyqb.exception.UserNotExistException;
import cn.quantgroup.xyqb.exception.VerificationCodeErrorException;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.UserModel;
import cn.quantgroup.xyqb.model.UserRegisterMqMessage;
import cn.quantgroup.xyqb.model.UserStatistics;
import cn.quantgroup.xyqb.service.merchant.IMerchantService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.service.user.IUserDetailService;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.service.wechat.IWechatService;
import cn.quantgroup.xyqb.util.MqUtils;
import cn.quantgroup.xyqb.util.PasswordUtil;
import cn.quantgroup.xyqb.util.ValidationUtil;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Random;
import java.util.concurrent.TimeUnit;

/**
 * Http服务接口：用户注册、登录、重置密码
 */
@RestController
@RequestMapping("/user")
public class UserController implements IBaseController {


  private static final Logger LOGGER = LoggerFactory.getLogger(UserController.class);

  private final String pwdSalt = "_lkb";

  @Autowired
  private IUserService userService;

  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> stringRedisTemplate;

  @Autowired
  private ISmsService smsService;

  @Autowired
  private ISessionService sessionService;

  @Autowired
  private IUserDetailService userDetailService;
  @Autowired
  private IMerchantService merchantService;
  @Autowired
  private IWechatService wechatService;

  private static final char[] PWD_BASE = {'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k',
          'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w',
          'x', 'y', 'z', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9'};


  /**
   * 登录（账号 + 密码）
   * 密码错误达到限定次数时执行图形验证码校验
   * 图形验证码累计错误达到限定次数时须重新获取
   *
   * @param channelId
   * @param appChannel
   * @param createdFrom
   * @param userId
   * @param key
   * @param request
   * @param openId
   * @param dimension
   * @return
   */
  @CaptchaFineteValidator
  @RequestMapping("/loginV1")
  public JsonResult loginV1(
          @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
          @RequestParam(required = false, defaultValue = "1") Long createdFrom,
          @RequestParam(required = false, defaultValue = "") String userId,
          @RequestParam(required = false,defaultValue = "xyqb") String key,
          HttpServletRequest request, String openId,
          @RequestParam(required = false) String dimension) {
    LOGGER.info("loginV1 -> channelId:{},appChennel:{},createdFrom:{},userId:{},key:{},openId:{},dimension:{}",channelId, appChannel, createdFrom, userId, key, openId, dimension);
    return login(channelId, appChannel, createdFrom, userId, key, request, openId, dimension);
  }

  /**
   * 快速登录（手机号 + 短信验证码），H5专用入口
   * 短信验证码错误达到限定次数时执行图形验证码校验
   * 图形验证码累计错误达到限定次数时须重新获取
   *
   * @param channelId
   * @param appChannel
   * @param createdFrom
   * @param key
   * @param btRegisterChannelId
   * @param dimension
   * @param request
   * @return
   */
  @OneTimeTokenValidator
  @RequestMapping("/login/fastV1")
  public JsonResult loginFastV1(
          @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
          @RequestParam(required = false, defaultValue = "1") Long createdFrom,
          @RequestParam(required = false,defaultValue = "xyqb") String key,
          @RequestParam(required = false)Long btRegisterChannelId,
          @RequestParam(required = false) String dimension ,HttpServletRequest request) {
    LOGGER.info("login/fastV1 -> channelId:{},appChennel:{},createdFrom:{},key:{},btRegisterChannelId:{},dimension:{}",channelId, appChannel, createdFrom, key, btRegisterChannelId, dimension);
    return loginFast(channelId, appChannel, createdFrom, key, btRegisterChannelId, dimension, request);
  }

  @LogHttpCaller
  @RequestMapping("/login")
  public JsonResult login(
          @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
          @RequestParam(required = false, defaultValue = "1") Long createdFrom,
          @RequestParam(required = false, defaultValue = "") String userId,
          @RequestParam(required = false,defaultValue = "xyqb") String key,
          HttpServletRequest request, String openId,
          @RequestParam(required = false) String dimension) {

    Merchant merchant = merchantService.findMerchantByName(key);
    if (merchant == null) {
      return JsonResult.buildErrorStateResult("未知的连接", null);
    }
    if (!StringUtils.isEmpty(userId) && userId.length() > 10) {

      return loginWithUserId(channelId, appChannel, createdFrom, userId, merchant,dimension);
    } else {
      return loginWithHttpBasic(channelId, appChannel, createdFrom, merchant, request, openId,dimension);
    }
  }
  //@LogHttpCaller
  //@RequestMapping("/login2")
  //public JsonResult login2(
  //    @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
  //    @RequestParam(required = false, defaultValue = "1") Long createdFrom,
  //    @RequestParam(required = false, defaultValue = "") String userId, HttpServletRequest request,  @RequestParam(required = false) String openId,@RequestParam(required = false) String dimension) {
  //
  //  if (!StringUtils.isEmpty(userId) && userId.length() > 10) {
  //
  //    return loginWithUserId(channelId, appChannel, createdFrom, userId, null, dimension);
  //  } else {
  //    return loginWithHttpBasic(channelId, appChannel, createdFrom, null, request, null,dimension);
  //  }
  //}

  @RequestMapping("/test")
  public JsonResult test() {
    return JsonResult.buildSuccessResult("", getCurrentUserFromRedis());
  }

  @RequestMapping("/login/fast")
  @LogHttpCaller
  public JsonResult loginFast(
          @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
          @RequestParam(required = false, defaultValue = "1") Long createdFrom,
          @RequestParam(required = false,defaultValue = "xyqb") String key,
          @RequestParam(required = false)Long btRegisterChannelId,
          @RequestParam(required = false) String dimension ,HttpServletRequest request) {
    Map<String, JsonResult> validMap = getHeaderParam(request);
    if (null != validMap.get("fail")) {
      return validMap.get("fail");
    }
    Merchant merchant = merchantService.findMerchantByName(key);
    if (merchant == null) {
      return JsonResult.buildErrorStateResult("未知的连接", null);
    }
    JsonResult successResult = validMap.get("success");
    String phoneNo = successResult.getData().toString();
    String verificationCode =  successResult.getMsg();
    // 执行短信验证码检查
    smsValidForFastLogin(phoneNo, verificationCode);
    User user = userService.findByPhoneWithCache(phoneNo);
    if (user != null && !user.getEnable()) {
      return JsonResult.buildErrorStateResult("登录失败", null);
    }
    if (user == null) {
      user = registerFastWhenLogin(phoneNo, channelId, createdFrom, appChannel,btRegisterChannelId,dimension);
      if (user == null) {
        throw new UserNotExistException("用户未找到");
      }
    }
    //增加登陆统计发送
    UserStatistics statistics=new UserStatistics(user,dimension,3,channelId);
    MqUtils.sendLoanVest(statistics);
    return new JsonResult(sessionService.createSession(channelId, createdFrom, appChannel, user, merchant));
    // return createSession(channelId, createdFrom, appChannel, user);
  }

  private User registerFastWhenLogin(String phoneNo, Long channelId, Long registerFrom, String appChannel, Long btRegisterChannelId,String dimension) {
    String password = genRandomPwd();
    LOGGER.info("用户快速注册, phoneNo:{}, channelId:{}, registerFrom:{},appChannel:{},btRegisterChannelId", phoneNo, channelId, registerFrom, appChannel,btRegisterChannelId);
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      LOGGER.info("用户快速注册失败，手机号错误, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      throw new UserNotExistException("手机号错误");
    }
    if (null == registerFrom) {
      registerFrom = 1L;
    }
    if (channelId == 222L) {
      registerFrom=222L;
    }
    User user=userService.registerAndReturn(phoneNo, password, registerFrom,btRegisterChannelId);
    LOGGER.info("用户快速注册成功, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
    UserStatistics statistics=new UserStatistics(user,dimension,2,channelId);
    MqUtils.sendLoanVest(statistics);
    //增加用户注册广播
    UserRegisterMqMessage registerMqMessage=new UserRegisterMqMessage(user);
    MqUtils.sendRegisterMessage(registerMqMessage);
    return user;
  }

  /**
   * 快速登录验证
   *
   * @param request
   * @return
   */
  private Map<String, JsonResult> getHeaderParam(HttpServletRequest request) {
    Map<String, JsonResult> result = new HashMap<String, JsonResult>();
    String verificationHeader = "Verification ";
    String credential = request.getHeader("authorization");
    if (StringUtils.isBlank(credential)) {
      result.put("fail", JsonResult.buildErrorStateResult("登录失败", null));
      return result;
    }
    if (!credential.startsWith(verificationHeader)) {
      result.put("fail", JsonResult.buildErrorStateResult("登录失败", null));
      return result;
    }
    credential = credential.substring(verificationHeader.length(), credential.length());
    byte[] buf = Base64.decodeBase64(credential);
    boolean validCharset = true;
    try {
      credential = new String(buf, "UTF-8");
    } catch (UnsupportedEncodingException e) {
      LOGGER.error("不支持的编码.");
      result.put("fail", JsonResult.buildErrorStateResult("登录失败", null));
    }
    if (!validCharset) {
      result.put("fail", JsonResult.buildErrorStateResult("登录失败", null));
      return result;
    }
    String[] credentialArr = credential.split(":");
    if (credentialArr.length != 2) {
      result.put("fail", JsonResult.buildErrorStateResult("登录失败", null));
      return result;
    }
    String phoneNo = credentialArr[0];
    String verificationCode = credentialArr[1];
    LOGGER.info("用户快速登录,phoneNo:{} , verificationCode:{}", phoneNo, verificationCode);
    if (!ValidationUtil.validatePhoneNo(phoneNo) || StringUtils.isBlank(verificationCode)) {
      result.put("fail", JsonResult.buildErrorStateResult("登录失败", null));
      return result;
    }
    result.put("success", JsonResult.buildSuccessResult(verificationCode, phoneNo));
    return result;
  }

  /**
   * 用户快速注册
   *
   * @param phoneNo
   * @param verificationCode
   * @param channelId
   * @return
   */
  @LogHttpCaller
  @RequestMapping("/register/fast")
  public JsonResult registerFast(@RequestParam String phoneNo, @RequestParam String verificationCode, @RequestParam(required = false) Long channelId,
                                 @RequestParam(required = false) Long registerFrom, @RequestParam(required = false, defaultValue = "") String appChannel,
                                 @RequestParam(required = false)Long btRegisterChannelId,@RequestParam(required = false)String dimension) {
    String password = genRandomPwd();
    LOGGER.info("用户快速注册, phoneNo:{}, verificationCode:{}, channelId:{}, registerFrom:{},appChannel:{}", phoneNo, verificationCode, channelId, registerFrom, appChannel);
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      LOGGER.info("用户快速注册失败，手机号错误, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("手机号错误", null);
    }
    if (StringUtils.isEmpty(password)) {
      LOGGER.info("用户快速注册失败，密码不能为空, registerFrom:{}, phoneNo:{}, password:{}", registerFrom, phoneNo, password);
      return JsonResult.buildErrorStateResult("密码不能为空", null);
    }
    if (password.length() < 6 || password.length() > 12) {
      LOGGER.info("用户快速注册失败，密码长度须在6位至12位之间, registerFrom:{}, phoneNo:{}, password:{}", registerFrom, phoneNo, password);
      return JsonResult.buildErrorStateResult("密码应为6-12位", null);
    }
    if (null == registerFrom) {
      registerFrom = 1L;
    }
    smsValidForFastLogin(phoneNo, verificationCode);
    if (userService.exist(phoneNo)) {
      LOGGER.info("用户快速注册失败，该手机号已经被注册, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("该手机号已经被注册", null);
    }
    if (!userService.register(phoneNo, password, registerFrom, getIp(), channelId,btRegisterChannelId,dimension)) {
      LOGGER.info("用户快速注册失败，请稍后重试, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("注册失败，请稍后重试", null);
    }

    LOGGER.info("用户快速注册成功, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);

    return JsonResult.buildSuccessResult(null, null);
  }


  /**
   * 用户注册
   *
   * @param phoneNo
   * @param password
   * @param verificationCode
   * @param channelId
   * @return
   */
  @LogHttpCaller
  @RequestMapping("/register")
  public JsonResult register(@RequestParam String phoneNo, @RequestParam String password,
                             @RequestParam String verificationCode, @RequestParam(required = false) Long channelId,
                             @RequestParam(required = false) Long registerFrom,
                             @RequestParam(required = false)Long btRegisterChannelId,@RequestParam(required = false)String dimension) {
    LOGGER.info("用户注册, phoneNo:{}, verificationCode:{}, channelId:{}, registerFrom:{},btRegisterChannelId:{}", phoneNo, verificationCode, channelId, registerFrom,btRegisterChannelId);
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      LOGGER.info("用户注册失败，手机号错误, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("手机号错误", null);
    }
    if (StringUtils.isEmpty(password)) {
      LOGGER.info("用户注册失败，密码不能为空, registerFrom:{}, phoneNo:{}, password:{}", registerFrom, phoneNo, password);
      return JsonResult.buildErrorStateResult("密码不能为空", null);
    }
    if (password.length() < 6 || password.length() > 12) {
      LOGGER.info("用户注册失败，密码长度须在6位至12位之间, registerFrom:{}, phoneNo:{}, password:{}", registerFrom, phoneNo, password);
      return JsonResult.buildErrorStateResult("密码应为6-12位", null);
    }
    if (null == registerFrom) {
      registerFrom = 1L;
    }
    smsValidForRegister(phoneNo, verificationCode);
    if (userService.exist(phoneNo)) {
      LOGGER.info("用户注册失败，该手机号已经被注册, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("该手机号已经被注册", null);
    }
    if (!userService.register(phoneNo, password, registerFrom, getIp(), channelId,btRegisterChannelId,dimension)) {
      LOGGER.info("用户快速注册失败，请稍后重试, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("注册失败，请稍后重试", null);
    }

    LOGGER.info("用户注册成功, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);

    return JsonResult.buildSuccessResult(null, null);
  }

  /**
   * 检查用户是否存在
   *
   * @param phoneNo 手机号
   * @return
   */
  @RequestMapping("/exist")
  public JsonResult exist(@RequestParam String phoneNo) {
    LOGGER.info("检查用户是否存在, phoneNo:{}", phoneNo);
    if (userService.exist(phoneNo)) {
      LOGGER.info("该手机号已经注册, phoneNo:{}", phoneNo);
      return JsonResult.buildErrorStateResult("该手机号已经注册", null);
    }
    return JsonResult.buildSuccessResult(null, null);
  }

  /**
   * 检查用户是否存在
   *
   * @param phoneNo 手机号
   * @return
   */
  @RequestMapping("/exist_check")
  public JsonResult existForResetPwd(@RequestParam String phoneNo) {
    LOGGER.info("检查用户是否存在, phoneNo:{}", phoneNo);
    return JsonResult.buildSuccessResult(null, userService.exist(phoneNo));
  }

  /**
   * 重置密码
   *
   * @param phoneNo
   * @param password
   * @param verificationCode
   * @return
   */
  @LogHttpCaller
  @RequestMapping("/reset_password")
  public JsonResult resetPassword(@RequestParam String phoneNo,
                                  @RequestParam String password,
                                  @RequestParam(required = false) String registerFrom,
                                  @RequestParam String verificationCode) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return JsonResult.buildErrorStateResult("手机号错误", null);
    }
    if (StringUtils.isEmpty(password)) {
      return JsonResult.buildErrorStateResult("密码不能为空", null);
    }
    if (password.length() < 6 || password.length() > 12) {
      return JsonResult.buildErrorStateResult("密码应为6-12位", null);
    }
    smsValidForRegister(phoneNo, verificationCode);
    if (!userService.exist(phoneNo)) {
      LOGGER.info("修改密码失败，该手机号尚未注册, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("该手机号尚未注册", null);
    }
    if (!userService.resetPassword(phoneNo, password)) {
      return JsonResult.buildErrorStateResult("修改密码失败", null);
    }

    // TODO  加渠道号
    LOGGER.info("修改密码成功, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return JsonResult.buildSuccessResult(null, null);
  }

  /**
   * 检查token是否已经过期不存在了
   *
   * @param token
   * @return
   */
  @RequestMapping("/exists_token")
  public JsonResult checkToken(@RequestParam String token) {
    String tokenKey = Constants.SESSION_PREFIX + token;
    return JsonResult.buildSuccessResult(null, stringRedisTemplate.hasKey(tokenKey));
  }


  private String genRandomPwd() {
    int pwdMax = PWD_BASE.length;
    int i; // 生成的随机数
    int count = 0; // 生成的密码的长度
    StringBuffer pwd = new StringBuffer();
    Random r = new Random();
    while (count < 15) {
      i = Math.abs(r.nextInt(pwdMax)); // 生成的数最大为36-1
      if (i >= 0 && i < PWD_BASE.length) {
        pwd.append(PWD_BASE[i]);
        count++;
      }
    }
    return pwd.toString();
  }


  @RequestMapping("/syncUserInfo")
  public JsonResult syncUserInfo() {
    User user = getCurrentUserFromRedis();
    if (null == user) {
      return JsonResult.buildErrorStateResult(null, null);
    }
    UserDetail detail = userDetailService.findByUserId(user.getId());
    //UserDetail detail = userDetailRepository.findByUserId(user.getId());
    UserModel userModel = new UserModel(user, detail);
    return JsonResult.buildSuccessResult("token校验成功", userModel);
  }

  private User registerFastWhenLogin(String phoneNo, Long channelId, Long registerFrom, String appChannel) {
    String password = genRandomPwd();
    LOGGER.info("用户快速注册, phoneNo:{}, channelId:{}, registerFrom:{},appChannel:{}", phoneNo, channelId, registerFrom, appChannel);
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      LOGGER.info("用户快速注册失败，手机号错误, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      throw new UserNotExistException("手机号错误");
    }
    if (null == registerFrom) {
      registerFrom = 1L;
    }
    User newUser = userService.registerAndReturn(phoneNo, password, registerFrom);
    if(newUser != null && newUser.getId() != null && newUser.getId() > 0){
      LOGGER.info("用户快速注册成功, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
    }
    return newUser;

  }

  private JsonResult loginWithHttpBasic(Long channelId, String appChannel, Long createdFrom, Merchant merchant, HttpServletRequest request, String openId,String dimension) {
    User user = verificateUserNameAndPassword(request, openId);
    if (user == null) {
      return JsonResult.buildErrorStateResult("用户名或密码不正确", null);
    }
    //增加登陆统计发送
    UserStatistics statistics=new UserStatistics(user,dimension,1,channelId);
    MqUtils.sendLoanVest(statistics);
    return new JsonResult(sessionService.createSession(channelId, createdFrom, appChannel, user, merchant));
  }

  private User verificateUserNameAndPassword(HttpServletRequest request, String openId) {
    String credential = request.getHeader("authorization");
    if (!credential.startsWith("Basic ")) {
      return null;
    }
    credential = credential.substring("Basic ".length(), credential.length());
    byte[] buf = Base64.decodeBase64(credential);
    String bufStr = "";
    try {
      bufStr = new String(buf, "UTF-8");
    } catch (UnsupportedEncodingException e) {
      LOGGER.error("不支持的编码: ", e);
    }
    String[] credentialArr = bufStr.split(":");
    if (credentialArr.length != 2) {
      return null;
    }
    String userName = credentialArr[0];
    String pass = credentialArr[1];
    User user = userService.findByPhoneWithCache(userName);
    if (user == null || !user.getEnable()) {
      return null;
    }
    //验证密码
    if (!validatePassword(pass, user.getPassword())) {
      // 密码错误时，给该账号添加计数器
      String key = Constants.REDIS_PASSWORD_ERROR_COUNT + userName;
      if (!stringRedisTemplate.hasKey(key)) {
        stringRedisTemplate.opsForValue().set(key, String.valueOf(0), Constants.ONE_DAY, TimeUnit.SECONDS);
      }
      // 密码错误计数
      Long errorCount = stringRedisTemplate.opsForValue().increment(key, 1L);
      if(errorCount > Constants.Image_Need_Count){
        throw new PasswordErrorLimitException("用户名或密码不正确");
      }else if(Objects.equals(errorCount, Constants.Image_Need_Count)){
        throw new PasswordErrorLimitException("请输入图形验证码");
      }
      return null;
    }
    return user;
  }

  private boolean validatePassword(String paramPass, String targetPassword) {
    return StringUtils.defaultString(targetPassword, "").equals(PasswordUtil.MD5(paramPass.toLowerCase() + pwdSalt));
  }

  private JsonResult loginWithUserId(Long channelId, String appChannel, Long createdFrom, String userId, Merchant merchant,String dimension) {
    //查询用户,存在则保存用户session信息,userId为uuid
    User user = userService.findByUuidInDb(userId);
    //用户信息存在,更新session中的最后访问时间,重新写入缓存.
    if (null != user || !user.getEnable()) {
      //增加登陆统计发送
      UserStatistics statistics=new UserStatistics(user,dimension,1,channelId);
      MqUtils.sendLoanVest(statistics);
      return new JsonResult(sessionService.createSession(channelId, createdFrom, appChannel, user, merchant));
    } else {
      return JsonResult.buildErrorStateResult("登录失败", null);
    }
  }

  @RequestMapping("/associate_wechat")
  public JsonResult associateWithWechat(String openId) {
    User user = getCurrentUserFromRedis();
    Long userId = user.getId();
    WechatUserInfo userInfo = wechatService.findWechatUserInfoFromDb(openId);
    LOGGER.info("微信关联openId,user:[{}],openId:[{}],wechatUserInfo:[{}]",user,openId,userInfo);

    // 已经绑定过了
    if (userInfo != null && StringUtils.isNotEmpty(userInfo.getPhoneNo())) {
      return JsonResult.buildSuccessResult(null, null);
    }
    // 前置绑定微信出错
    if (userInfo == null) {
      return JsonResult.buildSuccessResult(null, null);
    }
    // 未绑定信用钱包用户
    if (userInfo.getUserId() == null) {
      userInfo.setUserId(userId);
      userInfo.setPhoneNo(user.getPhoneNo());
      try {
        wechatService.saveWechatUserInfo(userInfo);
      } catch (Exception ex) {
        // 不做绑定
        return JsonResult.buildErrorStateResult("该手机号已绑定其他微信号码", null);
      }
      return JsonResult.buildSuccessResult(null, null);
    }
    return JsonResult.buildSuccessResult(null, null);
  }

  /**
   * 注册时校验短信验证码
   * @param phoneNo
   * @param verificationCode
   */
  private void smsValidForRegister(String phoneNo, String verificationCode) {
    if (!smsService.validRegisterOrResetPasswdVerificationCode(phoneNo, verificationCode)) {
      smsReSendOrNot(phoneNo);
      LOGGER.info("用户快速注册，验证码校验失败,phoneNo:{} , verificationCode:{}", phoneNo, verificationCode);
      throw new VerificationCodeErrorException("短信验证码错误");
    }
  }

  /**
   * 登录时校验短信验证码
   * @param phoneNo
   * @param verificationCode
   */
  private void smsValidForFastLogin(String phoneNo, String verificationCode) {
    if (!smsService.validateFastLoginVerificationCode(phoneNo, verificationCode)) {
      smsReSendOrNot(phoneNo);
      LOGGER.info("用户快速登录，验证码校验失败,phoneNo:{} , verificationCode:{}", phoneNo, verificationCode);
      throw new VerificationCodeErrorException("短信验证码错误");
    }
  }

  /**
   * 是否需要重新获取短信验证码
   * @param phoneNo
   */
  private void smsReSendOrNot(String phoneNo) {
    if(needRetSendCode(phoneNo)){
      String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
      stringRedisTemplate.delete(key);
      throw new VerificationCodeErrorException("验证码失效，请重新获取");
    }
  }

  /**
   * 是否需要重新发送短信验证码
   * @param phoneNo
   * @return
   */
  private boolean needRetSendCode(String phoneNo) {
    String verificationCountKey = Constants.REDIS_VERIFICATION_COUNT + phoneNo;
    Long getVerificationCount = stringRedisTemplate.opsForHash().increment(verificationCountKey, Constants.REDIS_VERIFICATION_COUNT, 1);
    return (getVerificationCount >= 5);
  }

}
