package cn.quantgroup.xyqb.aspect.captcha;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.thirdparty.jcaptcha.AbstractManageableImageCaptchaService;
import com.octo.captcha.service.CaptchaServiceException;

import java.io.PipedReader;
import java.nio.charset.Charset;
import java.security.PrivateKey;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

/**
 * 类名称：CaptchaValidateAdvisor
 * 类描述：
 *
 * @author 李宁
 * @version 1.0.0 创建时间：15/11/17 14:49 修改人： 修改时间：15/11/17 14:49 修改备注：
 */
@Aspect
@Component
public class CaptchaNewValidateAdvisor {

    private static final Logger LOGGER = LoggerFactory.getLogger(CaptchaNewValidateAdvisor.class);
    private static final String SUPER_CAPTCHA_ID = UUID.nameUUIDFromBytes("__QG_APPCLIENT_AGENT__".getBytes(Charset.forName("UTF-8"))).toString();
    private static final String SUPER_CAPTCHA = "__SUPERQG__";

    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> redisTemplate;

    @Autowired
    @Qualifier("customCaptchaService")
    private AbstractManageableImageCaptchaService imageCaptchaService;

    /**
     * 自动化测试忽略验证码
     */
    @Value("${xyqb.auth.captcha.autotest.enable:false}")
    private boolean autoTestCaptchaEnabled;

    /**
     * 图形验证码切面
     */
    @Pointcut("@annotation(cn.quantgroup.xyqb.aspect.captcha.CaptchaNewValidator)")
    private void needNewCaptchaValidate() {
    }

    private static final String IMAGE_IP_COUNT = "image:ip";
    private static final String IMAGE_PHONE_COUNT = "image:phone";
    private static final String IMAGE_DEVICEID_COUNT = "image:deviceId:";
    private static final Long FIVE_MIN = 24 * 5L;

    /**
     * 在受图形验证码保护的接口方法执行前, 执行图形验证码校验
     * captchaId 图形验证码key
     * captchaValue 图形验证码value
     *
     * @throws Throwable
     */
    @Around("needNewCaptchaValidate()")
    private Object doCapchaValidate(ProceedingJoinPoint pjp) throws Throwable {

        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        String registerFrom = Optional.ofNullable(request.getParameter("registerFrom")).orElse("");
        String captchaId = Optional.ofNullable(request.getParameter("captchaId")).orElse("");
        Object captchaValue = request.getParameter("captchaValue");
        String phoneNo = request.getParameter("phoneNo");
        String deviceId = Optional.ofNullable(request.getParameter("deviceId")).orElse("");
        String clientIp = getIp();
        Long countIP = countIP(clientIp);
        Long countPhone = countPhone(phoneNo);
        Long countDeviceId = countDeviceId(deviceId);

        if (countIP > Constants.Image_Need_Count || countPhone > Constants.Image_Need_Count || countDeviceId > Constants.Image_Need_Count) {

            if (shouldSkipCaptchaValidate(registerFrom, captchaId, captchaValue)) {
                LOGGER.info("使用超级图形验证码校验, registerFrom={}, clientIp={}", registerFrom, request.getRemoteAddr());
                return pjp.proceed();
            }

            JsonResult result = JsonResult.buildSuccessResult("图形验证码不正确", "");
            result.setBusinessCode("0002");
            if (captchaValue != null && StringUtils.isNotEmpty(String.valueOf(captchaValue))) {
                String captcha = String.valueOf(captchaValue);
                // 忽略用户输入的大小写
                captcha = StringUtils.lowerCase(captcha);
                // 验证码校验
                Boolean validCaptcha = false;
                try {
                    validCaptcha = imageCaptchaService.validateResponseForID(Constants.IMAGE_CAPTCHA_KEY + captchaId, captcha);
                } catch (CaptchaServiceException ex) {
                    LOGGER.error("验证码校验异常, {}, {}", ex.getMessage(), ex);
                }

                if (validCaptcha) {
                    return pjp.proceed();
                }
                return result;
            }
            result.setMsg("请输入图形验证码");
            return result;

        }

        return pjp.proceed();
    }

    private boolean shouldSkipCaptchaValidate(String registerFrom, String captchaId, Object captchaValue) {

        // 如果启用了超级验证码功能, 检查超级验证码, 超级验证码区分大小写
        if (autoTestCaptchaEnabled) {
            return true;
        }

        return StringUtils.equals(SUPER_CAPTCHA_ID, String.valueOf(captchaId)) && StringUtils.equals(SUPER_CAPTCHA, String.valueOf(captchaValue));
    }

    private Long countIP(String clientIp) {
        Long count = 1L;
        if (StringUtils.isBlank(clientIp)) {
            return count;
        } else {
            String countString = redisTemplate.opsForValue().get(IMAGE_IP_COUNT + clientIp);
            if (StringUtils.isBlank(countString)) {
                redisTemplate.opsForValue().set(IMAGE_IP_COUNT + clientIp, String.valueOf(count),
                        FIVE_MIN, TimeUnit.SECONDS);
            } else {
                count = Long.valueOf(countString) + 1L;
                redisTemplate.opsForValue().set(IMAGE_IP_COUNT + clientIp, String.valueOf(count),
                        FIVE_MIN, TimeUnit.SECONDS);
            }
            return count;
        }
    }

    private Long countPhone(String phoneNo) {
        Long count = 1L;
        String countString = redisTemplate.opsForValue().get(IMAGE_PHONE_COUNT + phoneNo);
        if (StringUtils.isBlank(countString)) {
            redisTemplate.opsForValue().set(IMAGE_PHONE_COUNT + phoneNo, String.valueOf(count),
                    FIVE_MIN, TimeUnit.SECONDS);
        } else {
            count = Long.valueOf(countString) + 1L;
            redisTemplate.opsForValue().set(IMAGE_PHONE_COUNT + phoneNo, String.valueOf(count),
                    FIVE_MIN, TimeUnit.SECONDS);
        }
        return count;
    }

    /**
     * 短信发送设备限制
     */
    private Long countDeviceId(String deviceId) {
        Long count = 1L;
        if (StringUtils.isBlank(deviceId)) {
            return count;
        } else {
            String countString = redisTemplate.opsForValue().get(IMAGE_DEVICEID_COUNT + deviceId);
            if (StringUtils.isBlank(countString)) {
                redisTemplate.opsForValue().set(IMAGE_DEVICEID_COUNT + deviceId, String.valueOf(count),
                        FIVE_MIN, TimeUnit.SECONDS);
            } else {
                count = Long.valueOf(countString) + 1L;
                redisTemplate.opsForValue().set(IMAGE_DEVICEID_COUNT + deviceId, String.valueOf(count),
                        FIVE_MIN, TimeUnit.SECONDS);
            }
            return count;
        }
    }

    private String getIp() {
        HttpServletRequest request = getRequest();

        String ip = request.getHeader("x-real-ip");
        if (StringUtils.isEmpty(ip)) {
            ip = request.getRemoteAddr();
        }

        //过滤反向代理的ip
        String[] stemps = ip.split(",");
        if (stemps.length >= 1) {
            //得到第一个IP，即客户端真实IP
            ip = stemps[0];
        }

        ip = ip.trim();
        if (ip.length() > 23) {
            ip = ip.substring(0, 23);
        }

        return ip;
    }

    private HttpServletRequest getRequest() {
        ServletRequestAttributes attrs = (ServletRequestAttributes) RequestContextHolder
                .getRequestAttributes();
        return attrs.getRequest();
    }
}
