package cn.quantgroup.xyqb.controller.internal.user;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.accessable.IpValidator;
import cn.quantgroup.xyqb.aspect.forbidden.AccessForbiddenValidator;
import cn.quantgroup.xyqb.config.data.WechatConfiguration;
import cn.quantgroup.xyqb.constant.UserConstant;
import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.controller.internal.user.resp.UserFullResp;
import cn.quantgroup.xyqb.entity.*;
import cn.quantgroup.xyqb.event.DisableActiveEvent;
import cn.quantgroup.xyqb.event.UserDisableRecordEvent;
import cn.quantgroup.xyqb.exception.UserNotExistException;
import cn.quantgroup.xyqb.model.*;
import cn.quantgroup.xyqb.remote.StmsRemoteService;
import cn.quantgroup.xyqb.repository.IUserDisableRecordRepository;
import cn.quantgroup.xyqb.repository.IUserInfoRepository;
import cn.quantgroup.xyqb.service.api.IUserApiService;
import cn.quantgroup.xyqb.service.merchant.IMerchantService;
import cn.quantgroup.xyqb.service.register.IUserDeregisterService;
import cn.quantgroup.xyqb.service.register.IUserRegisterService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.service.user.IUserBtRegisterService;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.service.wechat.IWechatService;
import cn.quantgroup.xyqb.util.*;
import cn.quantgroup.xyqb.util.encrypt.Md5Util;
import cn.quantgroup.xyqb.validator.ChineseName;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.google.common.base.MoreObjects;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.http.HttpStatus;
import org.hibernate.validator.constraints.Length;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.constraints.Min;
import javax.validation.constraints.NotNull;
import java.io.UnsupportedEncodingException;
import java.util.*;
import java.util.stream.Collectors;

import static cn.quantgroup.xyqb.constant.UserConstant.USER_ERROR_OR_ENABLE_ERROR;


/**
 * Created by Miraculous on 2016/12/19.
 */
@Slf4j
@RestController
@RequestMapping("/innerapi")
@Validated
public class InnerController implements IBaseController {

    @Autowired
    private IMerchantService merchantService;
    @Autowired
    private IUserService userService;
    @Autowired
    private IUserInfoRepository userInfoRepository;
    @Autowired
    private IUserBtRegisterService userBtRegisterService;

    @Autowired
    private IWechatService wechatService;
    @Autowired
    private ISessionService sessionService;
    @Autowired
    private IUserApiService userApiService;
    @Autowired
    private ISmsService smsService;
    @Autowired
    private IUserRegisterService userRegisterService;
    @Resource
    private ApplicationEventPublisher applicationEventPublisher;

    @Autowired
    private StmsRemoteService stmsRemoteService;
    @Autowired
    private WechatConfiguration wechatConfiguration;

    @Autowired
    private IUserDeregisterService userDeregisterService;
    @Autowired
    private IUserDisableRecordRepository iUserDisableRecordRepository;
    private static final String RESET_PWD_TOKEN = "ecf75c1f-2ccb-4661-8e4b-2874c0f45a2b";
    private static final String MODIFY_CONTACT_TOKEN = "@qwsdedad131323213w!";


    /**
     * 根据手机号或身份证号查询用户UUID
     *
     * @param phoneNo - 手机号
     * @return user表的uuid
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/187
     * @Deprecated 20210318
     */
    @Deprecated
    @IpValidator
    @RequestMapping("/fetchUuid")
    public JsonResult fetchUuid(String phoneNo,@RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {
        String uuid = userService.findUuid(phoneNo,tenantId);
        log.info("根据手机号或身份证号查询用户UUID,phoneNo:{},uuid:{}", phoneNo, uuid);
        if (org.apache.commons.lang.StringUtils.isBlank(uuid)) {
            return JsonResult.buildErrorStateResult("未找到该用户", null);
        }
        return JsonResult.buildSuccessResult(null, uuid);
    }

    /**
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/215
     */
    @RequestMapping("/user/search/phoneNo")
    public JsonResult findByPhoneNo(String phoneNo, Integer tenantId) {
        User user = userService.findByPhoneInDb(phoneNo,tenantId);
        if (user == null) {
            return JsonResult.buildErrorStateResult("查无此人", null, 4L);
        }

        return JsonResult.buildSuccessResult("", new UserRet(user));
    }

    /**
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/219
     */
    @RequestMapping("/user/search/uuid")
    public JsonResult findByUuidInfo(String uuid, Integer tenantId) {
        User user = userService.findByUuidWithCache(uuid,tenantId);
        if (user == null) {
            return JsonResult.buildErrorStateResult("查无此人", null);
        }

        return JsonResult.buildSuccessResult("", new UserRet(user));
    }

    /**
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/229
     */
    @RequestMapping("/userInfo/search/uuid")
    public JsonResult findUserInfoByUuid(@RequestParam(value = "uuid") String uuid,@RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        log.info("需要查询的用户uuidid, uuid:" + uuid);

        if (StringUtils.isBlank(uuid)) {
            return JsonResult.buildSuccessResult(null, null);
        }
        User user = userService.findByUuidWithCache(uuid,tenantId);
        if (null != user) {
            if (!user.getEnable()) {
                return JsonResult.buildSuccessResult(USER_ERROR_OR_ENABLE_ERROR, null);
            }
            UserInfoEntity userDetail = userInfoRepository.findByUserIdAndTenantId(user.getId(),tenantId);
            UserInfo info = new UserInfo(user, userDetail);
            return JsonResult.buildSuccessResult("", info);
        }
        return JsonResult.buildSuccessResult("", null);
    }

    /**
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/227
     */
    @AccessForbiddenValidator
    @RequestMapping("/userInfo/search/phone")
    public JsonResult findUserInfoByPhone(@RequestParam(value = "phone") String phone,@RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        log.info("需要查询的用户phone, phone:" + phone);

        if (StringUtils.isBlank(phone)) {
            return JsonResult.buildSuccessResult(null, null);
        }
        User user = userService.findByPhoneInDb(phone,tenantId);
        if (null != user) {
            if (!user.getEnable()) {
                return JsonResult.buildSuccessResult(USER_ERROR_OR_ENABLE_ERROR, null);
            }
            UserInfoEntity userDetail = userInfoRepository.findByUserIdAndTenantId(user.getId(),tenantId);
            UserInfo info = new UserInfo(user, userDetail);
            return JsonResult.buildSuccessResult("", info);
        }
        return JsonResult.buildSuccessResult("", null);
    }

    /**
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/205
     */
    @RequestMapping("/user/getPhoneByUserIds")
    public JsonResult findByIds(@RequestParam(value = "userIds") String userIdsString,
                                @RequestParam(required = false) Integer tenantId) {
        if (StringUtils.isEmpty(userIdsString)) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        //默认羊小咩租户
        if (TenantUtil.validationTenantIdIsNullOrZero(tenantId)) {
            tenantId = TenantUtil.TENANT_DEFAULT;
        }
        List<Long> userIds = JSONObject.parseObject(userIdsString, new TypeReference<List<Long>>() {
        });
        if (!CollectionUtils.isEmpty(userIds) && userIds.size() <= Constants.USER_ID_BATCH_SIZE) {
            Map<Long, String> userIdAndPhoneMap = userService.findPhoneByIdsInDb(userIds, tenantId);
            return JsonResult.buildSuccessResult("", userIdAndPhoneMap);
        } else {
            return JsonResult.buildErrorStateResult("批量查询每次最多进行500条用户信息的查询", null);
        }
    }

    /**
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/211
     */
    @RequestMapping("/user/save")
    public JsonResult saveUser(String phoneNo, Long registeredFrom, Integer tenantId,@RequestHeader(value =Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantIdHeader) {
        log.info("保存用户,phoneNo:{},registeredFrom:{}", phoneNo, registeredFrom);
        //参数验证
        if (StringUtils.isBlank(phoneNo)) {
            return JsonResult.buildErrorStateResult("用户手机号不能为空.", null);
        }
        if (registeredFrom == null) {
            registeredFrom = 0L;
        }
        if (tenantId == null) {
            tenantId = tenantIdHeader;
        }
        User user = userService.findByPhoneWithCache(phoneNo,tenantId);

        if (user != null) {
            //存在已注销
            if (!user.getEnable()) {
                log.info("用户已经注销，phoneNo：{}", phoneNo);
                return JsonResult.buildErrorStateResult(USER_ERROR_OR_ENABLE_ERROR, null);
            }


            //已存在用户
            return JsonResult.buildSuccessResult(null, new UserRet(user));
        }
        //注册新用户
        user = userRegisterService.register(registeredFrom, phoneNo, null, 0L, 0L, tenantId);

        return JsonResult.buildSuccessResult(null, new UserRet(user));
    }

    /**
     * 保存用户详细信息 适用于：创建 或 修改
     *
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/285
     */
    @RequestMapping("/user_detail/save")
    public JsonResult saveUserDetail(
            @Min(value = 1, message = "用户id为空") @RequestParam Long userId,
            String phoneNo,
            @ChineseName @RequestParam String name,
            String idNo,
            String email,
            String qq,
            @RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {
        log.info(
                "保存用户详细信息,[saveUserDetail] userId:{},phoneNo:{},name:{},idNo:{},email:{},qq:{}",
                userId, phoneNo, name, idNo, email, qq);
        //参数验证
        if (Objects.isNull(userId)) {
            return JsonResult.buildErrorStateResult("userId为空.", null);
        }
        if (StringUtils.isBlank(phoneNo)) {
            return JsonResult.buildErrorStateResult("用户手机号为空.", null);
        }
        if (StringUtils.isBlank(name)) {
            return JsonResult.buildErrorStateResult("用户姓名为空.", null);
        }

        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult("用户手机号错误.", null);
        }
        if (!ValidationUtil.validateChinese(name)) {
            return JsonResult.buildErrorStateResult("用户姓名错误.", null);
        }

        User user = userService.findById(userId,tenantId);
        if (Objects.isNull(user)) {
            return JsonResult.buildErrorStateResult("用户不存在", null);
        } else if (!Objects.equals(user.getPhoneNo(), phoneNo)) {
            return JsonResult.buildErrorStateResult("用户手机号不匹配", null);
        }
        /*
         * 如果已存在记录，则更新
         */
        UserInfoEntity userDetail = userInfoRepository.findByUserIdAndTenantId(userId,tenantId);
        if (Objects.isNull(userDetail)) {
            userDetail = userInfoRepository.findByPhoneNoAndTenantId(phoneNo,tenantId);
            // 按手机号查出记录，如果userId非空，说明是存疑数据或是其他用户的信息，停止修改操作，返回失败
            if (Objects.nonNull(userDetail) && Objects.nonNull(userDetail.getUserId())) {
                return JsonResult.buildErrorStateResult("手机号已使用.", null);
            }
        }
        if (Objects.isNull(userDetail)) {
            userDetail = new UserInfoEntity();
        }
        userDetail.setUserId(userId);
        userDetail.setName(name);
        userDetail.setPhoneNo(phoneNo);
        userDetail.setEmail(email);
        userDetail.setQq(qq);
        userDetail = userInfoRepository.save(userDetail);
        log.info("InnerController saveUserDetail, userId:{}, phoneNo:{}, name:{}", user.getId(),
                phoneNo, name);
        if (userDetail != null) {
            return JsonResult.buildSuccessResult(null, UserInfoRet.getUserDetail(userDetail));
        }
        return JsonResult.buildErrorStateResult("", null);
    }

    /**
     * 根据用户id查询用户的详细信息
     *
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/237
     */
    @AccessForbiddenValidator
    @RequestMapping("/user_detail/search/userId")
    public JsonResult findUserDetailByUserId(Long userId, Integer tenantId,@RequestHeader(value =Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantIdHeader) {
        if (tenantId == null) {
            tenantId = tenantIdHeader;
        }
        UserInfoEntity userDetail = null;
        // 增加容错性，防备DB中存在的脏数据触发异常
        if (userId != null && userId > 0) {
            userDetail = userInfoRepository.findByUserIdAndTenantId(userId,tenantId);
        }
        if (Objects.isNull(userDetail)) {
            return JsonResult.buildErrorStateResult("该用户无实名信息", null);
        }

        return JsonResult.buildSuccessResult("用户实名信息", UserInfoRet.getUserDetail(userDetail));
    }

    /**
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/217
     */
    @RequestMapping("/user/search/userId")
    public JsonResult findUserByUserId(Long userId, Integer tenantId) {
        User user = userService.findById(userId,tenantId);
        if (user != null) {
            return JsonResult.buildSuccessResult(null, new UserRet(user));
        }

        return JsonResult.buildErrorStateResult("查无此人", null);
    }

    /**
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/235
     */
    @AccessForbiddenValidator
    @RequestMapping("/user_detail/search/phone")
    public JsonResult findUserDetailByPhone(String phoneNo, Integer tenantId) {
        UserInfoEntity userDetail = userInfoRepository.findByPhoneNoAndTenantId(phoneNo,tenantId);
        if (userDetail != null) {
            return JsonResult.buildSuccessResult(null, UserInfoRet.getUserDetail(userDetail));
        }

        return JsonResult.buildErrorStateResult("该用户详情信息不存在", null);
    }

    /**
     * 根据phone查找用户完整信息
     *
     * @param phoneNo
     * @return
     */
    @RequestMapping("/user_full_info/search/phone")
    public JsonResult findUserFullInfoByPhone(String phoneNo,@RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isBlank(phoneNo)) {
            log.warn("[findUserFullInfoByPhone]phoneNo为空");
            return JsonResult.buildErrorStateResult(null, null);
        }
        Optional<UserFullInfo> userFullInfoOptional = userApiService.getUserFullInfoByPhone(phoneNo,tenantId);
        if (!userFullInfoOptional.isPresent()) {
            log.warn("[findUserFullInfoByPhone]没有用户信息，phoneNo={}", Md5Util.build(phoneNo));
            return JsonResult.buildErrorStateResult(null, null);
        }
        return JsonResult.buildSuccessResult(null, userFullInfoOptional.get());
    }

    /**
     * 根据uuid查找用户完整信息
     *
     * @param uuid
     * @return
     */
    @RequestMapping("/user_full_info/search/uuid")
    public JsonResult findUserFullInfoByUUuid(String uuid,@RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isBlank(uuid)) {
            log.warn("[findUserFullInfoByUUuid]uuid为空");
            return JsonResult.buildErrorStateResult(null, null);
        }
        Optional<UserFullInfo> userFullInfoOptional = userApiService.getUserFullInfoByUuid(uuid,tenantId);
        if (!userFullInfoOptional.isPresent()) {
            log.warn("[findUserFullInfoByUUuid]没有用户信息，uuid={}", uuid);
            return JsonResult.buildErrorStateResult(null, null);
        }
        return JsonResult.buildSuccessResult(null, userFullInfoOptional.get());
    }

    /**
     * 更新用户email和qq
     *
     * @param qq
     * @param email
     * @param userId - 用户主键
     * @return
     */
    @RequestMapping("/user_detail/update")
    public JsonResult updateUserDetail(String qq, String email, Long userId,@RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {
        if (Objects.isNull(userId) || userId == 0L) {
            return JsonResult.buildErrorStateResult("userId为空", null);
        }
        if (StringUtils.isNotBlank(qq)) {
            userInfoRepository.updateUserQq(qq,userId,tenantId);
        }
        if (StringUtils.isNotBlank(email)) {
            userInfoRepository.updateUserEmail(email,userId,tenantId);
        }
        return JsonResult.buildSuccessResult(null, null);
    }

    @RequestMapping("/user_detail/update/qq")
    public JsonResult updateUserQQ(String qq, Long userId,@RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isEmpty(qq) || userId == null || userId == 0L) {
            return JsonResult.buildErrorStateResult("参数校验失败,qq或用户id为空", null);
        }
        userInfoRepository.updateUserQq(qq,userId,tenantId);
        return JsonResult.buildSuccessResult(null, null);
    }

    @RequestMapping("/user/query/openId")
    public JsonResult queryOpenIdByUserId(Long userId,@RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (userId == null) {
            return JsonResult.buildErrorStateResult("userId不能为空", null);
        }
        WechatUserInfo wechatUserInfo = wechatService.queryByUserId(userId,tenantId);
        log.info("根据用户userId查询：微信关联：userId：{},WechatUserInfo:{}", userId, wechatUserInfo);
        if (wechatUserInfo == null) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        return JsonResult.buildSuccessResult("success", wechatUserInfo.getOpenId());
    }

    /**
     * 按照用户主键、账号或uuid查询用户详细信息
     *
     * @param id      - 用户表主键
     * @param phoneNo - 用户账号
     * @param uuid    - 用户全球唯一键（uuid）
     * @return 详细信息包含： { 账号信息 个人信息 }
     */
    @RequestMapping("/user-association/search")
    public JsonResult findUserAssociationModel(Long id, String phoneNo, String uuid,@RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        User user = null;
        if (!Objects.isNull(id) && id > 0) {
            user = userService.findById(id,tenantId);
        } else if (ValidationUtil.validatePhoneNo(phoneNo)) {
            user = userService.findByPhoneWithCache(phoneNo,tenantId);
        } else if (StringUtils.isNotBlank(uuid)) {
            user = userService.findByUuidWithCache(uuid,tenantId);
        }
        UserInfoEntity userDetail = null;
        if (Objects.nonNull(user) && Objects.nonNull(user.getId()) && user.getId() > 0) {
            userDetail = userInfoRepository.findByUserIdAndTenantId(user.getId(),tenantId);
        }
        if (Objects.isNull(user)) {
            return JsonResult.buildErrorStateResult("用户不存在", "");
        } else {
            return JsonResult.buildSuccessResult("", new UserInfo(user, userDetail));
        }
    }

    /**
     * 查询用户综合信息模型
     *
     * @param user - 用户User
     * @return 综合信息模型包含： { 账号信息 个人信息 扩展信息 地址信息 联系人信息 }
     */
    private UserAssociationModel findUserAssociationModelByUser(User user,Integer tenantId) {
        if (Objects.isNull(user) || Objects.isNull(user.getId()) || user.getId() < 1) {
            return null;
        }
        UserAssociationModel bean = new UserAssociationModel();
        bean.setId(user.getId());
        bean.setUuid(user.getUuid());
        bean.setRegisterFrom(user.getRegisteredFrom());
        UserInfoEntity userDetail = userInfoRepository.findByUserIdAndTenantId(user.getId(),tenantId);
        if (!Objects.isNull(userDetail)) {
            bean.setPhoneNo(userDetail.getPhoneNo());
            bean.setName(userDetail.getName());
            bean.setQq(userDetail.getQq());
            bean.setEmail(userDetail.getEmail());
            bean.setGender(Optional.ofNullable(userDetail.getGender()).orElse(Gender.UNKNOWN).getName());
        }
        UserBtRegister userBtRegister = userBtRegisterService.findByUserId(user.getId());
        if (!Objects.isNull(userBtRegister)) {
            bean.setMerchantId(userBtRegister.getRegisterBtMerchantId());
        }
        return bean;
    }

    @RequestMapping(path = "/user-association/search/userId", method = {RequestMethod.GET,
            RequestMethod.POST})
    public JsonResult findUserAssociationModelByUserId(
            @RequestParam(name = "userId", required = false) Long userId,
            @RequestParam(name = "phoneNo", required = false) String phoneNo
            ,@RequestHeader(value =Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        boolean userIdOk = Objects.nonNull(userId) && userId > 0;
        boolean phoneNoOk = ValidationUtil.validatePhoneNo(phoneNo);
        if (!userIdOk && !phoneNoOk) {
            return JsonResult.buildErrorStateResult("params invalid",
                    String.valueOf(userId).concat(":").concat(phoneNo));
        }
        UserAssociationModel bean = null;
        User user = userIdOk ? userService.findById(userId,tenantId) : userService.findByPhoneWithCache(phoneNo,tenantId);
        if (!Objects.isNull(user)) {
            bean = findUserAssociationModelByUser(user,tenantId);
        }
        return JsonResult.buildSuccessResult("", bean);
    }

    /**
     * 查询用户全量信息 for koala
     *
     * @param userId 用户id
     * @return
     */
    @GetMapping("/user-full/search/userId")
    public JsonResult<UserFullResp> findUserFullSearchByUserId(@RequestParam Long userId,@RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        return JsonResult.buildSuccessResultGeneric(userService.findUserFullSearchByUserId(userId,tenantId));
    }

    @RequestMapping("/user-association/search/phone")
    public JsonResult findUserAssociationByPhone(String phoneNo,@RequestHeader(value =Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        UserInfoEntity userInfoEntity = userInfoRepository.findByPhoneNoAndTenantId(phoneNo,tenantId);
        UserAssociation bean = getUserAssociation(userInfoEntity);
        return JsonResult.buildSuccessResult("", bean);
    }

    @RequestMapping("/user-association/search/uid")
    public JsonResult findUserAssociationByUid(Long uid,@RequestHeader(value =Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        UserInfoEntity userDetail = userInfoRepository.findByUserIdAndTenantId(uid,tenantId);
        UserAssociation bean = getUserAssociation(userDetail);
        return JsonResult.buildSuccessResult("", bean);
    }

    /**
     * 获取用户综合信息
     *
     * @param userInfoEntity
     * @return
     */
    private UserAssociation getUserAssociation(UserInfoEntity userInfoEntity) {
        UserAssociation bean = new UserAssociation();
        if (null != userInfoEntity) {
            bean.setUserId(userInfoEntity.getUserId());
            bean.setPhoneNo(userInfoEntity.getPhoneNo());
            bean.setName(userInfoEntity.getName());
            bean.setQq(userInfoEntity.getQq());
            bean.setGender(Optional.ofNullable(userInfoEntity.getGender()).orElse(Gender.UNKNOWN).getName());
        }
        return bean;
    }


    @RequestMapping("/user/wechat/phone_no")
    public JsonResult queryOpenIdByPhoneNo(String phoneNo,@RequestHeader(value =Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ,@RequestHeader(value = Constants.X_AUTH_APP_ID,defaultValue = UserConstant.defaultAppId) String  appId) {

        if (org.apache.commons.lang3.StringUtils.isEmpty(appId)) {
            appId = wechatConfiguration.getDefault().getAppId();
        }

        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult("手机号格式错误", null);
        }
        WechatUserInfo wechatUserInfo = wechatService.findWechatUserInfoByPhoneNo(phoneNo, tenantId);
        return JsonResult.buildSuccessResult(null,
                null == wechatUserInfo ? null : wechatUserInfo.getOpenId());
    }

    /**
     * 刷新用户激活状态
     *
     * @param userId - 用户主键
     * @param enable - 用户激活状态
     * @return
     */
    private boolean flushUserStatus(Long userId, boolean enable,Integer tenantId) {
        if (null == userId || 0L == userId) {
            log.info("刷新用户激活状态失败：userId:{},enable:{}", userId, enable);
            return false;
        }
        User user = userService.findById(userId,tenantId);
        if (null == user) {
            log.info("刷新用户激活状态失败：userId:{},enable:{},user:{}", userId, enable, user);
            return false;
        }
        user.setEnable(enable);
        log.info("刷新用户激活状态失败：userId:{},enable:{},user:{}", userId, enable, user);
        user = userService.saveUser(user);
        if (!user.getEnable()) {
            sessionService.deleteByUserId(userId,tenantId);
            sessionService.deleteUserCatch(user,tenantId);
            // 禁用微信
            wechatService.forbiddenUserWeChat(userId,tenantId);
        }
        applicationEventPublisher.publishEvent(new DisableActiveEvent(this, user));
        return Objects.equals(enable, user.getEnable());
    }

    @RequestMapping("/user/findByPhones")
    public JsonResult getUserIdByPhones(@RequestParam("userPhones") String userPhones, @RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isBlank(userPhones)) {
            return JsonResult.buildErrorStateResult("传入用户手机号不可为空", null);
        }
        List<String> phones = JSONObject.parseObject(userPhones, new TypeReference<List<String>>() {
        });
        if (!CollectionUtils.isEmpty(phones)) {
            phones = phones.stream().filter(s -> StringUtils.isNotBlank(s)).collect(Collectors.toList());
            if (phones.size() <= MAX_SIZE) {
                List<User> users = userService.findByPhones(phones,tenantId);
                if (org.apache.commons.collections.CollectionUtils.isNotEmpty(users)) {
                    return JsonResult.buildSuccessResult(null,
                            users.stream().collect(Collectors.toMap(User::getPhoneNo, User::getId)));
                } else {
                    return JsonResult.buildSuccessResult(null, null);
                }
            }
            return JsonResult.buildErrorStateResult("单次批量查询不可超过500个手机号", null);
        }
        return JsonResult.buildErrorStateResult("传入用户手机号不可为空", null);
    }

    private static long MAX_SIZE = 500L;

    /**
     * 手机号批量查询uuid
     * todo 代码增加读写分离. 这里需要查从库
     *
     * @param userPhones
     * @return
     */
    @RequestMapping("/uuid/findByPhones")
    public JsonResult getUuidsByPhones(@RequestParam("userPhones") String userPhones,@RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isBlank(userPhones)) {
            return JsonResult.buildErrorStateResult("传入用户手机号不可为空", null);
        }
        List<String> phones = JSONObject.parseObject(userPhones, new TypeReference<List<String>>() {
        });
        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(phones)) {
            if (phones.size() <= MAX_SIZE) {
                List<UserInfo> userInfos = userService.findUserInfosByPhones(phones,tenantId);
                if (org.apache.commons.collections.CollectionUtils.isNotEmpty(userInfos)) {

                    Map<String, UserInfo> userInfoMap = Maps.newHashMapWithExpectedSize(userInfos.size());
                    userInfos.forEach(userInfo -> {
                        userInfoMap.put(userInfo.getPhoneNo(), userInfo);
                    });
                    return JsonResult.buildSuccessResult(null, userInfoMap);
                } else {
                    return JsonResult.buildSuccessResult(null, null);
                }
            }
            return JsonResult.buildErrorStateResult("uuid单次批量查询不可超过500个手机号", null);
        }
        return JsonResult.buildErrorStateResult("uuid批量传入用户手机号不可为空", null);
    }

    /**
     * 保存用户信息，地址信息，联系人信息
     *
     * @author v0.2 jinsong.zhu 2018年05月16日14:22:13
     * 保存用户信息，其中地址信息和联系人信息为非必要条件；注意：如果同时提供了province和address字段则需要保证Address的完整
     */
    @RequestMapping("/user/save_multi")
    public JsonResult saveMulti(String registeredFrom, String channelId, String phoneNo, String name,
                                String idNo, String provinceCode, String province, String cityCode, String city,
                                String districtCode, String district, String address, String contacts,
                                @RequestParam(defaultValue = "0", required = false) Long btRegisterChannelId
            , @RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (!NumberUtils.isDigits(registeredFrom)) {
            return JsonResult.buildErrorStateResult("注册渠道异常.", null);
        }
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult("手机号异常.", null);
        }
        if (StringUtils.isBlank(name)) {
            return JsonResult.buildErrorStateResult("用户名异常.", null);
        }

        //地址信息,同时存在provinceCode和address时才校验地址信息
        if (StringUtils.isNotEmpty(provinceCode) && StringUtils.isNotEmpty(address)) {
            if (!NumberUtils.isDigits(provinceCode)) {
                return JsonResult.buildErrorStateResult("省份编号异常.", null);
            }
            if (StringUtils.isBlank(province)) {
                return JsonResult.buildErrorStateResult("省份异常.", null);
            }
            if (!NumberUtils.isDigits(cityCode)) {
                return JsonResult.buildErrorStateResult("城市编号异常.", null);
            }
            if (StringUtils.isBlank(city)) {
                return JsonResult.buildErrorStateResult("城市异常.", null);
            }
            if (!NumberUtils.isDigits(districtCode)) {
                return JsonResult.buildErrorStateResult("区县编号异常.", null);
            }
            if (StringUtils.isBlank(district)) {
                return JsonResult.buildErrorStateResult("区县异常.", null);
            }
            if (StringUtils.isBlank(address)) {
                return JsonResult.buildErrorStateResult("详细地址异常.", null);
            }
        }
        channelId = MoreObjects.firstNonNull(channelId, "-1");
        // 验证用户是否已存在
        User user = userService.findByPhoneInDb(phoneNo,tenantId);
        if (null != user) {
            return JsonResult.buildErrorStateResult("用户已存在，手机号被占用", null);
        }

        user = userRegisterService.register(Long.valueOf(registeredFrom), Long.valueOf(channelId),
                phoneNo, name, contacts, btRegisterChannelId);
        UserRet userRet = new UserRet(user);
        return JsonResult.buildSuccessResult(null, userRet);
    }

    //根据日期时间段查询新注册用户信息并返回
    @RequestMapping("/contract/queryRegisterUsers")
    public JsonResult findRegisterUserByTime(String beginTime, String endTime, @RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (null == beginTime || endTime == null) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        List<User> userList = userService.findRegisterUserByTime(beginTime, endTime,tenantId);
        return JsonResult.buildSuccessResult(null, userList);
    }

    /**
     * 技术网关 - 验证手机号和验证码是否匹配 仅供可信任的内部服务调用，不执行限次记数、销毁等安全策略 注意：只使用于快速登录类似场景调用
     *
     * @param phoneNo          手机号
     * @param verificationCode 验证码（短信/语音）
     * @return
     */
    @RequestMapping("/verifyPhoneAndCode")
    public JsonResult verifyPhoneAndCode(
            @RequestParam String phoneNo, @RequestParam String verificationCode,
            @RequestParam String appChannel,
            @RequestParam(required = false, defaultValue = "1") Long channelId,
            @RequestParam(required = false, defaultValue = "1") Long createdFrom,
            @RequestParam(required = false, defaultValue = "xyqb") String key,
            @RequestParam(required = false) Long btRegisterChannelId,
            @RequestParam(required = false) String dimension , @RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {
        log.info(
                "/innerapi/verifyPhoneAndCode -> phoneNo:{},verificationCode:{},channelId:{},appChannel:{},createdFrom:{},btRegisterChannelId:{},key:{},dimension:{}",
                phoneNo, verificationCode, channelId, appChannel, createdFrom, btRegisterChannelId, key,
                dimension);
        // 验证接入方
        Merchant merchant = merchantService.findMerchantByName(key);
        if (merchant == null) {
            return JsonResult.buildErrorStateResult("未知的连接", null);
        }
        // 验证手机号
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            log.info("验证手机号和验证码是否匹配，手机号错误, phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("手机号错误", "");
        }
        // 验证短信验证码
        if (!smsService.verifyPhoneAndCode(phoneNo, verificationCode)) {
            // 是否需要重新获取
            if (smsService.needResendCode(phoneNo)) {
                return JsonResult.buildErrorStateResult("验证码失效，请重新获取", "");
            }
            log.info("验证码校验失败,phoneNo:{} , verificationCode:{}", phoneNo, verificationCode);
            return JsonResult.buildErrorStateResult("短信验证码错误", "");
        }
        User user = userService.findByPhoneWithCache(phoneNo,tenantId);
        // 检查用户有效性
        if (user != null && !user.getEnable()) {
            log.error("用户不存在，或者已经注销，phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult(USER_ERROR_OR_ENABLE_ERROR, null);
        }
        // 用户不存在时自动注册
        if (Objects.isNull(user)) {
            user = userRegisterService.register(phoneNo, channelId, createdFrom, appChannel,
                    btRegisterChannelId, dimension);
            // 注册失败
            if (Objects.isNull(user)) {
                return JsonResult.buildErrorStateResult("用户不存在", "");
            }
        }
        // 推送老的登陆统计信息
        UserStatistics statistics = new UserStatistics(user, dimension, 3, channelId);
        MqUtils.sendLoanVest(statistics);
        return JsonResult.buildSuccessResult("校验成功", new UserRet(user));
    }

    @RequestMapping("/login")
    public JsonResult login(@RequestParam String phoneNo, @RequestParam String password , @RequestHeader(value =Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        User user = checkPhoneNoAndPassword(phoneNo, password,tenantId);
        if (user == null) {
            return JsonResult.buildErrorStateResult("用户名或密码不正确", null);
        }
        UserRet model = new UserRet(user);
        return JsonResult.buildSuccessResult("校验成功", model);
    }

    private User checkPhoneNoAndPassword(String phoneNo, String password,Integer tenantId) {
        byte[] byPhoneNo = Base64.decodeBase64(phoneNo);
        String bufPhoneNo = "";
        try {
            bufPhoneNo = new String(byPhoneNo, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            log.error("不支持的编码: ", e);
        }
        if (!ValidationUtil.validatePhoneNo(bufPhoneNo)) {
            log.info("手机号错误, phoneNo:{}", bufPhoneNo);
            throw new UserNotExistException("手机号错误");
        }
        byte[] byPassword = Base64.decodeBase64(password);
        String bufPassword = "";
        try {
            bufPassword = new String(byPassword, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            log.error("不支持的编码: ", e);
        }
        User user = userService.findByPhoneWithCache(bufPhoneNo,tenantId);
        if (Objects.isNull(user) || Objects.equals(Boolean.FALSE, user.getEnable())) {
            return null;
        }

        // 有限校验新密码加密方式
        if (StringUtils.isNotBlank(user.getCipherPassword())) {
            if (!BctyptPasswordUtil.BCryptCheckPw(password, user.getCipherPassword())) {
                return null;
            }
        } else {
            //验证密码
            if (!PasswordUtil.validatePassword(bufPassword, user.getPassword(),user.getPasswordType())) {
                return null;
            }
            // 老密码校验后更新新密码加密方式
            user.setCipherPassword(BctyptPasswordUtil.BCryptWithSalt(password));
            userService.saveUser(user);
        }
        return user;
    }

    /**
     * 查询用户是否存在
     *
     * @param phoneNo
     * @return
     */
    @RequestMapping("/user/enable")
    public JsonResult isEnable(String phoneNo , @RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        boolean flag = false;
        Map validMap = Maps.newHashMap();
        validMap.put("valid", flag);
        User user = userService.findByPhoneInDb(phoneNo,tenantId);
        if (user != null && user.getEnable()) {
            validMap.put("valid", true);
            validMap.put("id", user.getId());
            validMap.put("uuid", user.getUuid());
            return JsonResult.buildSuccessResult("查询成功", validMap);
        }
        return JsonResult.buildErrorStateResult("用户不存在", validMap);
    }

    /**
     * 激活/启用 用户 -- 供内部系统免密调用
     *
     * @param userId
     * @return
     */
    @RequestMapping(path = "/user/active", method = RequestMethod.POST)
    public JsonResult activeUser(Long userId
            , @RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId,
                                 @RequestHeader(value = "Access-Token", required = false) String accessToken) {
        boolean flushed = flushUserStatus(userId, true,tenantId);
        UserDisableRecord userDisableRecord = new UserDisableRecord();
        userDisableRecord.setUserId(userId);
        if (StringUtils.isNotEmpty(accessToken)) {
            LoginUserContext loginUserContext = getLoginUserContextFromStms(accessToken);
            userDisableRecord.setCreatedBy(loginUserContext.getUserName());
            userDisableRecord.setUpdatedBy(loginUserContext.getUserName());
        }
        userDisableRecord.setOperation_type((short) 1);
        applicationEventPublisher.publishEvent(new UserDisableRecordEvent(this, userDisableRecord));
        log.info("激活/启用 用户,userId:[{}]，result:[{}]", userId, flushed);
        return JsonResult.buildSuccessResult("用户已激活.", flushed);
    }

    /**
     * 注销/禁用 用户 -- 供内部系统免密调用
     *
     * @param userId
     * @return
     */
    @RequestMapping(path = "/user/disable", method = RequestMethod.POST)
    public JsonResult forbiddenUser(Long userId,
                                    @Length(max = 10, message = "原因不能大于10个字符") String reason
            , @RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId,
                                    @RequestHeader(value = "Access-Token", required = false) String accessToken) {
        boolean flushed = flushUserStatus(userId, false,tenantId);
        UserDisableRecord userDisableRecord = new UserDisableRecord();
        userDisableRecord.setUserId(userId);
        if (StringUtils.isNotEmpty(accessToken)) {
            LoginUserContext loginUserContext = getLoginUserContextFromStms(accessToken);
            userDisableRecord.setCreatedBy(loginUserContext.getUserName());
            userDisableRecord.setUpdatedBy(loginUserContext.getUserName());
        }
        userDisableRecord.setReason(reason);
        userDisableRecord.setOperation_type((short) 0);
        applicationEventPublisher.publishEvent(new UserDisableRecordEvent(this, userDisableRecord));
        log.info("注销/禁用 用户,userId:[{}]，result:[{}]", userId, flushed);
        return JsonResult.buildSuccessResult("用户已禁用.", flushed);
    }

    /**
     * 根据用户id获取操作用户冻结记录列表
     *
     * @param userId 用户id
     * @return List<UserDisableRecord>
     */
    @RequestMapping(path = "/user/records/{userId}", method = RequestMethod.POST)
    public JsonResult disableRecordList(@PathVariable("userId") Long userId,
                                        @RequestParam(value = "page", defaultValue = "0") Integer page,
                                        @RequestParam(value = "size", defaultValue = "20") Integer size) {
        Pageable pageable = new PageRequest(page, size);
        return JsonResult.buildSuccessResultGeneric(
                iUserDisableRecordRepository.findByUserIdOrderByCreatedAtDesc(
                        userId, pageable).map(UserDisableRecordModel::from));
    }

    private LoginUserContext getLoginUserContextFromStms(String accessToken) {
        LoginUserContext loginUserContext = new LoginUserContext();
        OauthResult oauthResult = stmsRemoteService.checkToken(accessToken);
        if (oauthResult == null || !Objects.equals(2000,
                oauthResult.getCode())) {
            throw new UserNotExistException("stms用户异常，请检查token");
        } else {
            LinkedHashMap onlineUser = (LinkedHashMap) oauthResult.getData();
            loginUserContext.setUserId(onlineUser.get("id").toString());
            loginUserContext.setUserName(String.valueOf(onlineUser.get("account")));
        }
        return loginUserContext;
    }

    /**
     * 重置密码接口 -- 供内部系统免密调用
     */
    @RequestMapping(path = "/user/password/reset", method = RequestMethod.POST)
    public JsonResult resetPassword(@RequestParam("phone") String phone,
                                    @RequestParam(required = false) String password , @RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (ValidationUtil.validatePhoneNo(phone)) {
            // 默认重置的密码是123456
            if (StringUtils.isBlank(password)) {
                password = PasswordUtil.PASSWORD_DEFAULT;
            } else if (!ValidationUtil.validatePassword(password)) {
                return JsonResult.buildErrorStateResult(PasswordUtil.TOAST_MSG, null);
            }
            try {
                boolean result = userService.resetPassword(phone, password,tenantId);
                log.info("重置用户密码,phoneNo:[{}]，password:[{}],result:[{}]", phone, password, result);
                if (result) {
                    return JsonResult.buildSuccessResult("用户密码已重置.", password);
                }
            } catch (Exception e) {
                log.error("密码重置失败,phone[{}]", phone, e);
            }
        }
        return JsonResult.buildErrorStateResult("用户密码重置失败.", phone);
    }

    /**
     * 重置密码接口 -- 供内部人员使用（例如绝影）
     */
    @RequestMapping("/user/password/reset/{key}/{phone}")
    public JsonResult resetPasswordByKey(@PathVariable("key") String key,
                                         @PathVariable("phone") String phone , @RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        log.info("密码重置请求,phone:[{}]", phone);
        if (!RESET_PWD_TOKEN.equalsIgnoreCase(key)) {
            log.error("密码重置失败,key错误!@!,phone:[{}]", phone);
            return JsonResult.buildErrorStateResult("用户密码重置失败.", HttpStatus.SC_UNAUTHORIZED);
        }
        return resetPassword(phone, null,tenantId);
    }

    /**
     * 注销/禁用 用户 -- 供内部人员使用（例如绝影）
     *
     * @param phoneNo
     * @param enable
     * @param reason
     * @param content
     * @return
     */
    @RequestMapping("/forbiddenUserOrNot")
    public JsonResult forbiddenUserOrNot(@RequestParam String phoneNo, @RequestParam Boolean enable,
                                         @RequestParam String reason, @RequestParam String content , @RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isBlank(phoneNo) || StringUtils.isBlank(reason)) {
            return JsonResult.buildErrorStateResult("参数有误", null);
        }
        //密文
        if (null != content) {
            content = content.replaceAll(" ", "+");
        }
        String str = AesUtil.decryptAfterBase64Decode(content, Constants.AES_KEY);
        if (null == str || !str.equals(phoneNo)) {
            log.info("[forbiddenUserOrNot][禁用或激活用户]：解密有误, phoneNo:{},aes:{}", phoneNo,
                    AesUtil.encryptAndBase64Encode(phoneNo, Constants.AES_KEY));
            return JsonResult.buildErrorStateResult("解密有误", null);
        }
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            log.info("[forbiddenUserOrNot][禁用或激活用户]：用户手机号错误, phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("用户手机号错误", null);
        }
        int affectedRows = userService.forbiddenUser(enable, phoneNo,tenantId);
        // 清除缓存，解除微信关联
        User user = userService.findByPhoneInDb(phoneNo,tenantId);
        if (null != user && !user.getEnable()) {
            sessionService.deleteByUserId(user.getId(),tenantId);
            sessionService.deleteUserCatch(user,tenantId);
            // 禁用微信
            wechatService.forbiddenUserWeChat(user.getId(),tenantId);
        }
        log.info(
                "[forbiddenUserOrNot][禁用或激活用户]：理由：reason:{},手机号phoneNo:{},受影响的行数affectedRows:{}",
                reason, phoneNo, affectedRows);
        return JsonResult.buildSuccessResult("用户禁用或重启成功成功", affectedRows);
    }

    /**
     * 清除微信关联 -- 供内部人员使用（例如绝影）
     *
     * @param userId
     * @param reason
     * @param content
     * @return
     */
    @RequestMapping("/forbiddenUserWeChat")
    public JsonResult forbiddenUserWeChat(@RequestParam Long userId, @RequestParam String reason,
                                          @RequestParam String content,@RequestHeader(value =Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isBlank(reason) || null == userId || userId <= 0) {
            return JsonResult.buildErrorStateResult("参数有误", null);
        }
        //密文
        if (null != content) {
            content = content.replaceAll(" ", "+");
        }
        String str = AesUtil.decryptAfterBase64Decode(content, Constants.AES_KEY);
        if (null == str || !str.equals(userId.toString())) {
            return JsonResult.buildErrorStateResult("解密有误", null);
        }
        int affectedRows = wechatService.forbiddenUserWeChat(userId,tenantId);
        log.info("用户微信禁用,userId:{},禁用的原因reason:{},受影响的行数affectedRows:{}", userId,
                reason, affectedRows);
        return JsonResult.buildSuccessResult("用户禁用微信成功", affectedRows);
    }

    //    @RequestMapping("/clean")
    public JsonResult clean() {
        return JsonResult.buildSuccessResult("已经开始清洗");
    }

    @RequestMapping("/delete/session")
    public JsonResult deleteSession(Long userId , @RequestHeader(value = Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        sessionService.deleteByUserId(userId,tenantId);
        return JsonResult.buildSuccessResult();
    }

    /**
     * 注销用户
     *
     * @param userId 用户id
     * @return 销户结果
     */
    @GetMapping("/user/delete/{userId}")
    public JsonResult<?> deregister(@NotNull @PathVariable("userId") Long userId , @RequestHeader(value =Constants.X_AUTH_TENANT,defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {

        log.info("用户id[{}], 销户开始", userId);

        /* 执行销户 */
        userService.deregister(userId,tenantId);

        log.info("用户id[{}], 销户成功", userId);
        return JsonResult.buildSuccessResult("用户销户成功", true);
    }


    /**
     * 测试定时删除注销记录
     *
     * @return
     */
    @PostMapping("/user/deleteDeregister")
    public JsonResult<?> deleteDeregister() {
        userDeregisterService.executeTask();
        return JsonResult.buildSuccessResult("用户销户成功1231", true);
    }
}