package cn.quantgroup.xyqb.controller.external.user.center;

import cn.quantgroup.user.enums.EducationEnum;
import cn.quantgroup.user.enums.IncomeRangeEnum;
import cn.quantgroup.user.enums.MaritalStatus;
import cn.quantgroup.user.enums.OccupationEnum;
import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.limit.PasswordFreeAccessValidator;
import cn.quantgroup.xyqb.entity.*;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.service.user.*;
import cn.quantgroup.xyqb.util.EmojiUtil;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.sql.Timestamp;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Created by 11 on 2017/3/22.
 */
@RestController
@RequestMapping("/user/center")
public class UserCenterController {

    @Autowired
    private UserCenterService userCenterService;
    @Autowired
    private IUserDetailService userDetailService;
    @Autowired
    private IUserExtInfoService userExtInfoService;
    @Autowired
    private IContactService contactService;
    @Autowired
    private IAddressService addressService;
    @Autowired
    private IUserService userService;

    private static final Logger LOGGER = LoggerFactory.getLogger(UserCenterController.class);

    /**
     * 用户中心首页，显示用户头像、昵称、姓名
     *
     * @param phoneNo
     * @return
     */
    @PasswordFreeAccessValidator
    @RequestMapping("/index")
    public JsonResult userCenterIndex(String phoneNo) {
        Long userId = queryUserId(phoneNo);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        UserAttached userAttached = userCenterService.searchUserAttachedByUserId(userId);
        Map<String, String> result = new HashMap<>();
        if (userAttached != null) {
            result.put("avatar", userAttached.getAvatar());
            result.put("nick", userAttached.getNick());
        }
        UserDetail userDetail = userDetailService.findByUserId(userId);
        if (userDetail != null) {
            result.put("name", userDetail.getName());
            result.put("sex", Optional.ofNullable(userDetail.getGender()).orElse(cn.quantgroup.xyqb.model.Gender.UNKNOWN).ordinal() + "");
            result.put(Constants.PHONE_NO, userDetail.getPhoneNo().substring(0, 3) + "****" + userDetail.getPhoneNo().substring(7, 11));
        }
        return JsonResult.buildSuccessResult(null, result);
    }

    /**
     * 上传后调用该接口保存用户头像
     *
     * @param phoneNo   用户手机号
     * @param avatarUrl 上传后生成的头像地址
     * @return
     */
    @RequestMapping("/save/avatar")
    public JsonResult SaveUserAvatarAddr(String phoneNo, String avatarUrl) {
        if (StringUtils.isBlank(avatarUrl) || StringUtils.isBlank(phoneNo)) {
            LOGGER.error("参数不合法:avatarUrl:{}, phoneNo:{}", avatarUrl, phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        if (StringUtils.isBlank(avatarUrl)) {
            avatarUrl = Constants.UserAvatar.AVATAR_DEFAULT;
        }
        UserAttached userAttached = userCenterService.searchUserAttachedByUserId(userId);
        //查询到数据,直接更新头像和更新时间
        if (null != userAttached) {
            userAttached.setAvatar(avatarUrl);
            userAttached.setUpdatedAt(new Timestamp(System.currentTimeMillis()));
        } else {
            userAttached = createUserAttached(userId, avatarUrl, "");
        }
        UserAttached result = userCenterService.saveUserAttached(userAttached);
        if (null == userAttached) {
            LOGGER.error("保存用户头像昵称失败.");
            return JsonResult.buildErrorStateResult("信息保存失败,请稍后再试.", null);
        }
        return JsonResult.buildSuccessResult("保存成功", result);
    }

    /**
     * 根据用户手机号查询昵称.
     *
     * @param phoneNo
     * @return
     */
    @RequestMapping("/queryNick")
    public JsonResult queryUserNick(String phoneNo) {
        if (StringUtils.isEmpty(phoneNo)) {
            LOGGER.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        UserAttached userAttached = userCenterService.searchUserAttachedByUserId(userId);
        if (null != userAttached) {
            LOGGER.info("获取用户昵称:{}", userAttached.getNick());
            return JsonResult.buildSuccessResult(null, userAttached.getNick());
        }
        return JsonResult.buildSuccessResult(null, null);
    }

    /**
     * 修改用户的昵称
     *
     * @param phoneNo
     * @param nick
     * @return
     */
    @RequestMapping("/saveNick")
    public JsonResult saveUserNick(String phoneNo, String nick) {
        if (StringUtils.isEmpty(phoneNo)) {
            LOGGER.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        if (StringUtils.isNotBlank(nick)) {
            // 替换所有,UTF-8编码时4字节的Emoji表情字符
            nick = EmojiUtil.filterUnicode4(nick);
        }
        UserAttached userAttached = userCenterService.searchUserAttachedByUserId(userId);
        if (null == userAttached) {
            userAttached = createUserAttached(userId, "", nick);
        } else {
            userAttached.setNick(nick);
            userAttached.setUpdatedAt(new Timestamp(System.currentTimeMillis()));
        }
        UserAttached result = userCenterService.saveUserAttached(userAttached);
        return JsonResult.buildSuccessResult(null, result.getNick());
    }

    /**
     * 查询用户是否实名认证.
     *
     * @param phoneNo
     * @return
     */
    @RequestMapping("/query/verified")
    public JsonResult queryVerified(String phoneNo) {
        if (StringUtils.isEmpty(phoneNo)) {
            LOGGER.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        UserDetail userDetail = userDetailService.findByUserId(userId);
        if (null != userDetail) {
            return JsonResult.buildSuccessResult(null, userDetail);
        }
        return JsonResult.buildSuccessResult(null, null);
    }


    /**
     * 个人资料信息
     *
     * @param phoneNo
     * @return
     */
    @PasswordFreeAccessValidator
    @RequestMapping("/personalData")
    public JsonResult personalData(String phoneNo) {
        if (StringUtils.isBlank(phoneNo)) {
            LOGGER.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        UserExtInfo userExtInfo = userExtInfoService.findByUserId(userId);
        List<Contact> contacts = contactService.findByUserId(userId, true);
        Map<String, Object> result = new HashMap<>();
        if (null != userExtInfo) {
            result.put("contacts", null != contacts && contacts.size() > 0);
            result.put("occupation", userExtInfo.getOccupationEnum() == null ? "" : userExtInfo.getOccupationEnum().ordinal());
            result.put("education", userExtInfo.getEducationEnum() == null ? "" : userExtInfo.getEducationEnum().ordinal());
            result.put("income", userExtInfo.getIncomeRangeEnum() == null ? "" : userExtInfo.getIncomeRangeEnum().ordinal());
            result.put("marryStatus", userExtInfo.getMarryStatus() == null ? "" : userExtInfo.getMarryStatus().ordinal());
        }
        Address address = addressService.findByUserId(userId);
        if (null != address) {
            //用户住址信息,返回二级信息:
            result.put("address", address.getAddress());
            result.put("provinceCode", address.getProvinceCode());
            result.put("province", address.getProvince());
            result.put("cityCode", address.getCityCode());
            result.put("city", address.getCity());
            result.put("districtCode", address.getDistrictCode());
            result.put("district", address.getDistrict());
        }
        return JsonResult.buildSuccessResult(null, result);
    }

    /**
     * 保存用户联系人,
     *
     * @param phoneNo     用户手机号
     * @param contactJson 要保存的联系人json串
     * @return
     */
    @RequestMapping("/contacts/save")
    public JsonResult saveUserContact(String phoneNo, String contactJson) {
        if (StringUtils.isEmpty(phoneNo)) {
            LOGGER.error("保存联系人,参数错误. phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数有误.", null);
        }
        Long userId = queryUserId(phoneNo);
        if (null == userId || userId == 0L) {
            LOGGER.error("保存联系人,未获取到用户id. userId:{}", userId);
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        if (StringUtils.isEmpty(contactJson)) {
            LOGGER.error("保存联系人,联系人参数为空.contactJson:{}", contactJson);
            return JsonResult.buildErrorStateResult(null, null);
        }
        List<Contact> contacts = JSONObject.parseObject(contactJson, new TypeReference<List<Contact>>() {
        });
        if (CollectionUtils.isEmpty(contacts)) {
            LOGGER.info("联系人不能空");
            return JsonResult.buildErrorStateResult(null, null);
        }
        for (Contact contact : contacts) {
            if (!contact.valid()) {
                LOGGER.info("用户手机号或姓名错误, phoneNo:{},name:{}", contact.getPhoneNo(), contact.getName());
                return JsonResult.buildErrorStateResult(null, null);
            }
        }
        List<Contact> result = contactService.save(userId, contacts);
        return JsonResult.buildSuccessResult(null, result);
    }

    /**
     * 保存用户住址信息
     *
     * @param province
     * @param provinceCode
     * @param city
     * @param cityCode
     * @param address
     * @return
     */
    @RequestMapping("/address/save")
    public JsonResult saveUserAddress(String phoneNo, String province, Long provinceCode, String city, Long cityCode, String address,
                                      Long districtCode, String district) {
        LOGGER.info("保存用户地址信息, phoneNo:{}, province:{}, provinceCode:{}, city:{}, cityCode:{},district:{}, address:{}", phoneNo, province, provinceCode, city, cityCode, district, address);
        if (StringUtils.isEmpty(phoneNo) || provinceCode == null || cityCode == null || StringUtils.isBlank(address)) {
            return JsonResult.buildErrorStateResult("参数错误", null);
        }
        Long userId = queryUserId(phoneNo);
        if (null == userId || userId == 0L) {
            LOGGER.error("保存联系人,未获取到用户id. userId:{}", userId);
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        //查询用户是否有保存地址信息
        Address addressInfo = addressService.findByUserId(userId);
        if (null == addressInfo) {
            Timestamp now = new Timestamp(System.currentTimeMillis());
            addressInfo = new Address();
            addressInfo.setCreatedAt(now);
            addressInfo.setUpdateAt(now);
        }
        addressInfo.setUserId(userId);
        addressInfo.setProvince(province);
        addressInfo.setProvinceCode(provinceCode);
        addressInfo.setCity(city);
        addressInfo.setCityCode(cityCode);
        addressInfo.setDistrictCode(districtCode);
        addressInfo.setDistrict(district);
        addressInfo.setAddress(address);
        if (StringUtils.isNotBlank(address)) {
            // 替换所有,UTF-8编码时4字节的Emoji表情字符
            addressInfo.setAddress(EmojiUtil.filterUnicode4(address));
        }
        //更新时间
        addressInfo.setUpdateAt(new Timestamp(System.currentTimeMillis()));
        addressService.save(addressInfo);
        return JsonResult.buildSuccessResult(null, addressInfo);
    }

    /**
     * 查询用户住址信息
     *
     * @param phoneNo
     * @return
     */
    @RequestMapping("/searchAddress/phoneNo")
    public JsonResult searchUserAddress(String phoneNo) {
        if (StringUtils.isEmpty(phoneNo)) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        Long userId = queryUserId(phoneNo);
        if (null == userId || userId == 0L) {
            LOGGER.error("保存联系人,未获取到用户id. userId:{}", userId);
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        Address address = addressService.findByUserId(userId);
        return JsonResult.buildSuccessResult(null, address);
    }

    /**
     * 查询用户联系人
     *
     * @param phoneNo
     * @return
     */
    @RequestMapping("/searchContacts/phoneNo")
    public JsonResult searchUserContacts(String phoneNo) {
        if (StringUtils.isEmpty(phoneNo)) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        Long userId = queryUserId(phoneNo);
        if (null == userId || userId == 0L) {
            LOGGER.error("保存联系人,未获取到用户id. userId:{}", userId);
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        List<Contact> contacts = contactService.findByUserId(userId, true);
        return JsonResult.buildSuccessResult(null, contacts);
    }

    /**
     * 保存用户经济信息
     *
     * @param educationEnum   学历
     * @param occupationEnum  职业
     * @param incomeRangeEnum 月收入范围
     * @param maritalStatus   婚姻状态
     * @return
     */
    @RequestMapping("/save/userExtInfo")
    public JsonResult saveUserExtInfo(String phoneNo, EducationEnum educationEnum, MaritalStatus maritalStatus, IncomeRangeEnum incomeRangeEnum, OccupationEnum occupationEnum) {
        if (StringUtils.isEmpty(phoneNo)) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        Long userId = queryUserId(phoneNo);
        if (null == userId || userId == 0L) {
            LOGGER.error("保存用户经济信息,未获取到用户id. userId:{}", userId);
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        //查询用户是否保存过.
        UserExtInfo userExtInfo = userExtInfoService.findByUserId(userId);
        Timestamp now = new Timestamp(System.currentTimeMillis());
        if (null == userExtInfo) {
            userExtInfo = new UserExtInfo();
            userExtInfo.setUserId(userId);
        }
        userExtInfo.setCreatedAt(now);
        userExtInfo.setUpdateAt(now);
        userExtInfo.setEducationEnum(educationEnum);
        userExtInfo.setMarryStatus(maritalStatus);
        userExtInfo.setOccupationEnum(occupationEnum);
        userExtInfo.setIncomeRangeEnum(incomeRangeEnum);

        userExtInfo = userExtInfoService.save(userExtInfo);
        return JsonResult.buildSuccessResult(null, userExtInfo);
    }

    /**
     * 创建用户附加信息实体
     *
     * @param userId
     * @param avatar
     * @param nick
     * @return
     */
    private UserAttached createUserAttached(Long userId, String avatar, String nick) {
        UserAttached userAttached = new UserAttached();
        Timestamp now = new Timestamp(System.currentTimeMillis());
        userAttached.setUserId(userId);
        userAttached.setAvatar(avatar);
        userAttached.setNick(nick);
        userAttached.setCreatedAt(now);
        userAttached.setUpdatedAt(now);
        return userAttached;
    }

    /**
     * 根据手机号查询用户id, nodejs调用接口无法获取到userId.
     * 所以增加该接口来查询用户id
     *
     * @param phoneNo
     * @return
     */
    private Long queryUserId(String phoneNo) {
        User user = userService.findByPhoneInDb(phoneNo);
        return null == user ? null : user.getId();
    }

}
