package cn.quantgroup.xyqb.controller.external.user;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.accessable.IpValidator;
import cn.quantgroup.xyqb.aspect.logcaller.LogHttpCaller;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.service.api.IUserApiService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.session.XyqbSessionContextHolder;
import cn.quantgroup.xyqb.util.ValidationUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.Objects;

/**
 * Created by FrankChow on 15/12/16.
 */
@Slf4j
@RestController
@RequestMapping("/api")
public class UserApiController {

  @Resource
  private IUserService userService;
  @Resource
  private IUserApiService userApiService;
  @Resource
  private ISessionService sessionService;
  @Resource
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> stringRedisTemplate;

  @RequestMapping("/user/check")
  public JsonResult userImportCheck(String phoneNo, String registerFrom) {
    if ("244".equals(registerFrom)) {
      log.info("[user_import_check]用户导入检查拒绝。phoneNo=[{}], registerFrom=[{}]", phoneNo, registerFrom);
      return JsonResult.buildErrorStateResult("用户导入检查拒绝", false);
    }
    if (StringUtils.isEmpty(phoneNo) || StringUtils.isEmpty(registerFrom)) {
      log.error("[user_import_check]检查传入的参数，参数不全。phoneNo=[{}], registerFrom=[{}]", phoneNo, registerFrom);
      return JsonResult.buildErrorStateResult("检查传入的参数，参数不全。", null);
    }
    boolean checkPassed = userApiService.userImportCheck(phoneNo);
    if (checkPassed) {
      log.info("[user_import_check]用户可以导入。phoneNo=[{}], registerFrom=[{}]", phoneNo, registerFrom);
      return JsonResult.buildSuccessResult("用户可以导入", checkPassed);
    }
    log.info("[user_import_check]用户导入检查拒绝。phoneNo=[{}], registerFrom=[{}]", phoneNo, registerFrom);
    return JsonResult.buildErrorStateResult("用户导入检查拒绝", checkPassed);
  }

  @RequestMapping("/user/is_passwd_set")
  public JsonResult isPasswordSet(String key, String phoneNo) {
    if (!"abc1234".equals(key) || StringUtils.isEmpty(phoneNo)) {
      return JsonResult.buildErrorStateResult(null, null);
    }
    User user = userService.findByPhoneWithCache(phoneNo);
    if (null == user) {
      return JsonResult.buildErrorStateResult(null, null);
    }
    if (StringUtils.length(user.getPassword()) == 32) {
      return JsonResult.buildSuccessResult(null, null);
    }
    return JsonResult.buildErrorStateResult(null, null, 2L);
  }

  /**
   * 检查token是否有效
   * 如果有效，可选择是否延续生命期（延续后有效期24Hour）
   *
   * @param token - sid,session的id
   * @param prolong - 是否延续生命期，可选参数，默认为: false - 不延续
   * @return
   */
  @LogHttpCaller
  @IpValidator
  @RequestMapping(value = "/valid/{token}", method = RequestMethod.POST)
  public JsonResult checkToken(@PathVariable("token") String token, @RequestParam(name = "prolong", required = false, defaultValue = "false") Boolean prolong) {
    if(Objects.isNull(token) || !ValidationUtil.validateToken(token)){
      return JsonResult.buildErrorStateResult("token invalid", token);
    }
    String tokenKey = Constants.SESSION_PREFIX + token;
    String tokenKey2 = Constants.Session.USER_SESSION_CACHE + token;
    // 判断token是否存在
    boolean exist = stringRedisTemplate.hasKey(tokenKey)||stringRedisTemplate.hasKey(tokenKey2);
    /* token存在且需要延续时，进一步判断session是否有效，有效时，自动续期 */
    if(Boolean.logicalAnd(exist, prolong)){
      // 获取session信息
      SessionStruct sessionStruct = XyqbSessionContextHolder.getXSessionFromRedis(token);
      if(Objects.isNull(sessionStruct)) {
        /* 如果没有获取到session信息则返回错误信息 */
        return JsonResult.buildErrorStateResult("session invalid", token);
      }else{
        /* 延续session生命期 */
        try {
          sessionService.persistSession(sessionStruct.getSid(), sessionStruct.getValues());
        } finally {
          XyqbSessionContextHolder.releaseSession();
        }
      }
    }
    return JsonResult.buildSuccessResult("token valid", token);
  }

}
