package cn.quantgroup.xyqb.controller;

import cn.quantgroup.xyqb.exception.*;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.util.IpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.jdbc.BadSqlGrammarException;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import javax.validation.ValidationException;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Created by Miraculous on 15/7/6.
 * 出现异常，进入这个handler。
 */
@Slf4j
@ControllerAdvice
@RestController
public class ExceptionHandlingController implements IBaseController {

    private static final JsonResult EXCEPTION_RESULT = new JsonResult("服务出错。", (long) HttpStatus.INTERNAL_SERVER_ERROR.value(), null);

    /**
     * 密码错误次数达到上限异常
     *
     * @param vce
     * @return
     */
    @ExceptionHandler(PasswordErrorLimitException.class)
    public JsonResult passwordErrorLimitException(PasswordErrorLimitException vce) {
        log.info("throw PasswordErrorLimitException,msg={},businessCode={}, clientIp={}", vce.getMessage(), 2L, IpUtil.getRemoteIP(getRequest()));
        JsonResult<Object> objectJsonResult = JsonResult.buildErrorStateResult(vce.getMessage(), null, 2L);
        log.info("PasswordErrorLimitException异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),vce);
        return objectJsonResult;
    }

    /**
     * 短信验证码错误或失效异常
     *
     * @param vce
     * @return
     */
    @ExceptionHandler(VerificationCodeErrorException.class)
    public JsonResult verificationCodeErrorException(VerificationCodeErrorException vce) {
        log.info("throw VerificationCodeErrorException,msg={},businessCode={}, clientIp={}", vce.getMessage(), 1L, IpUtil.getRemoteIP(getRequest()));
        JsonResult<Object> objectJsonResult = JsonResult.buildErrorStateResult(vce.getMessage(), null, 1L);
        log.info("VerificationCodeErrorException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),vce);
        return objectJsonResult;
    }

    /**
     * 用户不存在异常
     *
     * @param unee
     * @return
     */
    @ExceptionHandler(UserNotExistException.class)
    @ResponseStatus(HttpStatus.UNAUTHORIZED)
    public JsonResult userNotExistException(UserNotExistException unee) {
        log.info("throw UserNotExistException,msg={},businessCode={},code={}", unee.getMessage(), 1L, (long) HttpStatus.UNAUTHORIZED.value());
        JsonResult<Object> objectJsonResult = new JsonResult(unee.getMessage(), (long) HttpStatus.UNAUTHORIZED.value(), null);
        log.info("userNotExistException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),unee);
        return objectJsonResult;
    }

    /**
     * 用户查询或导出异常
     *
     * @param uqle
     * @return
     */
    @ExceptionHandler(UserQueryLogException.class)
    public JsonResult userQueryLogException(UserQueryLogException uqle) {
        log.info("throw UserQueryLogException,msg={},businessCode={},code={}", uqle.getMessage(), 1L, 0L);
        JsonResult<Object> objectJsonResult = JsonResult.buildErrorStateResult(uqle.getMessage(), null);
        log.info("userQueryLogException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),uqle);
        return objectJsonResult;
    }

    /**
     * 微信关联异常
     *
     * @param wrue
     * @return
     */
    @ExceptionHandler(WechatRelateUserException.class)
    @ResponseStatus(HttpStatus.UNAUTHORIZED)
    public JsonResult wechatRelateUserException(WechatRelateUserException wrue) {
        log.info("throw WechatRelateUserException,msg={},businessCode={},code={}", wrue.getMessage(), 1L, (long) HttpStatus.UNAUTHORIZED.value());
        JsonResult<Object> objectJsonResult =  new JsonResult(wrue.getMessage(), (long) HttpStatus.UNAUTHORIZED.value(), null);
        log.info("wechatRelateUserException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),wrue);
        return objectJsonResult;
    }

    /**
     * 其他全局异常
     *
     * @param e
     * @return
     */
    @ExceptionHandler(Exception.class)
    public JsonResult exceptionOccurs(Exception e) {
        HttpServletRequest request = getRequest();
        String uri = request.getRequestURI();
        String registerFrom = request.getParameter("registerFrom");
        log.error("[exception][global_exception]接口异常 URI:{}, registerFrom:{},error={}", uri, registerFrom, e);
        JsonResult<Object> objectJsonResult =   EXCEPTION_RESULT;
        log.info("exceptionOccurs 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),e);
        return objectJsonResult;
    }

    @ExceptionHandler(ValidationException.class)
    public JsonResult validException(ValidationException exception) {
        ConstraintViolationException exs = (ConstraintViolationException) exception;
        Set<ConstraintViolation<?>> constraintViolations = exs.getConstraintViolations();
        if (constraintViolations.isEmpty()) {
            log.error("这里出了个错....", exception.getMessage());
            return null;
        }
        String message = constraintViolations.iterator().next().getMessage();
        JsonResult<Object> objectJsonResult =   JsonResult.buildErrorStateResult(message, null);
        log.info("validException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),exception);
        return objectJsonResult;
    }

    @ExceptionHandler({MethodArgumentNotValidException.class, BindException.class})
    @ResponseBody
    @ResponseStatus(HttpStatus.OK)
    public JsonResult handleBindException(Exception e) {
        BindingResult result = null;
        if (e instanceof MethodArgumentNotValidException) {
            result = ((MethodArgumentNotValidException) e).getBindingResult();
        } else if (e instanceof BindException) {
            result = ((BindException) e).getBindingResult();
        }
        if (result != null) {
            List<ObjectError> errors = result.getAllErrors();
            StringBuilder sb = new StringBuilder();
            if (errors != null && errors.size() > 0) {
                Iterator<ObjectError> iterator = errors.iterator();
                ObjectError err = null;
                while ((err = iterator.next()) != null) {
                    sb.append(err.getDefaultMessage());
                    if (iterator.hasNext()) {
                        sb.append("; ");
                    } else {
                        sb.append("。");
                        break;
                    }
                }
            }
            JsonResult<Object> objectJsonResult =   JsonResult.buildErrorStateResult(sb.toString(), null);
            log.info("handleBindException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),e);
            return objectJsonResult;
        }
        return null;
    }

    @ExceptionHandler({DataException.class})
    @ResponseBody
    @ResponseStatus(HttpStatus.OK)
    public JsonResult handleDataException(DataException e) {
        JsonResult<Object> objectJsonResult =   JsonResult.buildErrorStateResult(e.getMessage(), null);
        log.info("handleDataException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),e);
        return objectJsonResult;
    }
    @ExceptionHandler({AppletException.class})
    @ResponseBody
    @ResponseStatus(HttpStatus.OK)
    public JsonResult handleAppletException(AppletException e) {
        JsonResult<Object> objectJsonResult =   JsonResult.buildErrorResultWhithCode(e.getMessage(), e.getCode());
        log.info("handleAppletException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),e);
        return objectJsonResult;
    }

    @ExceptionHandler(BadSqlGrammarException.class)
    @ResponseBody
    @ResponseStatus(HttpStatus.OK)
    public JsonResult handleBadSqlGrammarException(BadSqlGrammarException e) {
        log.error("sql语法解析异常 error sql = 【{}】", e.getSql(), e);
        JsonResult<Object> objectJsonResult =   JsonResult.buildErrorStateResult("参数错误。", null);
        log.info("handleBadSqlGrammarException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),e);
        return objectJsonResult;
    }

    @ExceptionHandler(ResubmissionException.class)
    @ResponseStatus(HttpStatus.OK)
    @ResponseBody
    public JsonResult resubmissionException(ResubmissionException re) {
        JsonResult<Object> objectJsonResult =   JsonResult.buildErrorStateResult(re.getMessage(), null);
        log.info("resubmissionException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),re);
        return objectJsonResult;
    }

    @ExceptionHandler(MissingServletRequestParameterException.class)
    @ResponseStatus(HttpStatus.OK)
    @ResponseBody
    public JsonResult handelMissingServletRequestParameterException(MissingServletRequestParameterException re) {
        JsonResult<Object> objectJsonResult = JsonResult.buildErrorStateResult(re.getMessage(), null);
        log.info("handelMissingServletRequestParameterException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),re);
        return objectJsonResult;
    }

    /**
     * 用户注册登录异常处理
     * @param e 异常
     * @return JsonResult<String>
     */
    @ExceptionHandler(UserRegisterLoginException.class)
    public JsonResult<?> handleUserDeregisterException(UserRegisterLoginException e) {
        JsonResult<Object> objectJsonResult =   JsonResult.buildErrorStateResult(e.getMessage(), null);
        log.info("handleUserDeregisterException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),e);
        return objectJsonResult;
    }

    @ExceptionHandler(AccessForbiddenException.class)
    public JsonResult<?> handleAccessForbiddenException(AccessForbiddenException e) {
        JsonResult<Object> objectJsonResult =   JsonResult.buildErrorStateResult(e.getMessage(), null);
        log.info("handleAccessForbiddenException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),e);
        return objectJsonResult;
    }

    @ExceptionHandler(BizException.class)
    public JsonResult<?> handleBizException(BizException e) {
        JsonResult<Object> objectJsonResult =   JsonResult.buildErrorResultWhithCode(e.getMsg(),e.getBusinessCode());
        log.info("handleBizException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),e);
        return objectJsonResult;
    }
    @ExceptionHandler(SilentBizException.class)
    public JsonResult<?> handleSilentBizException(SilentBizException e) {
        JsonResult<Object> objectJsonResult =   JsonResult.buildErrorResultWhithCode(e.getMsg(),e.getBusinessCode());
        log.info("handleSilentBizException 异常响应结果：{}", JSONObject.toJSONString(objectJsonResult),e);
        return objectJsonResult;
    }
}
