package cn.quantgroup.xyqb.config.http;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.SerializationFeature;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.ConnectionKeepAliveStrategy;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.AllowAllHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContextBuilder;
import org.apache.http.cookie.Cookie;
import org.apache.http.cookie.CookieOrigin;
import org.apache.http.cookie.CookieSpecProvider;
import org.apache.http.cookie.MalformedCookieException;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.client.StandardHttpRequestRetryHandler;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.impl.cookie.BestMatchSpecFactory;
import org.apache.http.impl.cookie.BrowserCompatSpec;
import org.apache.http.impl.cookie.BrowserCompatSpecFactory;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.convert.converter.ConverterRegistry;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.web.filter.CharacterEncodingFilter;

import javax.net.ssl.SSLContext;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;

/**
 * Created by Miraculous on 15/7/12.
 */
@Configuration
public class HttpConfig {

  @Bean
  public FilterRegistrationBean filterRegistrationBean() {
    FilterRegistrationBean registrationBean = new FilterRegistrationBean();
    CharacterEncodingFilter characterEncodingFilter = new CharacterEncodingFilter();
    registrationBean.setFilter(characterEncodingFilter);
    characterEncodingFilter.setEncoding("UTF-8");
    characterEncodingFilter.setForceEncoding(true);
    registrationBean.setOrder(Integer.MIN_VALUE);
    registrationBean.addUrlPatterns("/*");
    return registrationBean;
  }

  /**
   * 该方法主要对对象json序列化产生影响，其功能如下：
   * 1. 禁用缩进输出
   * 2. 日期格式排版
   * 3. null被过滤掉
   * 4. 将enum转为其ordinal
   *
   * @return Jackson2ObjectMapperBuilder
   */
  @Bean
  public Jackson2ObjectMapperBuilder jacksonBuilder() {
    Jackson2ObjectMapperBuilder builder = new Jackson2ObjectMapperBuilder();
    builder.indentOutput(false)
        .dateFormat(new SimpleDateFormat("yyyy-MM-dd"))
        .defaultViewInclusion(false)
        .serializationInclusion(JsonInclude.Include.NON_NULL)
        .featuresToEnable(SerializationFeature.WRITE_ENUMS_USING_INDEX,
            DeserializationFeature.FAIL_ON_NUMBERS_FOR_ENUMS);
    return builder;
  }

  // hack spring mvc.
  @Bean
  public IntegerToEnumConverterFactory getIntegerToEnumConverterFactory(
      ConverterRegistry defaultConversionService, ConverterRegistry mvcConversionService, ConverterRegistry integrationConversionService) {
    IntegerToEnumConverterFactory factory = new IntegerToEnumConverterFactory();

    defaultConversionService.removeConvertible(String.class, Enum.class);
    mvcConversionService.removeConvertible(String.class, Enum.class);
    integrationConversionService.removeConvertible(String.class, Enum.class);

    defaultConversionService.addConverterFactory(factory);
    mvcConversionService.addConverterFactory(factory);
    integrationConversionService.addConverterFactory(factory);
    return factory;
  }

  @Bean(name = "httpClient")
  public CloseableHttpClient httpClient() throws KeyStoreException, NoSuchAlgorithmException, KeyManagementException {
    //  socket factory
    ConnectionSocketFactory plainSocketFactory = new PlainConnectionSocketFactory();
    SSLContext sslContext = new SSLContextBuilder().loadTrustMaterial(
        null, (x509Certificates, authType) -> true).build();

    LayeredConnectionSocketFactory sslSocketFactory =
        new SSLConnectionSocketFactory(sslContext, new AllowAllHostnameVerifier());
    Registry<ConnectionSocketFactory> socketFactoryRegistry = RegistryBuilder.<ConnectionSocketFactory>create()
        .register("http", plainSocketFactory)
        .register("https", sslSocketFactory).build();

    // cookie specification
    Registry<CookieSpecProvider> cookieSpecProviderRegistry = RegistryBuilder.<CookieSpecProvider>create()
        .register(CookieSpecs.BEST_MATCH, new BestMatchSpecFactory())
        .register(CookieSpecs.BROWSER_COMPATIBILITY, new BrowserCompatSpecFactory())
        .register("easy", httpContext -> new BrowserCompatSpec() {
          @Override
          public void validate(Cookie cookie, CookieOrigin origin) throws MalformedCookieException {
          }
        }).build();

    // connection manager
    PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager(socketFactoryRegistry);

    connectionManager.setMaxTotal(20000);
    connectionManager.setDefaultMaxPerRoute(2000);

    // retry handler
    HttpRequestRetryHandler retryHandler = new StandardHttpRequestRetryHandler(3, false);

    // keep alive strategy
    ConnectionKeepAliveStrategy keepAliveStrategy = new DefaultConnectionKeepAliveStrategy();

    // httpclient
    return HttpClients.custom()
        .setConnectionManager(connectionManager)
        .setRetryHandler(retryHandler)
        .setKeepAliveStrategy(keepAliveStrategy)
        .setDefaultCookieSpecRegistry(cookieSpecProviderRegistry).build();
  }
}
