package cn.quantgroup.xyqb.controller.api.v2;

import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.controller.req.v2.BatchInfoReq;
import cn.quantgroup.xyqb.controller.req.v2.UserInfoReq;
import cn.quantgroup.xyqb.entity.BaseEntity;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserInfoEntity;
import cn.quantgroup.xyqb.entity.WechatUserInfo;
import cn.quantgroup.xyqb.exception.BizException;
import cn.quantgroup.xyqb.exception.BizExceptionEnum;
import cn.quantgroup.xyqb.exception.SilentBizException;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.UserBean;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.repository.IUserInfoRepository;
import cn.quantgroup.xyqb.repository.IUserRepository;
import cn.quantgroup.xyqb.service.wechat.IWechatService;
import cn.quantgroup.xyqb.session.XyqbSessionContextHolder;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.stream.Collectors;

@Slf4j
@RestController
@RequestMapping("/api/v2/user")
public class UserApiV2Controller implements IBaseController {

    private final IUserRepository userRepository;
    private final IWechatService wechatService;
    private final IUserInfoRepository userInfoRepository;

    public UserApiV2Controller(IUserRepository userRepository, IWechatService wechatService,IUserInfoRepository userInfoRepository) {
        this.userRepository = userRepository;
        this.wechatService = wechatService;
        this.userInfoRepository = userInfoRepository;
    }

    /**
     * 查询用户信息
     *
     * @param userInfoReq 查询条件
     * @return JsonResult<User>
     * @see <a href="http://yapi.quantgroups.com/project/17/interface/api/65724">查询用户信息</a>
     */
    @PostMapping("info")
    public JsonResult<UserBean> info(@RequestBody UserInfoReq userInfoReq) {
        //SessionStruct sessionStruct = XyqbSessionContextHolder.getXSession();
        Integer tenantId = getTenantId();
        User user = null;
        List<WechatUserInfo> wechatUserInfo = Lists.newArrayList();
        //1、校验
        if (userInfoReq.getUserId() == null && StringUtils.isEmpty(userInfoReq.getUuid())
                && StringUtils.isEmpty(userInfoReq.getPhoneNo()) && (StringUtils.isEmpty(userInfoReq.getAppId()) && StringUtils.isEmpty(userInfoReq.getOpenId()))
                && (StringUtils.isEmpty(userInfoReq.getAppId()) && StringUtils.isEmpty(userInfoReq.getUnionId()))) {
            throw new BizException(BizExceptionEnum.ERROR_PARAM);
        }

        //2、查询
        if (userInfoReq.getUserId() != null) {
            user = userRepository.findByIdAndTenantId(userInfoReq.getUserId(), tenantId);
            if (StringUtils.isNotEmpty(userInfoReq.getAppId())) {
                wechatUserInfo = wechatService.queryByUserIdAndAppId(userInfoReq.getUserId(), userInfoReq.getAppId(), tenantId);
            }
        }

        if (StringUtils.isNotEmpty(userInfoReq.getUuid())) {
            user = userRepository.findByUuidAndTenantId(userInfoReq.getUuid(), tenantId);
            if (user != null && StringUtils.isNotEmpty(userInfoReq.getAppId())) {
                wechatUserInfo = wechatService.queryByUserIdAndAppId(user.getId(), userInfoReq.getAppId(), tenantId);
            }
        }

        if (StringUtils.isNotEmpty(userInfoReq.getPhoneNo())) {
            user = userRepository.findByPhoneNoAndTenantId(userInfoReq.getPhoneNo(), tenantId);
            if (user != null && StringUtils.isNotEmpty(userInfoReq.getAppId())) {
                wechatUserInfo = wechatService.queryByUserIdAndAppId(user.getId(), userInfoReq.getAppId(), tenantId);
            }
        }


        if (StringUtils.isNotEmpty(userInfoReq.getAppId()) && StringUtils.isNotEmpty(userInfoReq.getOpenId())) {
            WechatUserInfo wechatUserInfoFromDb = wechatService.findWechatUserInfoFromDb(userInfoReq.getOpenId(), userInfoReq.getAppId(), tenantId);
            if (wechatUserInfo == null || wechatUserInfoFromDb == null ||  wechatUserInfoFromDb.getUserId() == null) {
                //throw new BizException(BizExceptionEnum.UN_EXIT_USER);
                return JsonResult.buildErrorResultWhithCode(BizExceptionEnum.UN_EXIT_USER.getMsg(),BizExceptionEnum.UN_EXIT_USER.getBusinessCode());
            }
            wechatUserInfo.add(wechatUserInfoFromDb);
            user = userRepository.findByIdAndTenantId(wechatUserInfoFromDb.getUserId(), tenantId);
        }

        if (StringUtils.isNotEmpty(userInfoReq.getAppId()) && StringUtils.isNotEmpty(userInfoReq.getUnionId())) {
            WechatUserInfo wechatUserInfoFromDb = wechatService.findByUnionIdAndAppIdAndTenantId(userInfoReq.getUnionId(), userInfoReq.getAppId(), tenantId);
            if (wechatUserInfo == null || wechatUserInfoFromDb.getUserId() == null) {
                //throw new BizException(BizExceptionEnum.UN_EXIT_USER);
                return JsonResult.buildErrorResultWhithCode(BizExceptionEnum.UN_EXIT_USER.getMsg(),BizExceptionEnum.UN_EXIT_USER.getBusinessCode());
            }
            wechatUserInfo.add(wechatUserInfoFromDb);
            user = userRepository.findByIdAndTenantId(wechatUserInfoFromDb.getUserId(), tenantId);
        }

        //3、异常处理
        if (user == null) {
            //throw new BizException(BizExceptionEnum.UN_EXIT_USER);
            return JsonResult.buildErrorResultWhithCode(BizExceptionEnum.UN_EXIT_USER.getMsg(),BizExceptionEnum.UN_EXIT_USER.getBusinessCode());
        }
        UserInfoEntity userInfoEntity = userInfoRepository.findByUserIdAndTenantId(user.getId(), tenantId);
        return JsonResult.buildSuccessResultGeneric(UserBean.from(user, wechatUserInfo,userInfoEntity));
    }

    /**
     * 批量查询用户信息
     *
     * @param batchInfoReq 批量查询条件
     * @return JsonResult<List < User>>
     * @see <a href="http://yapi.quantgroups.com/project/17/interface/api/65734">批量查询用户信息</a>
     */
    @PostMapping("/batchInfo")
    public JsonResult<List<UserBean>> batchInfo(@RequestBody BatchInfoReq batchInfoReq) {
        //SessionStruct sessionStruct = XyqbSessionContextHolder.getXSession();
        Integer tenantId = getTenantId();
        List<User> userList = null;
        List<WechatUserInfo> wechatUserInfo = null;
        //1、校验
        if (CollectionUtils.isEmpty(batchInfoReq.getUserIds())
                && CollectionUtils.isEmpty(batchInfoReq.getUuids())
                && CollectionUtils.isEmpty(batchInfoReq.getPhoneNos())
                && (StringUtils.isEmpty(batchInfoReq.getAppId()) && CollectionUtils.isEmpty(batchInfoReq.getOpenIds()))
                && (StringUtils.isEmpty(batchInfoReq.getAppId()) && CollectionUtils.isEmpty(batchInfoReq.getUnionIds()))) {
            throw new BizException(BizExceptionEnum.ERROR_PARAM);
        }

        //2、查询
        if (CollectionUtils.isNotEmpty(batchInfoReq.getUserIds())) {
            userList = userRepository.findByIdInAndTenantId(batchInfoReq.getUserIds(), tenantId);
            if (CollectionUtils.isNotEmpty(userList) && StringUtils.isNotEmpty(batchInfoReq.getAppId())) {
                List<Long> uuids = userList.stream().map(BaseEntity::getId).collect(Collectors.toList());
                wechatUserInfo = wechatService.queryListByUserIdAndAppId(uuids, batchInfoReq.getAppId(), tenantId);
            }
        }

        if (CollectionUtils.isNotEmpty(batchInfoReq.getUuids())) {
            userList = userRepository.findByUuidInAndTenantId(batchInfoReq.getUuids(), tenantId);
            if (CollectionUtils.isNotEmpty(userList) && StringUtils.isNotEmpty(batchInfoReq.getAppId())) {
                List<Long> uuids = userList.stream().map(BaseEntity::getId).collect(Collectors.toList());
                wechatUserInfo = wechatService.queryListByUserIdAndAppId(uuids, batchInfoReq.getAppId(), tenantId);
            }
        }

        if (CollectionUtils.isNotEmpty(batchInfoReq.getPhoneNos())) {
            userList = userRepository.findByPhoneNoInAndTenantId(batchInfoReq.getPhoneNos(), tenantId);
            if (CollectionUtils.isNotEmpty(userList) && StringUtils.isNotEmpty(batchInfoReq.getAppId())) {
                List<Long> uuids = userList.stream().map(BaseEntity::getId).collect(Collectors.toList());
                wechatUserInfo = wechatService.queryListByUserIdAndAppId(uuids, batchInfoReq.getAppId(), tenantId);
            }
        }

        if (StringUtils.isNotEmpty(batchInfoReq.getAppId()) && CollectionUtils.isNotEmpty(batchInfoReq.getOpenIds())) {
            wechatUserInfo = wechatService.findWechatUserInfoFromDb(batchInfoReq.getOpenIds(), batchInfoReq.getAppId(), tenantId);
            List<Long> userIds = wechatUserInfo.stream().map(WechatUserInfo::getUserId).collect(Collectors.toList());
            if (CollectionUtils.isEmpty(wechatUserInfo) || CollectionUtils.isEmpty(userIds)) {
                throw new SilentBizException(BizExceptionEnum.UN_EXIT_USER);
            }

            userList = userRepository.findByIdInAndTenantId(userIds, tenantId);
        }

        if (StringUtils.isNotEmpty(batchInfoReq.getAppId()) && CollectionUtils.isNotEmpty(batchInfoReq.getUnionIds())) {
            wechatUserInfo = wechatService.findUnionIdsAndOpenIdAndTenantId(batchInfoReq.getUnionIds(), batchInfoReq.getAppId(), tenantId);
            List<Long> userIds = wechatUserInfo.stream().filter(e->e.getUserId() != null).map(WechatUserInfo::getUserId).collect(Collectors.toList());
            if (CollectionUtils.isEmpty(wechatUserInfo) || CollectionUtils.isEmpty(userIds)) {
                throw new SilentBizException(BizExceptionEnum.UN_EXIT_USER);
            }

            userList = userRepository.findByIdInAndTenantId(userIds, tenantId);
        }

        //3、异常处理
        if (CollectionUtils.isEmpty(userList)) {
            throw new SilentBizException(BizExceptionEnum.UN_EXIT_USER);
        }

        List<Long> userIdList = userList.stream().map(e -> e.getId()).collect(Collectors.toList());


        try {
            List<UserInfoEntity> userInfoEntityList = userInfoRepository.findByTenantIdAndUserIdIn(tenantId,userIdList);
            return JsonResult.buildSuccessResultGeneric(UserBean.from(userList, wechatUserInfo,userInfoEntityList));
        } catch (Exception e) {
            log.error("获取批量信息异常：{}",e.getMessage(),e);
        }
        return JsonResult.buildErrorStateResult("获取用户信息执行出错",null );
    }

}
