package cn.quantgroup.xyqb.controller.internal.user;

import cn.quantgroup.tech.util.TechEnvironment;
import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.captcha.CaptchaFiniteValidator;
import cn.quantgroup.xyqb.aspect.lock.PasswordErrorFiniteValidator;
import cn.quantgroup.xyqb.aspect.logcaller.LogHttpCaller;
import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.entity.Merchant;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserDetail;
import cn.quantgroup.xyqb.entity.WechatUserInfo;
import cn.quantgroup.xyqb.exception.UserNotExistException;
import cn.quantgroup.xyqb.exception.VerificationCodeErrorException;
import cn.quantgroup.xyqb.exception.WechatRelateUserException;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.UserModel;
import cn.quantgroup.xyqb.model.UserStatistics;
import cn.quantgroup.xyqb.service.merchant.IMerchantService;
import cn.quantgroup.xyqb.service.register.IUserRegisterService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.service.user.ILockIpv4Service;
import cn.quantgroup.xyqb.service.user.IUserDetailService;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.service.wechat.IWechatService;
import cn.quantgroup.xyqb.util.IPUtil;
import cn.quantgroup.xyqb.util.MqUtils;
import cn.quantgroup.xyqb.util.PasswordUtil;
import cn.quantgroup.xyqb.util.ValidationUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * Http服务接口：用户注册、登录、重置密码
 * Created by FrankChow on 15/7/5.
 */
@RestController
@RequestMapping("/user")
public class UserController implements IBaseController {

  private static final Logger LOGGER = LoggerFactory.getLogger(UserController.class);

  @Autowired
  private IUserService userService;

  @Autowired
  private ILockIpv4Service lockIpv4Service;

  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> stringRedisTemplate;

  @Autowired
  private ISmsService smsService;

  @Autowired
  private ISessionService sessionService;

  @Autowired
  private IUserDetailService userDetailService;
  @Autowired
  private IMerchantService merchantService;
  @Autowired
  private IWechatService wechatService;
  @Autowired
  private IUserRegisterService userRegisterService;

  @RequestMapping("/test")
  public JsonResult test() {
    HttpServletRequest request = getRequest();
    IPUtil.logIp(LOGGER, request);
    return JsonResult.buildSuccessResult("", getCurrentUserFromRedis());
  }

  /**
   * 登录（账号 + 密码）
   * 密码错误达到限定次数时执行图形验证码校验
   * 图形验证码累计错误达到限定次数时须重新获取
   *
   * @param channelId
   * @param appChannel
   * @param createdFrom
   * @param userId
   * @param key
   * @param request
   * @param dimension
   * @return
   */
  @CaptchaFiniteValidator
  @RequestMapping("/loginV1")
  public JsonResult loginV1(
          @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
          @RequestParam(required = false, defaultValue = "1") Long createdFrom,
          @RequestParam(required = false, defaultValue = "") String userId,
          @RequestParam(required = false,defaultValue = "xyqb") String key,
          @RequestParam(required = false) String dimension,
          HttpServletRequest request) {
    LOGGER.info("loginV1 -> channelId:{},appChennel:{},createdFrom:{},userId:{},key:{},dimension:{}",channelId, appChannel, createdFrom, userId, key, dimension);
    return login(channelId, appChannel, createdFrom, userId, key, dimension, request);
  }

  @LogHttpCaller
  @PasswordErrorFiniteValidator
  @RequestMapping("/login")
  public JsonResult login(
          @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
          @RequestParam(required = false, defaultValue = "1") Long createdFrom,
          @RequestParam(required = false, defaultValue = "") String userId,
          @RequestParam(required = false,defaultValue = "xyqb") String key,
          @RequestParam(required = false) String dimension,
          HttpServletRequest request) {
    LOGGER.info("login -> channelId:{},appChannel:{},createdFrom:{},userId:{},key:{},dimension:{}",channelId,appChannel,createdFrom,userId,key,dimension);
    Merchant merchant = merchantService.findMerchantByName(key);
    if (merchant == null) {
      return JsonResult.buildErrorStateResult("未知的连接", null);
    }
    if (!StringUtils.isEmpty(userId) && userId.length() > 10) {
      return loginWithUserId(channelId, appChannel, createdFrom, userId, merchant,dimension, request);
    } else {
      return loginWithHttpBasic(channelId, appChannel, createdFrom, merchant,dimension, request);
    }
  }

  /**
   * 快速登录（手机号 + 短信验证码），H5专用入口
   * 短信验证码错误达到限定次数时执行图形验证码校验
   * 图形验证码累计错误达到限定次数时须重新获取
   *
   * @param channelId
   * @param appChannel
   * @param createdFrom
   * @param key
   * @param btRegisterChannelId
   * @param dimension
   * @param request
   * @return
   */
  @RequestMapping("/login/fastV1")
  public JsonResult loginFastV1(
          @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
          @RequestParam(required = false, defaultValue = "1") Long createdFrom,
          @RequestParam(required = false,defaultValue = "xyqb") String key,
          @RequestParam(required = false) Long btRegisterChannelId,
          @RequestParam(required = false) String dimension,
          @RequestParam(name = "click_id", required = false) String clickId,
          HttpServletRequest request) {
    LOGGER.info("login/fastV1 -> channelId:{},appChennel:{},createdFrom:{},key:{},btRegisterChannelId:{},dimension:{},clickId:{}",channelId, appChannel, createdFrom, key, btRegisterChannelId, dimension, clickId);
    return loginFast(channelId, appChannel, createdFrom, key, btRegisterChannelId, dimension, clickId, request);
  }

  @RequestMapping("/login/fast")
  public JsonResult loginFast(
          @RequestParam(required = false, defaultValue = "1") Long channelId, String appChannel,
          @RequestParam(required = false, defaultValue = "1") Long createdFrom,
          @RequestParam(required = false,defaultValue = "xyqb") String key,
          @RequestParam(required = false)Long btRegisterChannelId,
          @RequestParam(required = false) String dimension,
          @RequestParam(name = "click_id", required = false) String clickId,
          HttpServletRequest request) {
    Map<String, JsonResult> validMap = getHeaderParam(request);
    LOGGER.info("login/fast -> channelId:{},appChannel:{},createdFrom:{},btRegisterChannelId:{},key:{},dimension:{},clickId:{}",channelId,appChannel,createdFrom,btRegisterChannelId,key,dimension, clickId);
    if (null != validMap.get("fail")) {
      return validMap.get("fail");
    }
    Merchant merchant = merchantService.findMerchantByName(key);
    if (merchant == null) {
      return JsonResult.buildErrorStateResult("未知的连接", null);
    }
    JsonResult successResult = validMap.get("success");
    String phoneNo = successResult.getData().toString();
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      LOGGER.info("用户快速登录失败，手机号错误, createdFrom:{},phoneNo:{}", createdFrom, phoneNo);
      throw new UserNotExistException("手机号错误");
    }
    String verificationCode =  successResult.getMsg();
    // 执行短信验证码检查
    verifyPhoneAndCode(phoneNo, verificationCode);
    User user = userService.findByPhoneWithCache(phoneNo);
    if (user != null && !user.getEnable()) {
      LOGGER.error("用户不存在，或者已经注销，phoneNo:{}",phoneNo);
      return JsonResult.buildErrorStateResult("登录失败", null);
    }
    if (user == null) {
      // Service层会负责发送注册消息到LoanVest
      user = userRegisterService.register(phoneNo, channelId, createdFrom, appChannel,btRegisterChannelId,dimension);
      if (user == null) {
        throw new UserNotExistException("用户未找到");
      }
      //广点通转化注册 - 发送消息 - 方法内过滤
      MqUtils.sendRegisterMessageForGdt(phoneNo, clickId);
    }
    if(!wechatRelateUserIfNecessary(user, request)){
      return JsonResult.buildErrorStateResult("登录时微信关联失败", null);
    }
    //增加登陆统计发送
    UserStatistics statistics = new UserStatistics(user,dimension,3,channelId);
    MqUtils.sendLoanVest(statistics);
    return new JsonResult(sessionService.createSession(channelId, createdFrom, appChannel, user, merchant));
  }

  /**
   * 快速登录验证
   *
   * @param request
   * @return
   */
  private Map<String, JsonResult> getHeaderParam(HttpServletRequest request) {
    Map<String, JsonResult> result = new HashMap<String, JsonResult>();
    String verificationHeader = "Verification ";
    String credential = request.getHeader("authorization");
    if (StringUtils.isBlank(credential)) {
      result.put("fail", JsonResult.buildErrorStateResult("登录失败", null));
      return result;
    }
    if (!credential.startsWith(verificationHeader)) {
      result.put("fail", JsonResult.buildErrorStateResult("登录失败", null));
      return result;
    }
    credential = credential.substring(verificationHeader.length(), credential.length());
    byte[] buf = Base64.decodeBase64(credential);
    credential = new String(buf, Charset.forName("UTF-8"));
    String[] credentialArr = credential.split(":");
    if (credentialArr.length != 2) {
      result.put("fail", JsonResult.buildErrorStateResult("登录失败", null));
      return result;
    }
    String phoneNo = credentialArr[0];
    String verificationCode = credentialArr[1];
    LOGGER.info("用户快速登录,phoneNo:{} , verificationCode:{}", phoneNo, verificationCode);
    if (!ValidationUtil.validatePhoneNo(phoneNo) || StringUtils.isBlank(verificationCode)) {
      result.put("fail", JsonResult.buildErrorStateResult("登录失败", null));
      return result;
    }
    result.put("success", JsonResult.buildSuccessResult(verificationCode, phoneNo));
    return result;
  }

  /**
   * 用户注册
   *
   * @param phoneNo
   * @param password
   * @param verificationCode
   * @param channelId
   * @return
   */
  @RequestMapping("/register")
  public JsonResult register(@RequestParam String phoneNo, @RequestParam String password,
                             @RequestParam String verificationCode, @RequestParam(required = false) Long channelId,
                             @RequestParam(required = false) Long registerFrom,
                             @RequestParam(required = false)Long btRegisterChannelId,@RequestParam(required = false)String dimension) {
    LOGGER.info("用户注册：register -> phoneNo:{}, verificationCode:{}, channelId:{}, registerFrom:{},btRegisterChannelId:{},dimension:{}", phoneNo, verificationCode, channelId, registerFrom,btRegisterChannelId,dimension);
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      LOGGER.info("用户注册失败，手机号错误：register -> registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("手机号错误", null);
    }
    if (StringUtils.isEmpty(password)) {
      LOGGER.info("用户注册失败，密码不能为空：register -> registerFrom:{}, phoneNo:{}, password:{}", registerFrom, phoneNo, password);
      return JsonResult.buildErrorStateResult("密码不能为空", null);
    }
    if (password.length() < 6 || password.length() > 12) {
      LOGGER.info("用户注册失败，密码长度须在6位至12位之间：register -> registerFrom:{}, phoneNo:{}, password:{}", registerFrom, phoneNo, password);
      return JsonResult.buildErrorStateResult("密码应为6-12位", null);
    }
    if (null == registerFrom) {
      registerFrom = 1L;
    }
    verifyPhoneAndCode(phoneNo, verificationCode);
    if (userService.exist(phoneNo)) {
      LOGGER.info("用户注册失败，该手机号已经被注册：register -> registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("该手机号已经被注册", null);
    }
    if (!userRegisterService.register(phoneNo, password, registerFrom, getIp(), channelId,btRegisterChannelId,dimension)) {
      LOGGER.info("用户快速注册失败，请稍后重试, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("注册失败，请稍后重试", null);
    }
    LOGGER.info("用户注册成功：register -> registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
    return JsonResult.buildSuccessResult(null, null);
  }

  /**
   * 检查用户是否存在
   *
   * @param phoneNo 手机号
   * @return
   */
  @RequestMapping("/exist")
  public JsonResult exist(@RequestParam String phoneNo) {
    LOGGER.info("检查用户是否存在, phoneNo:{}", phoneNo);
    if (userService.exist(phoneNo)) {
      LOGGER.info("该手机号已经注册, phoneNo:{}", phoneNo);
      return JsonResult.buildErrorStateResult("该手机号已经注册", null);
    }
    return JsonResult.buildSuccessResult(null, null);
  }

  /**
   * 检查用户是否存在
   *
   * @param phoneNo 手机号
   * @return
   */
  @RequestMapping("/exist_check")
  public JsonResult existForResetPwd(@RequestParam String phoneNo) {
    LOGGER.info("检查用户是否存在, phoneNo:{}", phoneNo);
    return JsonResult.buildSuccessResult(null, userService.exist(phoneNo));
  }

  /**
   * 重置密码
   *
   * @param phoneNo
   * @param password
   * @param verificationCode
   * @return
   */
  @LogHttpCaller
  @RequestMapping("/reset_password")
  public JsonResult resetPassword(@RequestParam String phoneNo,
                                  @RequestParam String password,
                                  @RequestParam(required = false) String registerFrom,
                                  @RequestParam String verificationCode) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return JsonResult.buildErrorStateResult("手机号错误", null);
    }
    if (StringUtils.isEmpty(password)) {
      return JsonResult.buildErrorStateResult("密码不能为空", null);
    }
    if (password.length() < 6 || password.length() > 12) {
      return JsonResult.buildErrorStateResult("密码应为6-12位", null);
    }
    verifyPhoneAndCode(phoneNo, verificationCode);
    if (!userService.exist(phoneNo)) {
      LOGGER.info("修改密码失败，该手机号尚未注册, registerFrom:{}, phoneNo:{}", registerFrom, phoneNo);
      return JsonResult.buildErrorStateResult("该手机号尚未注册", null);
    }
    if (!userService.resetPassword(phoneNo, password)) {
      return JsonResult.buildErrorStateResult("修改密码失败", null);
    }

    // TODO  加渠道号
    LOGGER.info("修改密码成功, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return JsonResult.buildSuccessResult(null, null);
  }

  /**
   * 检查token是否已经过期不存在了
   *
   * @param token
   * @return
   */
  @RequestMapping("/exists_token")
  public JsonResult checkToken(@RequestParam String token) {
    if(StringUtils.isEmpty(token)){
      return JsonResult.buildSuccessResult(null,false);
    }
    if(token.contains("*")){
      return JsonResult.buildSuccessResult(null,false);
    }

    String tokenKey = Constants.SESSION_PREFIX + token;
    String tokenKey2 = Constants.Session.USER_SESSION_CACHE + token;
    return JsonResult.buildSuccessResult(null, stringRedisTemplate.hasKey(tokenKey)||stringRedisTemplate.hasKey(tokenKey2));
  }

  @RequestMapping("/syncUserInfo")
  public JsonResult syncUserInfo() {
    User user = getCurrentUserFromRedis();
    if (null == user) {
      return JsonResult.buildErrorStateResult(null, null);
    }
    UserDetail detail = userDetailService.findByUserId(user.getId());
    UserModel userModel = new UserModel(user, detail);
    return JsonResult.buildSuccessResult("token校验成功", userModel);
  }

  private JsonResult loginWithHttpBasic(Long channelId, String appChannel, Long createdFrom, Merchant merchant,String dimension, HttpServletRequest request) {
    User user = verificateUserNameAndPassword(request);
    if (user == null) {
      return JsonResult.buildErrorStateResult("用户名或密码不正确", null);
    }else if(!wechatRelateUserIfNecessary(user, request)){
      return JsonResult.buildErrorStateResult("登录时微信关联失败", null);
    }
    //增加登陆统计发送
    UserStatistics statistics=new UserStatistics(user,dimension,1,channelId);
    MqUtils.sendLoanVest(statistics);
    LOGGER.info("用户登陆成功phonNo:{},当前ip:{}，用户登陆信息：{}",user.getPhoneNo(), IPUtil.getRemoteIP(request), statistics);
    return new JsonResult(sessionService.createSession(channelId, createdFrom, appChannel, user, merchant));
  }

  private User verificateUserNameAndPassword(HttpServletRequest request) {
    String credential = request.getHeader("authorization");
    if (StringUtils.isBlank(credential) || !credential.startsWith("Basic ")) {
      return null;
    }
    credential = credential.substring("Basic ".length(), credential.length());
    byte[] buf = Base64.decodeBase64(credential);
    String bufStr = "";
    try {
      bufStr = new String(buf, "UTF-8");
    } catch (UnsupportedEncodingException e) {
      LOGGER.error("不支持的编码: ", e);
    }
    String clientIp = getIp();
    String[] credentialArr = bufStr.split(":");
    if (credentialArr.length != 2) {
      LOGGER.info("用户登录失败：{}", bufStr);
      // 向该ipv4添加错误计数器
      lockIpv4Service.countErrorByIpv4(clientIp);
      return null;
    }
    String phoneNo = credentialArr[0];
    String pass = credentialArr[1];
    User user = userService.findByPhoneWithCache(phoneNo);
    if (user == null || !user.getEnable()) {
      // 向该phoneNo添加错误计数器
      lockIpv4Service.countErrorByPhoneNo(phoneNo);
      // 向该ipv4添加错误计数器
      lockIpv4Service.countErrorByIpv4(clientIp);
      return null;
    }
    //验证密码
    if (!validatePassword(pass, user.getPassword())) {
      // 向该phoneNo添加错误计数器
      lockIpv4Service.countErrorByPhoneNo(phoneNo);
      // 向该ipv4添加错误计数器
      lockIpv4Service.countErrorByIpv4(clientIp);
      return null;
    }
    // 向该ipv4添加成功计数器
    lockIpv4Service.countSuccessByIpv4(clientIp);
    return user;
  }

  private boolean validatePassword(String paramPass, String targetPassword) {
    return StringUtils.defaultString(targetPassword, "").equals(PasswordUtil.MD5(paramPass.toLowerCase() + Constants.PASSWORD_SALT));
  }

  private JsonResult loginWithUserId(Long channelId, String appChannel, Long createdFrom, String userId, Merchant merchant,String dimension,HttpServletRequest request) {
    //查询用户
    User user = userService.findByUuidInDb(userId);
    if (Objects.isNull(user) || !user.getEnable()) {
      LOGGER.error("用户不存在，或者已经注销，userId:{}",userId);
      return JsonResult.buildErrorStateResult("登录失败", null);
    } else if(!wechatRelateUserIfNecessary(user, request)){
      return JsonResult.buildErrorStateResult("登录时微信关联失败", null);
    }
    //增加登陆统计发送
    UserStatistics statistics=new UserStatistics(user,dimension,4,channelId);
    MqUtils.sendLoanVest(statistics);
    //更新session
    return new JsonResult(sessionService.createSession(channelId, createdFrom, appChannel, user, merchant));
  }

  /**
   * 如果必要的话，关联用户和微信
   * @param user - 用户标识
   * @param request - 当前请求
   * @return true - 继续登录，false - 微信关联失败，重新登录
   */
  private boolean wechatRelateUserIfNecessary(User user,HttpServletRequest request){
    Objects.requireNonNull(request, "无效请求");
    String clientIp = IPUtil.getRemoteIP(request);
    Set<String> paramKeys = request.getParameterMap().keySet();
    boolean ready = paramKeys.contains(Constants.OPEN_ID);
    if(!ready){
      return true;
    }else if(Objects.isNull(user) || Objects.isNull(user.getId()) || StringUtils.isBlank(request.getParameter(Constants.OPEN_ID))){
      LOGGER.warn("微信关联失败，user:{}, request:{}",user, JSON.toJSONString(request));
      return false;
    }
    Long userId = user.getId();
    String phoneNo = user.getPhoneNo();
    try{
      int rows = wechatService.relateUser(userId, phoneNo, request.getParameter(Constants.OPEN_ID));
      return rows > 0;
    }catch (Exception e){
      LOGGER.error("微信关联失败，user:{}, request:{}",user, JSON.toJSONString(request), e);
    }
    return false;
  }

  /**
   * 绑定微信
   * @param openId - 微信openId
   * @return
   */
  @LogHttpCaller
  @RequestMapping("/associate_wechat")
  public JsonResult associateWithWechat(String openId, HttpServletRequest request) {
    User user = getCurrentUserFromRedis();
    LOGGER.error("[UserController][associateWithWechat][非法请求]微信关联，user:{},openId:{},request:{}",user, openId, JSON.toJSONString(request));
    return JsonResult.buildSuccessResult("[非法请求]微信关联", null, 1L);
    /*if(Objects.isNull(user) || Objects.isNull(user.getId())){
      LOGGER.warn("微信关联失败，用户未登录：user:[{}],openId:[{}]",user,openId);
      return JsonResult.buildErrorStateResult("请登陆后重新操作", null);
    }
    Long userId = user.getId();
    *//* 判断用户是否已绑定其他微信 *//*
    WechatUserInfo userInfoInDb = wechatService.queryByUserId(userId);
    if (Objects.nonNull(userInfoInDb)) {
      if(Objects.equals(openId, userInfoInDb.getOpenId())){
        LOGGER.info("微信关联成功：重复关联：跳过：user:[{}],openId:[{}],wechatUserInfo:[{}]",user,openId,userInfoInDb);
        return JsonResult.buildSuccessResult("微信关联成功", null);
      }else if(StringUtils.isNotBlank(userInfoInDb.getOpenId())){
        // 不做绑定
        LOGGER.warn("微信关联失败，该用户已绑定其他微信：user:[{}],openId:[{}],wechatUserInfo:[{}]",user,openId,userInfoInDb);
        return JsonResult.buildErrorStateResult("该用户已绑定其他微信", null);
      }else{
        // 清除无效微信信息
        wechatService.forbiddenUserWeChat(userId);
      }
    }
    *//* 执行绑定 *//*
    WechatUserInfo userInfo = wechatService.findWechatUserInfoFromDb(openId);
    LOGGER.info("微信关联：user:[{}],openId:[{}],wechatUserInfo:[{}]",user,openId,userInfo);
    // 无微信信息
    if (userInfo == null) {
      LOGGER.warn("微信关联失败，缺少微信信息：user:[{}],openId:[{}],wechatUserInfo:[{}]",user,openId,userInfo);
      return JsonResult.buildErrorStateResult("还没有保存微信信息", null);
    }
    // 排他性 - 先入为主
    if (Objects.nonNull(userInfo.getUserId()) && userInfo.getUserId()>0 && !Objects.equals(userId, userInfo.getUserId())) {
      LOGGER.warn("微信关联失败，该微信已经被其他用户绑定：user:[{}],openId:[{}],wechatUserInfo:[{}]",user,openId,userInfo);
      return JsonResult.buildErrorStateResult("该微信已经被其他用户绑定", null);
    }
    *//*
     * 未绑定信用钱包用户
     *//*
    try {
      int rows = wechatService.relateUser(userId, user.getPhoneNo(), openId);
      if(rows > 0){
        LOGGER.info("微信关联成功：rows:{}, user:[{}],openId:[{}],wechatUserInfo:[{}]",rows,user,openId,userInfo);
        return JsonResult.buildSuccessResult("微信关联成功", null);
      }else {
        LOGGER.warn("微信关联失败，保存遇到问题：rows:{}, user:[{}],openId:[{}],wechatUserInfo:[{}]",rows,user,openId,userInfo);
        return JsonResult.buildErrorStateResult("微信关联失败", null);
      }
    } catch (Exception ex) {
      // 不做绑定
      LOGGER.warn("微信关联失败，保存遇到问题：user:[{}],openId:[{}],wechatUserInfo:[{}]",user,openId,userInfo, ex);
      return JsonResult.buildErrorStateResult("微信关联失败", null);
    }*/
  }

  /**
   * 校验短信验证码
   * @param phoneNo
   * @param verificationCode
   */
  private void verifyPhoneAndCode(String phoneNo, String verificationCode) {
    // 非生产环境直接跳过验证码检验
    if(!TechEnvironment.isPro()){
      return;
    }
    if (!smsService.verifyPhoneAndCode(phoneNo, verificationCode)) {
      // 是否需要重新发送短信验证码
      if(smsService.needResendCode(phoneNo)){
        throw new VerificationCodeErrorException("验证码失效，请重新获取");
      }
      LOGGER.info("验证码校验失败,phoneNo:{} , verificationCode:{}", phoneNo, verificationCode);
      throw new VerificationCodeErrorException("短信验证码错误");
    }
  }

}
