package cn.quantgroup.xyqb.controller.external.motan;

import cn.quantgroup.motan.bean.*;
import cn.quantgroup.motan.retbean.*;
import cn.quantgroup.motan.service.UserMotanService;
import cn.quantgroup.motan.vo.UserSysResult;
import cn.quantgroup.user.enums.*;
import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.entity.*;
import cn.quantgroup.xyqb.model.*;
import cn.quantgroup.xyqb.model.IdType;
import cn.quantgroup.xyqb.model.session.LoginInfo;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.repository.IUserRepository;
import cn.quantgroup.xyqb.service.api.IUserApiService;
import cn.quantgroup.xyqb.service.auth.IIdCardService;
import cn.quantgroup.xyqb.service.http.IHttpService;
import cn.quantgroup.xyqb.service.merchant.IMerchantService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.service.user.*;
import cn.quantgroup.xyqb.service.user.vo.UserDetailVO;
import cn.quantgroup.xyqb.service.wechat.IWechatService;
import cn.quantgroup.xyqb.session.XyqbSessionContextHolder;
import cn.quantgroup.xyqb.util.PasswordUtil;
import cn.quantgroup.xyqb.util.ValidationUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableMap;
import com.weibo.api.motan.config.springsupport.annotation.MotanService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.CollectionUtils;

import java.sql.Timestamp;
import java.text.ParseException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Random;
import java.util.stream.Collectors;

import static cn.quantgroup.xyqb.constant.UserConstant.USER_ERROR_OR_PASSWORD_ERROR;

/**
 * Created by 11 on 2017/2/27.
 */
@Slf4j
@MotanService(basicService = "baseServiceConfig")
public class MotanUserServiceImpl implements UserMotanService {


  private static final ObjectMapper MAPPER = new ObjectMapper();
  private final static Random random = new Random();

  static {
    MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
  }


  @Value("${xyqb.auth.url}")
  private String xyqbAuthUrl;

  @Autowired
  private IIdCardService idCardService;
  @Autowired
  private IUserService userService;
  @Autowired
  private IUserDetailService userDetailService;
  @Autowired
  private IUserRepository userRepository;
  @Autowired
  private ISmsService smsService;
  @Autowired
  private ILkbUserService lkbUserService;
  @Autowired
  private IMerchantService merchantService;
  @Autowired
  private ISessionService sessionService;
  @Autowired
  private IContactService contactService;
  @Autowired
  private IAddressService addressService;
  @Autowired
  private IUserExtInfoService userExtInfoService;
  @Autowired
  private IWechatService wechatService;
  @Autowired
  private IUserSpouseService userSpouseService;

  @Autowired
  private IHttpService httpService;

  @Autowired
  private IUserApiService userApiService;
  @Override
  public UserSysResult<XUser> saveUserRelatedInfo(UserRelatedBean userRelatedBean) {
    return null;
  }

  @Override
  public UserSysResult<XUser> findUserByPhoneNo(String phoneNo) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return returnErrorValue("手机号格式不正确");
    }
    User user = userService.findByPhoneInDb(phoneNo);
    return returnSuccessValue(fromUser(user));
  }

  @Override
  public UserSysResult<XUser> findUserByUuid(String uuid) {
    if (StringUtils.isBlank(uuid)) {
      return returnErrorValue("参数不能为空");
    }
    User user = userService.findByUuidInDb(uuid);
    return returnSuccessValue(fromUser(user));
  }

  @Override
  public UserSysResult<XUser> saveUser(UserBean userBean) {
    if (null == userBean || !ValidationUtil.validatePhoneNo(userBean.getPhoneNo())) {
      return returnErrorValue(userBean == null ? "要保存的用户对象不能为null" : "手机号不正确");
    }
    Timestamp now = new Timestamp(System.currentTimeMillis());
    User user = userService.findByPhoneWithCache(userBean.getPhoneNo());
    if (user == null) {
      user = new User();
      user.setCreatedAt(now);
      user.setPhoneNo(userBean.getPhoneNo());
      user.setEnable(true);
      String password = userBean.getPassword();
      if (StringUtils.isBlank(password)) {
        password = String.valueOf(random.nextInt(899999) + 100000);
      }
      user.setPassword(PasswordUtil.MD5(password + Constants.PASSWORD_SALT));
    } else {
      if (StringUtils.isNotBlank(user.getPassword())) {
        user.setPassword(PasswordUtil.MD5(userBean.getPassword() + Constants.PASSWORD_SALT));
      }
    }
    user.setUpdatedAt(now);
    user.setUuid(userBean.getUuid());
    user.setRegisteredFrom(null==userBean.getRegisteredFrom()?0L:userBean.getRegisteredFrom());
    User saveUser = userService.saveUser(user);
    return returnSuccessValue(fromUser(saveUser));
  }

  @Override
  public UserSysResult<XUserDetail> findUserDetailByUserId(Long userId) {
    if (userId == null || userId < 1) {
      return returnErrorValue("参数必须是正整数");
    }
    UserDetail userDetail = userDetailService.findByUserId(userId);
    XUserDetail xUserDetail = fromUserDetail(userDetail);
    return returnSuccessValue(xUserDetail);
  }

  private XUserDetail fromUserDetail(UserDetail userDetail) {
    return userDetail == null ? null : userDetail.toXUserDetail();
  }

  private XUserDetail fromUserDetailVO(UserDetailVO userDetail) {
    return userDetail == null ? null : userDetail.toXUserDetail();
  }

  private XUser fromUser(User user) {
    return user == null ? null : user.toXUser();
  }

  @Override
  public UserSysResult<XUser> findUserByUserId(Long id) {
    if (id == null || id < 1) {
      return returnErrorValue("参数必须是正整数");
    }
    User user = userService.findById(id);
    return returnSuccessValue(fromUser(user));
  }


  @Override
  public UserSysResult<XUserDetail> findUserDetailByPhone(String phoneNo) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return returnErrorValue("手机号格式不正确");
    }
    UserDetail userDetail = userDetailService.findByPhoneNo(phoneNo);
    return returnSuccessValue(fromUserDetail(userDetail));
  }

  @Override
  public UserSysResult<XUserDetail> updateUserDetail(UserDetailUpdateBean userDetailUpdateBean) {
    if (null == userDetailUpdateBean) {
      return returnErrorValue("要保存的用户详情对象不能为null");
    }
    UserDetail userDetail = new UserDetail();
    if (!ValidationUtil.validateChinese(userDetailUpdateBean.getName())) {
      return returnErrorValue("用户姓名错误");
    }
    IdCardInfo info = null;
    try {
      info = idCardService.getIdCardInfo(userDetailUpdateBean.getIdNo());
    } catch (ParseException ex) {
      log.error("用户身份证号错误, idNo: {}", userDetailUpdateBean.getIdNo(), ex);
    }
    if (Objects.isNull(info) || !info.isValid()) {
      return returnErrorValue("身份证号码错误");
    }
    if (null != userDetailUpdateBean.getId() && userDetailUpdateBean.getId() > 0L) {
      userDetail.setId(userDetailUpdateBean.getId());
    }
    userDetail.setUserId(userDetailUpdateBean.getUserId());
    userDetail.setName(userDetailUpdateBean.getName());
    userDetail.setPhoneNo(userDetailUpdateBean.getPhoneNo());
    userDetail.setIdNo(userDetailUpdateBean.getIdNo());
    Timestamp time = new Timestamp(System.currentTimeMillis());
    userDetail.setCreatedAt(time);
    userDetail.setUpdatedAt(time);
    userDetail.setIdType(IdType.ID_CARD);
    userDetail.setGender(info.getGender());
    userDetail.setEmail(userDetailUpdateBean.getEmail());
    userDetail = userDetailService.saveUserDetail(userDetail);
    if (userDetail != null) {
      XUserDetail xUserDetail = convertObject(JSON.toJSONString(userDetail), new TypeReference<XUserDetail>() {
      });
      return returnSuccessValue(xUserDetail);
    }
    return returnErrorValue("保存失败");
  }

  @Override
  public UserSysResult<XUserDetail> saveUserDetail(UserDetailSaveBean userDetailSaveBean) {
    if (userDetailSaveBean == null) {
      return returnErrorValue("数据不能为空");
    }
    Long userId = userDetailSaveBean.getUserId();
    if (userId == null || userId == 0L) {
      return returnErrorValue("用户id为空");
    }
    if (StringUtils.isBlank(userDetailSaveBean.getPhoneNo())) {
      return returnErrorValue("用户手机号为空");
    }
    if (StringUtils.isBlank(userDetailSaveBean.getName())) {
      return returnErrorValue("用户名为空");
    }
    if (StringUtils.isBlank(userDetailSaveBean.getIdNo())) {
      return returnErrorValue("用户身份证为空");
    }
    if (!ValidationUtil.validateChinese(userDetailSaveBean.getName())) {
      return returnErrorValue("用户姓名错误");
    }
    IdCardInfo info = null;
    try {
      info = idCardService.getIdCardInfo(userDetailSaveBean.getIdNo());
    } catch (ParseException ex) {
      log.error("用户身份证号错误, idNo: {}", userDetailSaveBean.getIdNo(), ex);
    }
    if (Objects.isNull(info) || !info.isValid()) {
      return returnErrorValue("身份证号码错误");
    }
    Timestamp time = new Timestamp(System.currentTimeMillis());
    UserDetail userDetail = userDetailService.findByUserId(userId);
    if (userDetail == null) {
      userDetail = new UserDetail();
      userDetail.setUserId(userId);
      userDetail.setCreatedAt(time);
      userDetail.setUpdatedAt(time);
    } else {
      userDetail.setUpdatedAt(time);
    }
    userDetail.setName(userDetailSaveBean.getName());
    userDetail.setPhoneNo(userDetailSaveBean.getPhoneNo());
    userDetail.setIdType(IdType.ID_CARD);
    userDetail.setGender(info.getGender());
    userDetail.setIdNo(userDetailSaveBean.getIdNo());
    userDetail.setEmail(userDetailSaveBean.getEmail());
    userDetail = userDetailService.saveUserDetail(userDetail);
    if (userDetail != null) {
      return returnSuccessValue(fromUserDetail(userDetail));
    }
    return returnErrorValue("操作失败");
  }

  @Override
  public UserSysResult<XLoginInfo> appLoginAndFetchLoginInfo(AppLoginParam appLoginParam, String requestIp) {
    if (!ValidationUtil.validatePhoneNo(appLoginParam.getPhoneNo())) {
      return returnErrorValue("账户或者密码错误");
    }

    log.info("第三方用户登录, loginFrom:{}, requestIp:{}", appLoginParam.getRegisterFrom(), requestIp);
    User user = userService.findByPhoneInDb(appLoginParam.getPhoneNo());
    if (user == null) {
      user = register(appLoginParam.getRegisterFrom(), appLoginParam.getPhoneNo(), appLoginParam.getIdNo(), appLoginParam.getName(),appLoginParam.getChannelId());
    }
    if (user == null) {
      return returnErrorValue("账户或者密码错误");
    }
    Merchant merchant = merchantService.findMerchantByName(appLoginParam.getKey());
    if (merchant == null) {
      return returnErrorValue("无效的商户");
    }
    LoginProperties loginProperties = new LoginProperties();
    loginProperties.setChannelId(appLoginParam.getChannelId());
    loginProperties.setMerchantName(merchant.getName());
    loginProperties.setCreatedFrom(appLoginParam.getRegisterFrom());
    loginProperties.setAppChannel(appLoginParam.getAppChannel());
    SessionStruct sessionStruct = sessionService.createSessionAndPersist(user, loginProperties);

    LoginInfo loginInfo = new LoginInfo();
    loginInfo.setUser(new UserRet(user));
    loginInfo.setToken(sessionStruct.getSid());
    LoginInfo.LoginContext context = new LoginInfo.LoginContext();
    context.setChannelId(sessionStruct.getValues().getLoginProperties().getChannelId());
    context.setCreatedFrom(sessionStruct.getValues().getLoginProperties().getCreatedFrom());
    context.setAppChannel(appLoginParam.getAppChannel());
    loginInfo.setLoginContext(context);

    log.info("第三方用户获取信息登录成功, loginFrom:{}, phoneNo:{},appChannel:{}", appLoginParam.getRegisterFrom(), appLoginParam.getPhoneNo(), appLoginParam.getAppChannel());
    XLoginInfo xLoginInfo = convertObject(JSON.toJSONString(loginInfo), new TypeReference<XLoginInfo>() {
    });
    return returnSuccessValue(xLoginInfo);
  }

  private <T> T convertObject(String json, TypeReference typeReference) {
    try {
      if (StringUtils.isEmpty(json)) {
        return null;
      }
      return MAPPER.readValue(json, typeReference);
    } catch (Exception ex) {
      log.error("用户信息转换出错,json:{}", json, ex);
      return null;
    }

  }

  @Override
  public void updateUserQQ(String qq, Long userId) {
    if (StringUtils.isEmpty(qq) || userId == null || userId == 0L) {
      log.error("QQ或用户id非法,不允许修改:qq{}, userid:{}", qq, userId);
      return;
    }
    userDetailService.updateUserQQ(userId, qq);
  }

  @Override
  public UserSysResult<List<XContact>> findContactsByUserId(Long userId) {
    List<Contact> contacts = contactService.findByUserId(userId, true);
    List<XContact> xContacts = convertObject(JSON.toJSONString(contacts), new TypeReference<List<XContact>>() {
    });
    return returnSuccessValue(xContacts);
  }

  @Override
  public UserSysResult<List<XContact>> saveContacts(Long userId, List<XContact> contacts) {
    if (null == userId || userId == 0L) {
      return returnErrorValue("保存联系人列表失败,用户id为空");
    }
    if (CollectionUtils.isEmpty(contacts)) {
      return returnErrorValue("保存联系人列表失败,列表为空");
    }
    //做一次类型转换,
    String jsonContacts = JSON.toJSONString(contacts);
    List<Contact> contactList = null;
    try {
      contactList = MAPPER.readValue(jsonContacts, new TypeReference<List<Contact>>(){});
    } catch (Exception e) {
      log.error("联系人列表转换错误", e);
      return returnErrorValue("联系人转换错误");
    }
    for (Contact c : contactList) {
      if (!c.valid()) {
        log.info("用户手机号或姓名错误, phoneNo:{},name:{}",  c.getPhoneNo(),  c.getName());
        return returnErrorValue("用户手机号或姓名错误");
      }
    }
    List<Contact> result = contactService.save(userId, contactList);
    List<XContact> xResult = convertObject(JSON.toJSONString(result), new TypeReference<List<XContact>>(){});
    return returnSuccessValue(xResult);
  }

  @Override
  public UserSysResult<XAddress> findAddressByUserId(Long userId) {
    UserSysResult<XAddress> userSysResult = returnErrorValue(null);
    Address address = addressService.findByUserId(userId);
    if (null == address) {
      return userSysResult;
    }
    XAddress xAddress = convertObject(JSON.toJSONString(address), new TypeReference<XAddress>() {
    });
    return returnSuccessValue(xAddress);
  }

  @Override
  public UserSysResult<XAddress> saveAddress(Long userId, Long provinceCode, String province, Long cityCode, String city,
                                             Long districtCode, String district, String address) {
    if (null == userId || null == provinceCode || null == cityCode || StringUtils.isBlank(address)) {
      return returnErrorValue("参数不能为空");
    }
    Address addressObj = addressService.findByUserId(userId);
    if (addressObj == null) {
      addressObj = new Address();
      Timestamp now = new Timestamp(System.currentTimeMillis());
      addressObj.setCreatedAt(now);
      addressObj.setUpdateAt(now);
    }
    addressObj.setUserId(userId);
    addressObj.setProvinceCode(provinceCode);
    addressObj.setCityCode(cityCode);
    addressObj.setCity(city);
    addressObj.setDistrictCode(districtCode);
    addressObj.setDistrict(district);
    addressObj.setAddress(address);
    addressObj.setProvince(province);
    addressObj = addressService.save(addressObj);
    XAddress xAddress = convertObject(JSON.toJSONString(addressObj), new TypeReference<XAddress>() {
    });
    return returnSuccessValue(xAddress);
  }


  @Override
  public UserSysResult<XUserExtInfo> updateUserExtInfo(
      Long userId, cn.quantgroup.motan.enums.IncomeEnum incomeEnum, cn.quantgroup.motan.enums.IncomeRangeEnum incomeRangeEnum,
      cn.quantgroup.motan.enums.OccupationEnum occupationEnum, cn.quantgroup.motan.enums.EducationEnum educationEnum, Boolean hasCar,
      Boolean hasSocialSecurity, Boolean hasHouse, Boolean hasCreditCard, cn.quantgroup.motan.enums.MaritalStatus maritalStatus) {
    UserSysResult<XUserExtInfo> userSysResult = returnValue(null, null);
    if (null == userId || 0L == userId) {
      return returnErrorValue("用户id为空");
    }
    UserExtInfo info = userExtInfoService.findByUserId(userId);
    if (info == null) {
      Timestamp now = new Timestamp(System.currentTimeMillis());
      info = new UserExtInfo();
      info.setUserId(userId);
      info.setCreatedAt(now);
      info.setUpdateAt(now);
    }
    if (incomeEnum != null) {
      info.setIncomeEnum(IncomeEnum.valueOf(incomeEnum.name()));
    }
    if (incomeRangeEnum != null) {
      info.setIncomeRangeEnum(IncomeRangeEnum.valueOf(incomeRangeEnum.name()));
    }
    if (occupationEnum != null) {
      info.setOccupationEnum(OccupationEnum.valueOf(occupationEnum.name()));
    }
    if (educationEnum != null) {
      info.setEducationEnum(EducationEnum.valueOf(educationEnum.name()));
    }
    if (hasCar != null) {
      info.setHasCar(hasCar);
    }
    if (hasCreditCard != null) {
      info.setHasCreditCard(hasCreditCard);
    }
    if (hasHouse != null) {
      info.setHasHouse(hasHouse);
    }
    if (maritalStatus != null) {
      info.setMarryStatus(MaritalStatus.valueOf(maritalStatus.name()));
    }
    info = userExtInfoService.save(info);
    XUserExtInfo userExtInfo = convertObject(JSON.toJSONString(info), new TypeReference<XUserExtInfo>() {
    });
    return returnSuccessValue(userExtInfo);
  }

  @Override
  public UserSysResult<String> queryOpenIdByXyqbUserId(Long userId) {
    if (userId == null || userId < 1) {
      return returnErrorValue("用户id不能为空");
    }
    WechatUserInfo wechatUserInfo = wechatService.queryOpenIdByUserId(userId);
    if (wechatUserInfo == null) {
      return returnErrorValue("wechat信息为空");
    }
    return returnValue(wechatUserInfo.getOpenId(), "success");
  }

  @Override
  public UserSysResult<Boolean> disableUser(Long userId) {
    if (null == userId || 0L == userId) {
      throw new RuntimeException("userId不能为空");
    }
    User user = userService.findById(userId);
    if (null == user) {
      throw new RuntimeException("未查询到该用户,用户id:"+userId);
    }
    user.setEnable(false);
    user.setUpdatedAt(new Timestamp(System.currentTimeMillis()));
    log.info("注销用户的信息,userId={}", userId);
    user = userService.saveUser(user);
    if (!user.getEnable()) {
      sessionService.deleteByUserId(userId);
    }
    return returnSuccessValue(Boolean.TRUE);
  }

  @Override
  public UserSysResult<String> queryOpenIdByPhoneNo(String phoneNo) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return returnErrorValue("手机号格式不正确");
    }
    WechatUserInfo wechatUserInfo = wechatService.findWechatUserInfoByPhoneNo(phoneNo);
    String openId = null == wechatUserInfo ? null : wechatUserInfo.getOpenId();
    return returnSuccessValue(openId);
  }

  @Override
  public UserSysResult<UserRegisterTrace> findRegisterTraceByPhoneNo(String phoneNo) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return returnErrorValue("手机号格式不正确");
    }
    return null;
  }

  @Override
  public UserSysResult<XUserSpouse> findUserSpouseByUserId(Long userId) {
    if (userId == null || userId == 0) {
      return returnErrorValue("用户不能为空");
    }
    UserSpouse userSpouse = userSpouseService.findByUserId(userId);
    if (userSpouse == null || !userSpouse.valid()) {
      userSpouse = new UserSpouse(userId);
      userSpouse.setStatus(MaritalStatus.UNKNOWN);
    }
    return returnSuccessValue(fromUserSpouse(userSpouse));
  }

  private XUserSpouse fromUserSpouse(UserSpouse userSpouse) {
    if (userSpouse == null) {
      return null;
    }
    XUserSpouse xUserSpouse = new XUserSpouse();
    xUserSpouse.setUserId(userSpouse.getUserId());
    xUserSpouse.setSpousePhone(userSpouse.getSpousePhone());
    xUserSpouse.setSpouseName(userSpouse.getSpouseName());
    xUserSpouse.setCreatedAt(userSpouse.getCreatedAt());
    xUserSpouse.setUpdateAt(userSpouse.getUpdateAt());
    xUserSpouse.setStatus(cn.quantgroup.motan.enums.MaritalStatus.valueOf(userSpouse.getStatus().name()));
    return xUserSpouse;
  }

  @Override
  public UserSysResult<XUserExtInfo> findUserExtInfoByUserId(Long userId) {
    if (userId == null) {
      return returnErrorValue("userId不能为空");
    }
    UserExtInfo userExtInfo = userExtInfoService.findByUserId(userId);
    if (userExtInfo == null) {
      return returnErrorValue("未找到用户扩展信息");
    }
    return returnSuccessValue(fromUserExtInfo(userExtInfo));
  }

  private XUserExtInfo fromUserExtInfo(UserExtInfo userExtInfo) {
    return null==userExtInfo?null:userExtInfo.toXUserExtInfo();
  }

  @Override
  public UserSysResult<List<XUserDetail>> queryUserDetailBySpecification(String name, String phoneNo, String idNo) {
    List<UserDetailVO> userDetails = userDetailService.searchUserDetailList(name, phoneNo, idNo);
    if (CollectionUtils.isEmpty(userDetails)) {
      return null;
    }
    List<XUserDetail> xUserDetails = userDetails.stream().map(user -> fromUserDetailVO(user)).collect(Collectors.toList());
    return returnSuccessValue(xUserDetails);
  }

  @Override
  public UserSysResult<String> loginWithUuid(String uuid) {
    return returnErrorValue("MotanUserServiceImpl loginWithUuid(String uuid) need to be finish.");
  }

  @Override
  public UserSysResult<Boolean> userImportCheck(String phoneNo, String registerFrom) {
    if ("244".equals(registerFrom)) {
      return returnErrorValue("用户导入检查拒绝");
    }
    if (org.apache.commons.lang3.StringUtils.isEmpty(phoneNo) || org.apache.commons.lang3.StringUtils.isEmpty(registerFrom)) {
      return returnErrorValue("检查传入的参数，参数不全。");
    }
    boolean checkPassed = userApiService.userImportCheck(phoneNo);
    if (checkPassed) {
      return returnValue(Boolean.TRUE, "success");
    }
    return returnErrorValue("用户导入检查拒绝");
  }

  @Override
  public UserSysResult<XUserSpouse> saveUserSpouse(XUserSpouse xUserSpouse) {
    if (xUserSpouse.getUserId() == null || xUserSpouse.getUserId() == 0) {
      return returnErrorValue("用户不能为空");
    }
    cn.quantgroup.motan.enums.MaritalStatus status = xUserSpouse.getStatus();
    String spousePhone = xUserSpouse.getSpousePhone();
    String spouseName = xUserSpouse.getSpouseName();
    if (xUserSpouse.getStatus() == cn.quantgroup.motan.enums.MaritalStatus.MARRIED) {
      if (!ValidationUtil.validatePhoneNo(spousePhone)) {
        return returnErrorValue("手机号格式错误");
      }
      if (!ValidationUtil.validateChinese(spouseName)) {
        return returnErrorValue("配偶姓名错误");
      }
    }
    UserSpouse userSpouse = userSpouseService.findByUserId(xUserSpouse.getUserId());
    Timestamp timestamp = new Timestamp(System.currentTimeMillis());
    if (userSpouse == null) {
      userSpouse = new UserSpouse(xUserSpouse.getUserId());
      userSpouse.setCreatedAt(timestamp);
    }
    userSpouse.setSpouseName(status == cn.quantgroup.motan.enums.MaritalStatus.MARRIED ? spouseName : "");
    userSpouse.setSpousePhone(status == cn.quantgroup.motan.enums.MaritalStatus.MARRIED ? spousePhone : "");
    userSpouse.setStatus(cn.quantgroup.user.enums.MaritalStatus.valueOf(status.name()));
    userSpouse.setUpdateAt(timestamp);
    userSpouse = userSpouseService.save(userSpouse);
    return returnSuccessValue(fromUserSpouse(userSpouse));
  }

  @Override
  public UserSysResult<AppLoginResultBean> appLogin(AppLoginParam appLoginParam) {
    if (!ValidationUtil.validatePhoneNo(appLoginParam.getPhoneNo())) {
      throw new RuntimeException(USER_ERROR_OR_PASSWORD_ERROR);
    }

    log.info("第三方用户登录, appLoginParam:{} ", appLoginParam);
    User user = userService.findByPhoneInDb(appLoginParam.getPhoneNo());
    if (user == null) {
      user = register(appLoginParam.getRegisterFrom(), appLoginParam.getPhoneNo(), appLoginParam.getIdNo(), appLoginParam.getName(), appLoginParam.getChannelId());
    }
    if (user == null) {
      throw new RuntimeException(USER_ERROR_OR_PASSWORD_ERROR);
    }
    if (!user.getEnable()) {
      throw new RuntimeException("登录失败");
    }
    Merchant merchant = merchantService.findMerchantByName(appLoginParam.getKey());
    if (merchant == null) {
      throw new RuntimeException("无效的商户");
    }
    LoginProperties loginProperties = new LoginProperties();
    loginProperties.setChannelId(appLoginParam.getChannelId());
    loginProperties.setMerchantName(merchant.getName());
    loginProperties.setCreatedFrom(appLoginParam.getRegisterFrom());
    loginProperties.setAppChannel(appLoginParam.getAppChannel());
    SessionStruct sessionStruct = sessionService.createSessionAndPersist(user, loginProperties);

    AppLoginResultBean appLoginResultBean = new AppLoginResultBean();
    appLoginResultBean.setToken(sessionStruct.getSid());
    appLoginResultBean.setPhoneNo(appLoginParam.getPhoneNo());

    log.info("第三方用户登录成功, loginFrom:{}, phoneNo:{},appChannel:{}", appLoginParam.getRegisterFrom(), appLoginParam.getPhoneNo(), appLoginParam.getAppChannel());

    return returnSuccessValue(appLoginResultBean);
  }

  @Override
  public UserSysResult<XLoginInfo> getLoginInfo(String token) {
    SessionStruct sessionStruct = XyqbSessionContextHolder.getXSessionFromRedis(token);
    if(null != sessionStruct) {
      log.info("从用户中心获取到了用户登录信息:phone:[{}]", sessionStruct.getValues().getUser().getPhoneNo());
      XLoginInfo loginInfo = new XLoginInfo();
      loginInfo.setUser(sessionStruct.getValues().getUser().toXUser());
      loginInfo.setToken(sessionStruct.getSid());
      LoginContext context = new LoginContext();
      context.setChannelId(sessionStruct.getValues().getLoginProperties().getChannelId());
      context.setCreatedFrom(sessionStruct.getValues().getLoginProperties().getCreatedFrom());
      context.setAppChannel(sessionStruct.getValues().getLoginProperties().getAppChannel());
      loginInfo.setLoginContext(context);
      return returnSuccessValue(loginInfo);
    }else {
      // 函谷关去查token 返回值高仿
      log.info("去向函谷关查询用户信息");
      if(StringUtils.isBlank(token) ||token.length() != 36){
        return returnErrorValue("用户未登录");
      }
      String checkUrl = xyqbAuthUrl+"/innerapi/is_login";
      ImmutableMap<String, String> headMap = ImmutableMap.of("x-auth-token", token);
      String response = httpService.get(checkUrl, headMap, null);
      log.info("去向函谷关查询用户信息,response:[{}]",response);
      JsonResult result = JSONObject.parseObject(response, JsonResult.class);
      if (result == null || !"0000".equals(result.getCode()) || !"0000".equals(result.getBusinessCode())) {
        return returnErrorValue("用户未登录");
      }
      String phoneNo = ((Map<String, String>) result.getData()).get("phoneNo");
      User user = userService.findByPhoneWithCache(phoneNo);
      XLoginInfo loginInfo = new XLoginInfo();
      loginInfo.setUser(sessionStruct.getValues().getUser().toXUser());
      loginInfo.setToken(token);
      LoginContext context = new LoginContext();
      context.setChannelId(null);
      context.setCreatedFrom(user.getRegisteredFrom());
      context.setAppChannel("");
      loginInfo.setLoginContext(context);
      return returnSuccessValue(loginInfo);
    }
  }


  /*@Override
  public UserSysResult<XLoginInfo> getLoginInfo(String token) {
    SessionStruct sessionStruct = XyqbSessionContextHolder.getXSessionFromRedis(token);
    if(null != sessionStruct){

      LoginInfo loginInfo = new LoginInfo();
      loginInfo.setUser(UserRet.getUserRet(sessionStruct.getValues().getUser()));
      loginInfo.setToken(sessionStruct.getSid());
      XLoginInfo xLoginInfo = new XLoginInfo();
      xLoginInfo.setToken(token);
      xLoginInfo.setUser(sessionStruct.getValues().getUser().toXUser());
      LoginContext context = new LoginContext();
      context.setChannelId(sessionStruct.getValues().getLoginProperties().getChannelId());
      context.setCreatedFrom(sessionStruct.getValues().getLoginProperties().getCreatedFrom());
      context.setAppChannel(sessionStruct.getValues().getLoginProperties().getAppChannel());
      context.setBtMerchantId(sessionStruct.getValues().getLoginProperties().getBtMerchantId());
      xLoginInfo.setLoginContext(context);
      return returnSuccessValue(xLoginInfo);
    }else {
      log.info("去向函谷关查询用户信息");
      String checkUrl = xyqbAuthUrl+"/innerapi/is_login";
      ImmutableMap<String, String> headMap = ImmutableMap.of("x-auth-token", token);
      String response = httpService.get(checkUrl, headMap, null);
      log.info("去向函谷关查询用户信息,response:[{}]",response);
      JsonResult result = JSONObject.parseObject(response, JsonResult.class);
      if (result == null || !"0000".equals(result.getCode()) || !"0000".equals(result.getBusinessCode())) {
        return returnErrorValue("用户未登录");
      }
      String phoneNo = ((Map<String, String>) result.getData()).get("phoneNo");
      User user = userService.findByPhoneWithCache(phoneNo);
      XLoginInfo xLoginInfo = new XLoginInfo();
      xLoginInfo.setToken(token);
      xLoginInfo.setUser(user.toXUser());
      LoginContext context = new LoginContext();
      context.setChannelId(null);
      context.setCreatedFrom(user.getRegisteredFrom());
      context.setAppChannel("");
      context.setBtMerchantId(null);
      xLoginInfo.setLoginContext(context);
      return returnSuccessValue(xLoginInfo);
    }
  }*/

  /**
   * 封装返回结果.
   *
   * @param t
   * @param message
   * @param <T>
   * @return
   */
  private <T> UserSysResult<T> returnValue(T t, String message) {
    UserSysResult<T> userSysResult = new UserSysResult();
    userSysResult.setMsg(message);
    /**
     * 无返回结果时,返回的code和BusinessCode均是0001
     */
    if (null == t) {
      userSysResult.setBusinessCode("0001");
      userSysResult.setCode("0001");
    } else {
      userSysResult.setBusinessCode("0000");
      userSysResult.setCode("0000");
    }
    userSysResult.setData(t);
    return userSysResult;
  }

  private <T> UserSysResult<T> returnErrorValue(String message) {
    return returnValue(null, message);
  }

  private <T> UserSysResult<T> returnSuccessValue(T t) {
    return returnValue(t, "");
  }

  /**
   * 注册新用户
   *
   * @return
   */
  private User register(Long registerFrom, String phoneNo, String idNo, String name,Long channelId) {

    String randomCode = String.valueOf(random.nextInt(899999) + 100000);

    String uuid = lkbUserService.registerApp(phoneNo, randomCode);
    Timestamp currentTime = new Timestamp(System.currentTimeMillis());

    log.info("第三方登录用户，保存 User");

    User user = new User();
    if (channelId == 222L) {
      user.setRegisteredFrom(channelId);
    } else {
      user.setRegisteredFrom(registerFrom);
    }
    user.setRegisteredFrom(registerFrom);
    user.setUuid(uuid);
    user.setEnable(true);
    user.setPhoneNo(phoneNo);
    user.setPassword(PasswordUtil.MD5(randomCode + Constants.PASSWORD_SALT));
    user.setUpdatedAt(currentTime);
    user.setCreatedAt(currentTime);
    userRepository.saveAndFlush(user);

    if (registerFrom != 645L && registerFrom !=900L) {
      smsService.sendAfterRegister(phoneNo);
      log.info("第三方登录用户注册成功, registerFrom:{}, phoneNo:{},idNo:{},name:{} 并且已发送短信通知", registerFrom, phoneNo, idNo, name);
    }else if(registerFrom ==900L){
      smsService.sendAfterRegister(phoneNo,"123");
      log.info("第三方(聚美)登录用户注册成功, registerFrom:{}, phoneNo:{},idNo:{},name:{} 并且已发送短信通知", registerFrom, phoneNo, idNo, name);
    }

    if (StringUtils.isNotBlank(idNo) && StringUtils.isNotBlank(name) && ValidationUtil.validateChinese(name)) {
      IdCardInfo cardInfo;
      try {
        cardInfo = idCardService.getIdCardInfoWithExceptions(idNo);

        UserDetail userDetail = new UserDetail();
        userDetail.setIdNo(cardInfo.getIdNo());
        userDetail.setPhoneNo(phoneNo);
        userDetail.setName(name);
        userDetail.setIdType(IdType.ID_CARD);
        userDetail.setCreatedAt(currentTime);
        userDetail.setUpdatedAt(currentTime);
        userDetail.setUserId(user.getId());
        userDetail.setGender(cardInfo.getGender());
        userDetailService.saveUserDetail(userDetail);

        lkbUserService.userUpdate(user.getUuid(), name, idNo);
      } catch (Exception e) {
        log.error("保存 UserDetail 出现异常", e);
      }
    }

    return user;
  }

  @Override
  public UserSysResult<XUserDetail> modifyUserDetail(UserDetailUpdateBean userDetailUpdateBean) {
    return returnErrorValue("MotanUserServiceImpl modifyUserDetail(UserDetailUpdateBean userDetailUpdateBean) need to be finish.");
  }

  @Override
  public UserSysResult<Map<Long, String>> findPhoneByUserIds(List<Long> userIds) {

    if(!CollectionUtils.isEmpty(userIds) && userIds.size()<=500) {
      Map<Long, String> userIdAndPhoneMap = userService.findPhoneByIdsInDb(userIds);
      return returnSuccessValue(userIdAndPhoneMap);
    }else {
      return returnErrorValue("批量查询每次最多进行500条用户信息的查询");
    }
  }
}
