package cn.quantgroup.xyqb.event;

import cn.quantgroup.xyqb.config.data.ContractTemplateConfiguration;
import cn.quantgroup.xyqb.constant.RedisKeyConstant;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.util.RedisLock;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationListener;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 用户注册发送给合同中心事件监听
 */
@Slf4j
@Component
public class BlackHoleRegisteredEventListener implements ApplicationListener<RegisterEvent> {

    @Value("${registered.notify.black.hole.rabbitmq.connection.exchange}")
    private String exchange;
    @Value("${registered.notify.black.hole.rabbitmq.connection.routingKey}")
    private String routingKey;
    @Resource
    private RabbitTemplate registeredNotifyBlackHoleRabbitTemplate;

    @Autowired
    private ContractTemplateConfiguration contractTemplateConfiguration;
    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> redisTemplate;


    @Override
    public void onApplicationEvent(RegisterEvent event) {
        User user = event.getUserRegisterParam().getUser();
        log.info("onApplicationEvent start userId = 【{}】， registerFrom = 【{}】", user.getId(), event);
        LocalDate signDate = LocalDate.now();
        String dateStr = signDate.format(DateTimeFormatter.ofPattern("yyyy年MM月dd日"));
        int day = signDate.getDayOfMonth();
        List<Long> templates = contractTemplateConfiguration.getByTenantIdAndRegisteredFrom(user.getTenantId(), user.getRegisteredFrom());
        if (CollectionUtils.isNotEmpty(templates)) {
            //补签合同，如果需要补签
            String lockKey = RedisKeyConstant.USER_REGISTER_LOGIN_CONTRACT_LOCK.concat(user.getId().toString());
            RedisLock lock = new RedisLock(redisTemplate, lockKey);

            try {
                if (lock.lock()) {
                    //补签合同，如果需要补签 BlackHoleRegisteredEventListener 签署注册和隐私协议
                    //此处补签，对于合同中心，可能会重新签署，后续待合同中心处理并发问题或者重复签署逻辑
                    //Thread.sleep(2000);

                    templates.forEach(templateId -> {

                        String redisKey = String.format(RedisKeyConstant.USER_REGISTER_LOGIN_CONTRACT, user.getId(), templateId);
                        String redisValue = redisTemplate.opsForValue().get(redisKey);
                        if (org.apache.commons.lang3.StringUtils.isEmpty(redisValue)) {
                            return;
                        }

                        JSONArray array = new JSONArray();
                        JSONObject json = new JSONObject();
                        if (templateId == 8 || templateId == 280) {
                            JSONObject fields = new JSONObject();
                            fields.put("phoneNo", user.getPhoneNo());
                            fields.put("genarateDateStr", dateStr);
                            fields.put("day", day);

                            json.put("userId", user.getId());
                            json.put("mustReal", false);
                            json.put("templateId", templateId);
                            json.put("fields", fields);
                            array.add(json);
                        } else {
                            json.put("userId", user.getId());
                            json.put("templateId", templateId);
                            array.add(json);
                        }
                        log.info("注册签合同事件：{},模板：{}",JSONObject.toJSONString(event),templateId);
                        registeredNotifyBlackHoleRabbitTemplate.convertAndSend(exchange, routingKey,
                                array.toString());

                        redisTemplate.opsForValue().set(redisKey, array.toJSONString());
                        redisTemplate.expire(redisKey, 1L, TimeUnit.MINUTES);
                    });
                }
            } catch (Exception e) {
                log.error("注册签合同异常：事件：{},异常：{}",JSONObject.toJSONString(event), e.getMessage(), e);
            } finally {
                lock.unlock();
            }
        }
    }
}
