package cn.quantgroup.xyqb.controller.internal.sms;

import cn.quantgroup.sms.MsgParams;
import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.captcha.CaptchaValidator;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.util.ValidationUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * Created by FrankChow on 15/7/6.
 */
@RestController
@RequestMapping("/api/sms")
public class SmsController {

    private static final Logger LOGGER = LoggerFactory.getLogger(SmsController.class);
    @Autowired
    private ISmsService smsService;

    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> redisTemplate;

    @Value("${sms.is.debug}")
    private boolean smsIsDebug;

    private static final Random random = new Random();

    private static final long EXPIRE_MINUTES = 10;

    /**
     * 短信验证码: for H5
     * 使用 @FPLock 注解并加入自定义限制参数, 做针对手机号的发送次数限制
     */
    @CaptchaValidator
    @RequestMapping("/send_sms_verification_code")
    public JsonResult verifyPhoneNoH5(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
        LOGGER.info("注册-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
        return sendVerificationCode2(phoneNo);
    }

    @CaptchaValidator
    @RequestMapping("/send_reset_code")
    public JsonResult resetPasswordH5(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
        LOGGER.info("重置密码-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
        return sendVerificationCode2(phoneNo);
    }

    /**
     * 快速登陆发送验证码
     *
     * @param phoneNo
     * @param registerFrom
     * @return
     */
    @CaptchaValidator
    @RequestMapping("/send_login_code")
    public JsonResult sendLoginCode(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
        LOGGER.info("快速登陆-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
        return sendVerificationCode2(phoneNo);
    }

    /**
     * 快速注册发送验证码
     *
     * @param phoneNo
     * @param registerFrom
     * @return
     */
    @CaptchaValidator
    @RequestMapping("/send_regist_code")
    public JsonResult sendRegistCode(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
        LOGGER.info("快速注册-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
        return sendVerificationCode2(phoneNo);
    }


    private JsonResult sendVerificationCode2(String phoneNo) {
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult("手机号格式有误", null);
        }

        String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
        long expire = redisTemplate.getExpire(key, TimeUnit.MINUTES);
        if (expire >= EXPIRE_MINUTES - 1) {
            return JsonResult.buildSuccessResult("发送成功", null);
        }
        String randomCode = smsIsDebug ? "000000" : String.valueOf(random.nextInt(899999) + 100000);
        String uniqueId = phoneNo + UUID.randomUUID().toString().replaceAll("-", "");
        List<String> newList = new ArrayList<>();
        newList.add(randomCode);
        /*ConfirmableMsg confirmableMsg = new ConfirmableMsg(
                uniqueId, newList, "1", "1", phoneNo
        );*/
        MsgParams message = new MsgParams(Collections.singletonList(2), phoneNo, "1", "1", Collections.singletonList(randomCode), uniqueId);
        try {
            //smsService.getSmsSender().sendConfirmableMessage(confirmableMsg);
            smsService.getSmsSender().sendMsg(message);
            redisTemplate.opsForValue().set(key, uniqueId + ":" + randomCode, EXPIRE_MINUTES, TimeUnit.MINUTES);
            return JsonResult.buildSuccessResult("发送成功", uniqueId);
        } catch (Exception e) {
            LOGGER.error("发送短信验证码失败");
            return JsonResult.buildErrorStateResult("发送失败", null);
        }
    }

}
