package cn.quantgroup.xyqb.service.v2;

import cn.quantgroup.xyqb.constant.enums.LoginType;
import cn.quantgroup.xyqb.controller.req.v2.LoginReq;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.exception.BizException;
import cn.quantgroup.xyqb.exception.BizExceptionEnum;
import cn.quantgroup.xyqb.exception.UserNotExistException;
import cn.quantgroup.xyqb.model.AuthBean;
import cn.quantgroup.xyqb.model.LoginBean;
import cn.quantgroup.xyqb.model.LoginProperties;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.model.v2.login.BaseLoginParam;
import cn.quantgroup.xyqb.model.v2.login.PhonePasswordLoginParam;
import cn.quantgroup.xyqb.model.v2.login.SMSLoginParam;
import cn.quantgroup.xyqb.service.register.IUserRegisterService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.service.user.ILockIpv4Service;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.session.XyqbSessionContextHolder;
import com.sensorsdata.analytics.javasdk.ISensorsAnalytics;
import com.sensorsdata.analytics.javasdk.bean.EventRecord;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import static cn.quantgroup.xyqb.Constants.VERIFICATION_CODE_FINITE_COUNT_NEW;

/**
 * 短信验证码登录
 */
@Service
@Slf4j
public class SMSLoginStrategy implements LoginStrategy {

    private final ISmsService smsService;
    private final ILockIpv4Service lockIpv4Service;

    private final IUserService userService;
    private final ISessionService sessionService;
    @Value("${sc.is.open:false}")
    private Boolean scIsOpen;

    private final ISensorsAnalytics iSensorsAnalytics;

    private final IUserRegisterService userRegisterService;

    public SMSLoginStrategy(ISmsService smsService, ILockIpv4Service lockIpv4Service, IUserService userService, ISessionService sessionService, ISensorsAnalytics iSensorsAnalytics, IUserRegisterService userRegisterService) {
        this.smsService = smsService;
        this.lockIpv4Service = lockIpv4Service;
        this.userService = userService;
        this.sessionService = sessionService;
        this.iSensorsAnalytics = iSensorsAnalytics;
        this.userRegisterService = userRegisterService;
    }

    @Override
    public Integer getType() {
        return 1101;
    }

    @Override
    public LoginBean login(BaseLoginParam param) {
        SMSLoginParam smsLoginParam = (SMSLoginParam) param;
        SessionStruct sessionStruct = XyqbSessionContextHolder.getXSession();

        // 执行短信验证码检查
        verifyPhoneAndCodeForOnce(smsLoginParam.getPhone(), smsLoginParam.getCode());


        User user = userService.findByPhoneWithCache(smsLoginParam.getPhone(), sessionStruct.getTenantId());
        //需要返回用户是否首次注册
        boolean register = user == null;
        if (user != null && !user.getEnable()) {
            log.info("用户不存在，或者已经注销，phoneNo:{}", smsLoginParam.getPhone());
            throw new BizException(BizExceptionEnum.ERROR_OR_ENABLE_ERROR);
        } else {
            // Service层会负责发送注册消息到LoanVest
            user = userRegisterService.register(sessionStruct.getRegisteredFrom(), smsLoginParam.getPhone(), sessionStruct.getTenantId());
            if (user == null) {
                throw new UserNotExistException("用户未找到");
            }

        }

        if (scIsOpen) {
            try {
                if (!StringUtils.isEmpty(sessionStruct.getScDeviceId())) {
                    iSensorsAnalytics.trackSignUp(user.getUuid(), sessionStruct.getScDeviceId());
                }
                if (!StringUtils.isEmpty(sessionStruct.getTerminal()) && "APP".equals(sessionStruct.getTerminal()) || sessionStruct.getRegisteredFrom() != null && (214 == sessionStruct.getRegisteredFrom() || 217 == sessionStruct.getRegisteredFrom())) {

                    EventRecord userRecord = EventRecord.builder().setDistinctId(user.getUuid()).isLoginId(Boolean.TRUE)
                            .setEventName("App_RegisterEvent")
                            .build();
                    iSensorsAnalytics.track(userRecord);

                    iSensorsAnalytics.flush();
                } else {
                    log.info("老神策埋点{},{}------------", user.getRegisteredFrom(), user.getUuid());
                    EventRecord userRecord = EventRecord.builder().setDistinctId(user.getUuid()).isLoginId(Boolean.TRUE)
                            .setEventName("PD_WUXIEC_UserLoginVccorCash")
                            .addProperty("son_channel_id", user.getRegisteredFrom())
                            .addProperty("parent_channel_id", -1L)
                            .addProperty("vcccash_uuid", user.getUuid())
                            .build();
                    iSensorsAnalytics.track(userRecord);
                    iSensorsAnalytics.flush();
                }
            } catch (Exception e) {
                log.info("神策埋点出现问题", e);
            }
        }


        LoginProperties loginProperties = new LoginProperties(1, sessionStruct.getRegisteredFrom(), sessionStruct.getTenantId());
        AuthBean authBean = sessionService.createSession(user, loginProperties, LoginType.VERIFICATIONCODE.ordinal(), sessionStruct.getTenantId());
        LoginBean loginBean = new LoginBean();
        if (authBean != null) {
            loginBean.setToken(authBean.getToken());
            loginBean.setPhoneNo(authBean.getPhoneNo());
            loginBean.setUuid(authBean.getUuid());
            loginBean.setHasPassword(authBean.isHasPassword());
            loginBean.setUserId(user.getId());
            loginBean.setTenantId(user.getTenantId());
        }
        loginBean.setRegister(register);
        lockIpv4Service.unLockPhone(smsLoginParam.getPhone());

        return loginBean;
    }

    @Override
    public BaseLoginParam checkParam(LoginReq loginReq) {
        SMSLoginParam param = loginReq.getData().toJavaObject(SMSLoginParam.class);
        param.setClientType(loginReq.getClientType());
        validator.validate(param);
        return param;
    }


    /**
     * 验证手机号码逻辑
     *
     * @param phoneNo 手机号码
     * @param code    验证码
     */
    private void verifyPhoneAndCodeForOnce(String phoneNo, String code) {
        if (!smsService.verifyPhoneAndCode(phoneNo, code)) {
            // 是否需要重新发送短信验证码
            if (smsService.needResendCode(phoneNo, VERIFICATION_CODE_FINITE_COUNT_NEW, false)) {
                throw new BizException(BizExceptionEnum.EXPIRE_SMS_CODE);
            }
            smsService.deleteOnlyCodeFromCache(phoneNo);

            log.info("验证码校验失败,phoneNo:{} , verificationCode:{}", phoneNo, code);
            throw new BizException(BizExceptionEnum.ERROR_SMS_CODE);
        } else {
            smsService.deleteCodeFromCache(phoneNo);
        }
    }

}
