package cn.quantgroup.xyqb.controller.middleoffice.user;

import cn.quantgroup.user.enums.UserStatusEnum;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.event.DisableActiveEvent;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.util.TenantUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;

/**
 * 用户信息
 */
@Slf4j
@Api(tags = "middle_office 用户信息API")
@RestController("middleUserController")
@RequestMapping("/middle_office/user")
public class UserController {

    @Resource
    private IUserService userService;

    @Resource
    private ApplicationEventPublisher applicationEventPublisher;

    @ApiOperation(value = "根据userId启用用户", httpMethod = "PUT", notes="根据userId启用用户")
    @PutMapping("/enable/{userId}")
    public JsonResult enable(@PathVariable Long userId) {
        User user = userService.findById(userId);
        if (user == null) {
            //todo 用户不存在，怎么处理
            return JsonResult.buildSuccessResult();
        }
        if (UserStatusEnum.DELETE.ordinal() == user.getEnable()) {
            JsonResult.buildErrorStateResult("用户已销户, 不能启用", null);
        }
        user.setEnable(UserStatusEnum.NORMAL.ordinal());
        userService.saveUser(user);
        //通知消息中心
        applicationEventPublisher.publishEvent(new DisableActiveEvent(this, user));
        return JsonResult.buildSuccessResult();
    }

    @ApiOperation(value = "根据userId禁用用户", httpMethod = "PUT", notes="根据userId禁用用户")
    @PutMapping("/disable/{userId}")
    public JsonResult disable(@PathVariable Long userId) {
        User user = userService.findById(userId);
        if (user == null) {
            //todo 用户不存在，怎么处理
            return JsonResult.buildSuccessResult();
        }
        if (UserStatusEnum.DELETE.ordinal() == user.getEnable()) {
            JsonResult.buildErrorStateResult("用户已销户, 不能冻结", null);
        }
        user.setEnable(UserStatusEnum.FREEZE.ordinal());
        //清理缓存
        userService.saveUser(user);
        //通知消息中心
        applicationEventPublisher.publishEvent(new DisableActiveEvent(this, user));

        return JsonResult.buildSuccessResult();
    }

    @ApiOperation(value = "根据userId查询用户信息", httpMethod = "GET", notes="根据userId查询用户信息")
    @GetMapping("/userId/{userId}")
    public JsonResult user(@PathVariable Long userId) {
        User user = userService.findById(userId);
        //TODO convert to userVO
        return JsonResult.buildSuccessResultGeneric(user);
    }


    @ApiOperation(value = "根据phoneNo查询用户信息", httpMethod = "GET", notes="根据phoneNo查询用户信息")
    @GetMapping("/phoneNo/{phoneNo}")
    public JsonResult user(@PathVariable String phoneNo) {
        User user = userService.findByPhoneInDb(phoneNo);
        //TODO convert to userVO
        return JsonResult.buildSuccessResultGeneric(user);
    }

    @ApiOperation(value = "根据uuid查询用户信息", httpMethod = "GET", notes="根据uuid查询用户信息")
    @GetMapping("/uuid/{uuid}")
    public JsonResult uuid(@PathVariable String uuid) {
        User user = userService.findByUuidWithCache(uuid);
        //TODO convert to userVO
        return JsonResult.buildSuccessResultGeneric(user);
    }

    /**
     * 根据uuid或者userids 获取用户信息
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/29902
     * @param params
     * @return
     */
    @ApiOperation(value = "根据uuid或者userids 获取用户信息", httpMethod = "POST", notes="根据uuid或者userids 获取用户信息")
    @PostMapping("/getByUuidsOrUserIds")
    public JsonResult getByUuidsOrUserIds(@RequestBody Map<String, Object> params) {
        log.info("根据uuid或者userids 获取用户信息 {}", params);
        List<String> vals = (List<String>) params.get("vals");
        Integer type = (Integer) params.get("type");
        String tenantIdStr = String.valueOf(params.getOrDefault("tenantId", TenantUtil.TENANT_DEFAULT));
        Integer tenantId = TenantUtil.TENANT_DEFAULT;
        if (!tenantIdStr.equals("")) {
            tenantId = Integer.valueOf(String.valueOf(params.getOrDefault("tenantId", TenantUtil.TENANT_DEFAULT)));
        }

        if (type == null || (type != 1 && type != 2)) {
            return JsonResult.buildErrorStateResult("type错误", null);
        }
        if (vals.size() > 500) {
            vals = vals.subList(0,500);
        }
        //默认羊小咩租户
        if (TenantUtil.validationTenantIdIsNullOrZero(tenantId)) {
            tenantId = TenantUtil.TENANT_DEFAULT;
        }
        return JsonResult.buildSuccessResultGeneric(userService.findByUuidsOrUserIds(vals, type, tenantId));
    }
}
