package cn.quantgroup.xyqb.service.sms.impl;

import cn.quantgroup.sms.MsgParams;
import cn.quantgroup.sms.SmsSender;
import cn.quantgroup.tech.util.TechEnvironment;
import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.sms.SmsMerchant;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.util.DateUtils;
import cn.quantgroup.xyqb.util.IpUtil;
import cn.quantgroup.xyqb.util.ValidationUtil;
import lombok.Synchronized;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.TimeUnit;


/**
 * @author mengfan.feng
 * @time 2015-07-25 18:47
 */
@Slf4j
@Service
public class SmsServiceImpl implements ISmsService {

    private static SmsSender smsSender = null;
    @Value("${sms.is.debug}")
    private Boolean isDebug;
    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> stringRedisTemplate;

    private static final long EXPIRE_MINUTES = 10;
    private static final String IMAGE_IP_COUNT = "image:ip";
    private static final String IMAGE_PHONE_COUNT = "image:phone";
    private static final String IMAGE_DEVICEID_COUNT = "image:deviceId:";
    /** ip上限 */
    private static final Long IP_MAX_PER_DAY = 5000L;
    /** 手机号短信上限 */
    private static final Long PHONE_MAX_PER_DAY = 20L;
    /** 设备每天上限 */
    private static final Long DEVICE_MAX_PER_DAY = 20L;

    @Override
    @Synchronized
    public SmsSender getSmsSender() {
        if (null == smsSender) {
            smsSender = new SmsSender();
        }
        return smsSender;
    }


    /**
     * 24-“【信用钱包】么么哒，等您好久了~感谢您注册信用钱包，我们来给您送钱啦，点击提交借款申请，万元现金立即到手 s.xyqb.com/a”
     * 1005-“【信用钱包】等您好久了~感谢您的注册，尊享息费5折优惠。登录 s.xyqb.com/a 享更多优惠（合理消费，理性借贷）”
     *
     * @param phoneNo
     */
    @Override
    public void sendAfterRegister(String phoneNo) {
        try {
            MsgParams msgParams =
                    new MsgParams(Collections.singletonList(2), phoneNo, "1", "1005", Collections.emptyList());
            getSmsSender().sendMsg(msgParams);
            //smsSender.sendAndForget(new SendAndForgetMsg(Collections.emptyList(), "24", "1", phoneNo));
            log.info("注册完成，发送短信, phoneNo:{}", phoneNo);
        } catch (Exception e) {
            log.error("注册完成短信发送异常", e);
        }
    }

    @Override
    public void sendAfterRegister(String phoneNo, String contentId) {
        try {
            if (StringUtils.isBlank(contentId)) {
                contentId = "24";
            }
            MsgParams msgParams = new MsgParams(Collections.singletonList(2), phoneNo, "1", contentId,
                    Collections.emptyList());
            getSmsSender().sendMsg(msgParams);
            log.info("注册完成，发送短信, phoneNo:{}", phoneNo);
        } catch (Exception e) {
            log.error("注册完成短信发送异常", e);
        }
    }

    /**
     * 检查验证码是否正确
     */
    @Override
    public boolean verifyPhoneAndCode(String phoneNo, String verificationCode) {
        // 非生产环境直接跳过验证码检验
        if (!TechEnvironment.isPro()) {
            return true;
        }
        String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
        String randomCode = stringRedisTemplate.opsForValue().get(key);
        if (StringUtils.isBlank(randomCode)) {
            return false;
        }
        String[] arr = randomCode.split(":");
        if (arr.length != Constants.VERIFICATION_LEN) {
            return false;
        }
        String uniqueId = arr[0];
        String code = arr[1];
        return confirmSms(verificationCode, uniqueId, code);
    }

    private boolean confirmSms(String smsVerificationCode, String unqiueId, String code) {
        try {
            MsgParams message = new MsgParams(Collections.singletonList(2), unqiueId);
            //MsgParams messageVoice = new MsgParams(Collections.singletonList(4), unqiueId);
            getSmsSender().confirmMsg(message);
            //getSmsSender().confirmMsg(messageVoice);
            //smsSender.confirmSmsResult("1", unqiueId);
            log.info("confirmMsg send success, uniqueId={}", unqiueId);
        } catch (Exception e) {
            log.info("短信验证向短信中心确认失效", e);
        }
        return StringUtils.equals(code, smsVerificationCode);
    }

    @Override
    public boolean needResendCode(String phoneNo) {
        return needResendCode(phoneNo, Constants.VERIFICATION_CODE_FINITE_COUNT,true);
    }

    @Override
    public boolean needResendCode(String phoneNo, Long threshold,boolean isDelTryCount) {

        String verificationCountKey = Constants.REDIS_VERIFICATION_COUNT + phoneNo;
        Long getVerificationCount = stringRedisTemplate.opsForHash().increment(verificationCountKey, Constants.REDIS_VERIFICATION_COUNT, 1);
        boolean needResend = getVerificationCount >= threshold;
        if (needResend) {
            if (isDelTryCount) {
                deleteCodeFromCache(phoneNo);
            } else {
                deleteOnlyCodeFromCache(phoneNo);
            }
        }
        return needResend;
    }

    @Override
    public void deleteCodeFromCache(String phoneNo) {
        String verificationCountKey = Constants.REDIS_VERIFICATION_COUNT + phoneNo;
        stringRedisTemplate.opsForHash().delete(verificationCountKey, Constants.REDIS_VERIFICATION_COUNT);

        deleteOnlyCodeFromCache(phoneNo);
    }

    @Override
    public void deleteOnlyCodeFromCache(String phoneNo) {
        String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
        stringRedisTemplate.delete(key);
    }
    
    @Override
    public JsonResult sendVerificationCode2New(String phoneNo, String randomCode, String deviceId, boolean isApp, String appName, String smsMerchant, String clientIp) {
        if (!ValidationUtil.validatePhoneNo(phoneNo)) {
            return JsonResult.buildErrorStateResult("手机号格式有误", null);
        }
        log.info("请求短信新版本接口:phoneNo:{},deviceId:{},IP:{}", phoneNo, deviceId, clientIp);
        // 手机号计数器
        Long getPhoneVerificationCount = stringRedisTemplate.opsForHash().increment(Constants.REDIS_SMS_CODE_COUNT, phoneNo, 1);
        stringRedisTemplate.expire(Constants.REDIS_SMS_CODE_COUNT, DateUtils.getSeconds(), TimeUnit.SECONDS);
        // 设备号计数器
        Long getDeviceVerificationCount = 0L;
        if (StringUtils.isNotBlank(deviceId)) {
            getDeviceVerificationCount = stringRedisTemplate.opsForHash().increment(Constants.REDIS_SMS_DEVICE_COUNT, deviceId, 1);
            stringRedisTemplate.expire(Constants.REDIS_SMS_DEVICE_COUNT, DateUtils.getSeconds(), TimeUnit.SECONDS);
        }
        // IP计数器
        Long getIPVerificationCount = 0L;
        if (StringUtils.isNotBlank(clientIp)) {
            getIPVerificationCount = stringRedisTemplate.opsForHash().increment(Constants.REDIS_SMS_IP_COUNT, clientIp, 1);
            stringRedisTemplate.expire(Constants.REDIS_SMS_IP_COUNT, DateUtils.getSeconds(), TimeUnit.SECONDS);
        }
        // 手机号上限检查
        if (getPhoneVerificationCount > PHONE_MAX_PER_DAY) {
            log.info("您手机号已经达到获取今天短信验证码上限:phoneNo:{},count:{}", phoneNo, getPhoneVerificationCount);
            return JsonResult.buildErrorStateResult("今天已获取20次短信验证码，请使用语音验证码或明天再试", null);
        }
        // 设备号上限检查
        if (getDeviceVerificationCount > DEVICE_MAX_PER_DAY) {
            log.info("您设备已经达到获取今天验证码上限:deviceId:{},count:{}", deviceId, getDeviceVerificationCount);
            return JsonResult.buildErrorStateResult("您设备已经达到获取今天验证码上限", null);
        }
        // IP上限检查
        if (!IpUtil.whiteOf(clientIp) && getIPVerificationCount > IP_MAX_PER_DAY) {
            log.info("您当前ip已经达到获取今天短信验证码上限:ip:{},count:{}", clientIp, getIPVerificationCount);
            return JsonResult.buildErrorStateResult("您当前ip已经达到获取今天短信验证码上限", null);
        }

        String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
        long expire = stringRedisTemplate.getExpire(key, TimeUnit.MINUTES);
        if (expire >= EXPIRE_MINUTES - 1) {
            log.info("sendVerificationCode2New1分钟内不能重复获取验证码:phoneNo:{},deviceId:{},ip:{}", phoneNo, deviceId, clientIp);
            return JsonResult.buildErrorStateResult("1分钟内不能重复获取验证码", null);
        }
        String uniqueId = phoneNo + UUID.randomUUID().toString().replaceAll("-", "");
        List<String> newList = new ArrayList<>();
        newList.add(randomCode);
        MsgParams message = new MsgParams.Builder()
                .typeList(Collections.singletonList(2))
                .phoneNo(phoneNo)
                .merchantId(SmsMerchant.of(smsMerchant).getMerchantId())
                .contentId(SmsMerchant.of(smsMerchant).getContentId())
                .uniqueId(uniqueId)
                .contentArgs(Collections.singletonList(randomCode))
                .channel(appName)
                .build();

        try {
            this.getSmsSender().sendMsg(message);
            stringRedisTemplate.opsForValue().set(key, uniqueId + ":" + randomCode, EXPIRE_MINUTES, TimeUnit.MINUTES);
            //删除用户重置密码，多次错误逻辑
            deleteRetSendCode(phoneNo);
            if (isApp && needImageVlidate(clientIp, deviceId, phoneNo)) {
                return JsonResult.buildSuccessResult("发送成功", uniqueId, 3L);
            }
            log.info("sendVerificationCode2New获取短信成功:phone:{},deviceId:{},ip:{}", phoneNo, deviceId, clientIp);
            return JsonResult.buildSuccessResult("发送成功", uniqueId);
        } catch (Exception e) {
            log.error("发送短信验证码失败:phone:{},deviceId:{},ip:{}", phoneNo, deviceId, clientIp);
            return JsonResult.buildErrorStateResult("发送失败", null);
        }
    }

    /**
     * 判断下次是否提示图形验证码
     *
     * @param clientIp
     * @param deviceId
     * @param phoneNo
     * @return
     */
    private boolean needImageVlidate(String clientIp, String deviceId, String phoneNo) {
        boolean need = false;
        String countIP = stringRedisTemplate.opsForValue().get(IMAGE_IP_COUNT + clientIp);
        String countDeviceId = stringRedisTemplate.opsForValue().get(IMAGE_DEVICEID_COUNT + deviceId);
        String countPhoneNo = stringRedisTemplate.opsForValue().get(IMAGE_PHONE_COUNT + phoneNo);
        Long ip = StringUtils.isBlank(countIP) ? 1L : Long.valueOf(countIP);
        Long devId = StringUtils.isBlank(countDeviceId) ? 1L : Long.valueOf(countDeviceId);
        Long phNo = StringUtils.isBlank(countPhoneNo) ? 1L : Long.valueOf(countPhoneNo);
        if (ip >= Constants.Image_Need_Count || devId >= Constants.Image_Need_Count || phNo >= Constants.Image_Need_Count) {
            need = true;
        }
        return need;
    }

    /**
     * 删除用户重置密码时短信验证错误
     *
     * @param phoneNo
     */
    private void deleteRetSendCode(String phoneNo) {
        String verificationCountKey = Constants.REDIS_VERIFICATION_COUNT + phoneNo;
        stringRedisTemplate.opsForHash().delete(verificationCountKey, Constants.REDIS_VERIFICATION_COUNT);
    }
}
