package cn.quantgroup.xyqb.controller.internal.sms;

import cn.quantgroup.sms.MsgParams;
import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.captcha.CaptchaValidator;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.util.DateUtils;
import cn.quantgroup.xyqb.util.ValidationUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * Created by FrankChow on 15/7/6.
 */
@RestController
@RequestMapping("/api/sms")
public class SmsController {

  private static final Logger LOGGER = LoggerFactory.getLogger(SmsController.class);
  private static final Random random = new Random();
  private static final long EXPIRE_MINUTES = 10;
  @Autowired
  private ISmsService smsService;
  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> redisTemplate;
  @Value("${sms.is.debug}")
  private boolean smsIsDebug;

  /**
   * 短信验证码: for H5
   * 使用 @FPLock 注解并加入自定义限制参数, 做针对手机号的发送次数限制
   */
  @CaptchaValidator
  @RequestMapping("/send_sms_verification_code")
  public JsonResult verifyPhoneNoH5(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
    LOGGER.info("注册-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2(phoneNo);
  }

  @CaptchaValidator
  @RequestMapping("/send_reset_code")
  public JsonResult resetPasswordH5(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
    LOGGER.info("重置密码-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2(phoneNo);
  }

  /**
   * 短信验证码: for H5
   * 使用 @FPLock 注解并加入自定义限制参数, 做针对手机号的发送次数限制
   */
  @CaptchaValidator
  @RequestMapping("/send_sms_verification_code_voice")
  public JsonResult verifyPhoneNoH5New(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom,
      String usage) {
    if (StringUtils.isEmpty(usage) || !"4".equals(usage)) {
      LOGGER.error("参数校验失败,用户注册语音验证码usage参数为{}", usage);
      return JsonResult.buildErrorStateResult("参数校验失败.", null);
    }
    LOGGER.info("注册-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2Voice(phoneNo, usage);
  }

  @CaptchaValidator
  @RequestMapping("/send_reset_code_voice")
  public JsonResult resetPasswordH5New(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom,
      String usage) {
    if (StringUtils.isEmpty(usage) || !"5".equals(usage)) {
      LOGGER.error("参数校验失败,重置密码语音验证码usage参数为{}", usage);
      return JsonResult.buildErrorStateResult("参数校验失败.", null);
    }
    LOGGER.info("重置密码-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2Voice(phoneNo, usage);
  }

  /**
   * 快速登陆发送验证码
   */
  @CaptchaValidator
  @RequestMapping("/send_login_code_voice")
  public JsonResult sendLoginCodeNew(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom,
      String usage) {
    if (StringUtils.isEmpty(usage) || !"6".equals(usage)) {
      LOGGER.error("参数校验失败,用户登录语音验证码usage参数为{}", usage);
      return JsonResult.buildErrorStateResult("参数校验失败.", null);
    }
    LOGGER.info("快速登陆-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2Voice(phoneNo, usage);
  }

  /**
   * 快速登陆发送验证码
   */
  @CaptchaValidator
  @RequestMapping("/send_login_code")
  public JsonResult sendLoginCode(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
    LOGGER.info("快速登陆-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2(phoneNo);
  }

  /**
   * 快速注册发送验证码
   */
  @CaptchaValidator
  @RequestMapping("/send_regist_code")
  public JsonResult sendRegistCode(@RequestParam String phoneNo, @RequestParam(required = false) String registerFrom) {
    LOGGER.info("快速注册-发送验证码, phoneNo:{}, registerFrom:{}", phoneNo, registerFrom);
    return sendVerificationCode2(phoneNo);
  }

  private JsonResult sendVerificationCode2(String phoneNo) {
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return JsonResult.buildErrorStateResult("手机号格式有误", null);
    }

    String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
    long expire = redisTemplate.getExpire(key, TimeUnit.MINUTES);
    if (expire >= EXPIRE_MINUTES - 1) {
      return JsonResult.buildSuccessResult("发送成功", null);
    }
    String randomCode = smsIsDebug ? "0000" : String.valueOf(random.nextInt(8999) + 1000);
    String uniqueId = phoneNo + UUID.randomUUID().toString().replaceAll("-", "");
    List<String> newList = new ArrayList<>();
    newList.add(randomCode);
        /*ConfirmableMsg confirmableMsg = new ConfirmableMsg(
                uniqueId, newList, "1", "1", phoneNo
        );*/
    MsgParams message = new MsgParams(Collections.singletonList(2), phoneNo, "1", "1", Collections.singletonList(randomCode), uniqueId);
    try {
      //smsService.getSmsSender().sendConfirmableMessage(confirmableMsg);
      smsService.getSmsSender().sendMsg(message);
      redisTemplate.opsForValue().set(key, uniqueId + ":" + randomCode, EXPIRE_MINUTES, TimeUnit.MINUTES);
      return JsonResult.buildSuccessResult("发送成功", uniqueId);
    } catch (Exception e) {
      LOGGER.error("发送短信验证码失败");
      return JsonResult.buildErrorStateResult("发送失败", null);
    }
  }

  private JsonResult sendVerificationCode2Voice(String phoneNo, String usage) {
    String verificationCountKey = Constants.REDIS_VOICE_CODE_COUNT + phoneNo;
    Long getVerificationCount = redisTemplate.opsForHash().increment(verificationCountKey, usage.toString(), 1);
    if (getVerificationCount > 5) {
      return JsonResult.buildErrorStateResult("今天已获取5次语音验证码，请使用短信验证码或明天再试", null);
    }
    redisTemplate.expire(verificationCountKey, DateUtils.getSeconds(), TimeUnit.SECONDS);
    if (!ValidationUtil.validatePhoneNo(phoneNo)) {
      return JsonResult.buildErrorStateResult("手机号格式有误", null);
    }
    String key = Constants.REDIS_PREFIX_VERIFICATION_CODE + phoneNo;
    long expire = redisTemplate.getExpire(key, TimeUnit.MINUTES);
    if (expire >= EXPIRE_MINUTES - 1) {
      return JsonResult.buildSuccessResult("发送成功", null);
    }
    String randomCode = smsIsDebug ? "0000" : String.valueOf(random.nextInt(8999) + 1000);
    String uniqueId = phoneNo + UUID.randomUUID().toString().replaceAll("-", "");
    MsgParams message = new MsgParams(Collections.singletonList(4), phoneNo, "1", "4", Collections.singletonList(randomCode), uniqueId);
    try {
      smsService.getSmsSender().sendMsg(message);
      redisTemplate.opsForValue().set(key, uniqueId + ":" + randomCode, EXPIRE_MINUTES, TimeUnit.MINUTES);
      return JsonResult.buildSuccessResult("发送成功", uniqueId);
    } catch (Exception e) {
      LOGGER.error("发送语音短信验证码失败");
      return JsonResult.buildErrorStateResult("发送失败", null);
    }
  }
}
