package cn.quantgroup.xyqb.service.v2;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.config.data.WechatConfiguration;
import cn.quantgroup.xyqb.constant.enums.LoginType;
import cn.quantgroup.xyqb.controller.req.v2.LoginReq;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.WechatUserInfo;
import cn.quantgroup.xyqb.exception.BizException;
import cn.quantgroup.xyqb.exception.BizExceptionEnum;
import cn.quantgroup.xyqb.model.*;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.model.v2.login.BaseLoginParam;
import cn.quantgroup.xyqb.model.v2.login.WechatMiniLoginParam;
import cn.quantgroup.xyqb.model.v2.login.WechatMiniPhoneLoginParam;
import cn.quantgroup.xyqb.remote.WechatRemoteService;
import cn.quantgroup.xyqb.repository.IUserRepository;
import cn.quantgroup.xyqb.repository.IWeChatUserRepository;
import cn.quantgroup.xyqb.service.register.IUserRegisterService;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.session.XyqbSessionContextHolder;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;

/**
 * 微信小程序手机号码登录
 */
@Service
@Slf4j
public class WechatMiniPhoneLoginStrategy implements LoginStrategy {
    private final WechatRemoteService wechatRemoteService;
    private final WechatConfiguration wechatConfiguration;

    private final IWeChatUserRepository weChatUserRepository;
    private final ISessionService sessionService;

    private final IUserRepository userRepository;
    private final IUserRegisterService userRegisterService;

    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> redisTemplate;
    private final Gson gson;

    public WechatMiniPhoneLoginStrategy(WechatRemoteService wechatRemoteService, WechatConfiguration wechatConfiguration, IWeChatUserRepository weChatUserRepository, ISessionService sessionService, IUserRepository userRepository, IUserRegisterService userRegisterService, Gson gson) {
        this.wechatRemoteService = wechatRemoteService;
        this.wechatConfiguration = wechatConfiguration;
        this.weChatUserRepository = weChatUserRepository;
        this.sessionService = sessionService;
        this.userRepository = userRepository;
        this.userRegisterService = userRegisterService;
        this.gson = gson;
    }

    @Override
    public Integer getType() {
        return 2102;
    }

    @Override
    public LoginBean login(BaseLoginParam param) {
        WechatMiniPhoneLoginParam loginParam = (WechatMiniPhoneLoginParam) param;
        SessionStruct sessionStruct = XyqbSessionContextHolder.getXSession();

        //1、微信登录逻辑
        WechatConfigBean wechatConfigBean = wechatConfiguration.getByAppIdAndTenantId(loginParam.getAppId(), sessionStruct.getTenantId());
        String response = wechatRemoteService.login(wechatConfigBean.getAppId(), wechatConfigBean.getAppSecret(), ((WechatMiniLoginParam) param).getCode());

        Type resultType = new TypeToken<Map<String, String>>() {
        }.getType();

        Map<String, String> responseMap = gson.fromJson(response, resultType);

        String openid, sessionKey = null, unionId = null;


        assert responseMap != null;
        if (responseMap.containsKey("errcode") && Integer.parseInt(responseMap.get("errcode")) != 0) {
            log.error("code:{},errcode:{},errmsg:{}", ((WechatMiniLoginParam) param).getCode(), responseMap.get("errcode"),
                    responseMap.get("errmsg"));
            throw new BizException(BizExceptionEnum.ERROR_WECHAT_LOGIN, responseMap.get("errmsg"));
        } else {
            openid = responseMap.get("openid");
            if (responseMap.containsKey("session_key")) {
                sessionKey = responseMap.get("session_key");
            }
            if (responseMap.containsKey("unionid")) {
                unionId = responseMap.get("unionid");
            }
        }
        Map<String, String> phoneParam = new HashMap<>();
        boolean register = false;
        User user;
        phoneParam.put("code", loginParam.getPhoneCode());
        String accessToken = redisTemplate.opsForValue().get(Constants.WECHAT_ACCESS_TOKEN_REDIS + loginParam.getAppId());

        WechatPhoneBean wechatPhoneBean = wechatRemoteService.getUserPhoneNumber(accessToken, phoneParam);

        if (wechatPhoneBean.getErrcode() == 0) {
            //1、查询库中是否存在用户
            user = userRepository.findByPhoneNoAndTenantId(wechatPhoneBean.getPhone_info().getPurePhoneNumber(), sessionStruct.getTenantId());
            //2、如果没有用户，新建一条
            if (user == null) {
                user = userRegisterService.register(sessionStruct.getRegisteredFrom(), wechatPhoneBean.getPhone_info().getPurePhoneNumber(), sessionStruct.getTenantId());
            }
            if (user == null) {
                throw new BizException(BizExceptionEnum.UN_EXIT_USER);
            }
            WechatUserInfo wechatUserInfo = weChatUserRepository.findByOpenIdAndAppIdAndTenantId(openid, loginParam.getAppId(), sessionStruct.getTenantId());
            //如果微信用户为空，则补偿一条
            if (wechatUserInfo == null) {
                register = true;
                wechatUserInfo = new WechatUserInfo();
                wechatUserInfo.setOpenId(openid);
                wechatUserInfo.setUnionId(unionId);
                wechatUserInfo.setAppId(loginParam.getAppId());
                wechatUserInfo.setTenantId(sessionStruct.getTenantId());
                wechatUserInfo.setPhoneNo(user.getPhoneNo());
                wechatUserInfo.setUserId(user.getId());
                wechatUserInfo.setSessionKey(sessionKey);
            } else {
                wechatUserInfo.setSessionKey(sessionKey);
            }
            weChatUserRepository.save(wechatUserInfo);
        } else {
            throw new BizException(BizExceptionEnum.ERROR_WECHAT_LOGIN, wechatPhoneBean.getErrmsg());
        }

        LoginProperties loginProperties = new LoginProperties(1, sessionStruct.getRegisteredFrom(), sessionStruct.getTenantId());
        AuthBean authBean = sessionService.createSession(user, loginProperties, LoginType.WECHATLOGIN.ordinal(), sessionStruct.getTenantId());
        LoginBean loginBean = new LoginBean();
        if (authBean != null) {
            loginBean.setToken(authBean.getToken());
            loginBean.setPhoneNo(authBean.getPhoneNo());
            loginBean.setUuid(authBean.getUuid());
            loginBean.setHasPassword(authBean.isHasPassword());
            loginBean.setUserId(user.getId());
            loginBean.setTenantId(user.getTenantId());
        }
        loginBean.setRegister(register);

        return loginBean;
    }

    @Override
    public BaseLoginParam checkParam(LoginReq loginReq) {
        WechatMiniPhoneLoginParam param = loginReq.getData().toJavaObject(WechatMiniPhoneLoginParam.class);
        validator.validate(param);
        return param;
    }

}
