package cn.quantgroup.xyqb.controller.external.user;

import cn.quantgroup.xyqb.entity.*;
import cn.quantgroup.xyqb.entity.enumerate.*;
import cn.quantgroup.xyqb.model.*;
import cn.quantgroup.xyqb.service.auth.IIdCardService;
import cn.quantgroup.xyqb.service.user.*;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.sql.Timestamp;
import java.text.ParseException;
import java.util.Collections;
import java.util.List;

/**
 * Created by Miraculous on 2016/12/19.
 */
@RestController
@RequestMapping("/innerapi")
public class InnerController {

    private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(InnerController.class);

    @Autowired
    private IUserService userService;
    @Autowired
    private IUserDetailService userDetailService;
    @Autowired
    private IIdCardService idCardService;
    @Autowired
    private IUserExtInfoService userExtInfoService;
    @Autowired
    private IContactService contactService;
    @Autowired
    private IAddressService addressService;


    @RequestMapping("/user/search/phoneNo")
    public JsonResult findByPhoneNo(String phoneNo) {
        User user = userService.findByPhoneInDb(phoneNo);
        if (user == null) {
            return JsonResult.buildErrorStateResult("", null);
        }
        UserRet userRet = UserRet.getUserRet(user);
        return JsonResult.buildSuccessResult("", userRet);
    }

    @RequestMapping("/user/search/uuid")
    public JsonResult findByUuid(String uuid) {
        User user = userService.findByUuidInDb(uuid);
        if (user == null) {
            return JsonResult.buildErrorStateResult("", null);
        }
        UserRet userRet = UserRet.getUserRet(user);
        return JsonResult.buildSuccessResult("", userRet);
    }

    @RequestMapping("/user/save")
    public JsonResult saveUser(
            String phoneNo, Long registeredFrom, Long createdAt, Long updatedAt,
            String password, String uuid) {
        //参数验证
        if(StringUtils.isBlank(phoneNo)){
            return JsonResult.buildErrorStateResult("用户手机号不能为空.", null);
        }
        if(registeredFrom == null){
            registeredFrom = 0L;
        }
        if(StringUtils.isBlank(password)){
            password = "";
        }
        if(StringUtils.isBlank(uuid)){
            return JsonResult.buildErrorStateResult("用户uuid为空.", null);
        }
        if(createdAt == 0L || updatedAt == 0L){
            createdAt = System.currentTimeMillis();
            updatedAt = System.currentTimeMillis();
        }
        User user = userService.findByPhoneWithCache(phoneNo);
        if (user == null) {
            user = new User();
        }
        user.setPhoneNo(phoneNo);
        user.setCreatedAt(new Timestamp(createdAt));
        user.setUpdatedAt(new Timestamp(updatedAt));
        user.setEnable(true);
        user.setRegisteredFrom(registeredFrom);
        user.setUuid(uuid);
        user.setPassword(password);
        user = userService.saveUser(user);
        UserRet userRet = null;
        if(user != null){
            userRet = UserRet.getUserRet(user);
        }
        return JsonResult.buildSuccessResult(null, userRet);
    }

    /**
     * 保存用户详细信息
     * @param userId
     * @param phoneNo
     * @param name
     * @param idNo
     * @param email
     * @return
     */
    @RequestMapping("/user_detail/save")
    public JsonResult saveUserDetail(Long userId, String phoneNo, String name, String idNo,
                                     String email, Long id){
        //参数验证
        if(userId == null || userId == 0L){
            return JsonResult.buildErrorStateResult("用户id为空.", null);
        }
        if(StringUtils.isBlank(phoneNo)){
            return JsonResult.buildErrorStateResult("用户手机号为空.", null);
        }
        if(StringUtils.isBlank(name)){
            return JsonResult.buildErrorStateResult("用户名为空.", null);
        }
        if(StringUtils.isBlank(idNo)){
            return JsonResult.buildErrorStateResult("用户身份证为空.", null);
        }
        UserDetail userDetail = new UserDetail();
        if(id != null && id > 0){
            userDetail.setId(id);
        }
        userDetail.setUserId(userId);
        userDetail.setName(name);
        userDetail.setPhoneNo(phoneNo);
        userDetail.setIdNo(idNo);
        Timestamp time = new Timestamp(System.currentTimeMillis());
        userDetail.setCreatedAt(time);
        userDetail.setUpdatedAt(time);
        userDetail.setIdType(IdType.ID_CARD);
        try {
            userDetail.setGender(idCardService.getIdCardInfo(idNo).getGender());
        } catch (ParseException e) {
            LOGGER.error("根据身份证获取性别出错,错误信息:" + e);
            return JsonResult.buildErrorStateResult(null, null);
        }
        userDetail.setEmail(email);
        userDetail = userDetailService.saveUserDetail(userDetail);
        if(userDetail != null){
            return JsonResult.buildSuccessResult(null, UserDetailRet.getUserDetail(userDetail));
        }
        return JsonResult.buildErrorStateResult("",null);
    }

    /**
     * 根据用户id查询用户的详细信息
     * @param userId
     * @return
     */
    @RequestMapping("/user_detail/search/userId")
    public JsonResult findUserDetailByUserId(Long userId){
        UserDetail userDetail = userDetailService.findByUserId(userId);
        if(userDetail != null){
            return JsonResult.buildSuccessResult(null, UserDetailRet.getUserDetail(userDetail));
        }
        return JsonResult.buildErrorStateResult("", null);
    }

    @RequestMapping("/user/search/userId")
    public JsonResult findUserByUserId(Long userId){
        User user = userService.findById(userId);
        if(user != null){
            return JsonResult.buildSuccessResult(null, UserRet.getUserRet(user));
        }
        return JsonResult.buildErrorStateResult("", null);
    }

    @RequestMapping("/user_detail/search/phone")
    public JsonResult findUserDetailByPhone(String phoneNo){
        UserDetail userDetail = userDetailService.findByPhoneNo(phoneNo);
        if(userDetail != null){
            return JsonResult.buildSuccessResult(null, UserDetailRet.getUserDetail(userDetail));
        }
        return JsonResult.buildErrorStateResult("", null);
    }

    @RequestMapping("/user_detail/update/qq")
    public JsonResult updateUserQQ(String qq, Long userId){
        if(StringUtils.isEmpty(qq) || userId == null || userId == 0L){
            return JsonResult.buildErrorStateResult("参数校验失败,qq或用户id为空", null);
        }
        userDetailService.updateUserQQ(userId, qq);
        return JsonResult.buildSuccessResult(null, null);
    }

    @RequestMapping("/contact/search/user_id")
    public JsonResult findContactsByUserId(Long userId) {
        if (null == userId) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        List<Contact> contacts = contactService.findByUserId(userId);
        if(null == contacts || contacts.size() == 0) {
            return JsonResult.buildSuccessResult(null, Collections.emptyList());
        }
        return JsonResult.buildSuccessResult(null, ContactRet.contacts2ContactRets(contacts));
    }

    @RequestMapping("/contact/save/contacts")
    public JsonResult save2Contact(Long userId, @RequestParam(value = "contacts") String contactsStr) {
        if (StringUtils.isEmpty(contactsStr)) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        if (userId == null) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        List<Contact> userContact = contactService.findByUserId(userId);
        if(userContact != null && userContact.size() >= 2) {
            return JsonResult.buildErrorStateResult("用户联系人最多保存两条记录.", null);
        }
        List<Contact> contacts = JSONObject.parseObject(contactsStr, new TypeReference<List<Contact>>() {});
        if (CollectionUtils.isEmpty(contacts)) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        Timestamp now = new Timestamp(System.currentTimeMillis());
        for (Contact c : contacts) {
            c.setId(null);
            c.setUserId(userId);
            c.setCreatedAt(now);
            c.setUpdateAt(now);
        }
        List<Contact> result = contactService.save(contacts);
        return JsonResult.buildSuccessResult(null, ContactRet.contacts2ContactRets(result));
    }

    @RequestMapping("/address/search/user_id")
    public JsonResult findAddressByUserId(Long userId) {
        if (userId == null) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        Address address = addressService.findByUserId(userId);
        if (address == null) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        return JsonResult.buildSuccessResult(null, AddressRet.address2AddressRet(address));
    }

    @RequestMapping("/address/save")
    public JsonResult saveAddress(
            Long userId, Long provinceCode, Long cityCode, String city,
            Long districtCode, String district, String address, String province) {
        if (userId == null || provinceCode == null || cityCode == null) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        Address addressObj = addressService.findByUserId(userId);
        if (address == null) {
            addressObj = new Address();
            Timestamp now = new Timestamp(System.currentTimeMillis());
            addressObj.setCreatedAt(now);
            addressObj.setUpdateAt(now);
        }
        addressObj.setUserId(userId);
        addressObj.setProvinceCode(provinceCode);
        addressObj.setCityCode(cityCode);
        addressObj.setCity(city);
        addressObj.setDistrictCode(districtCode);
        addressObj.setDistrict(district);
        addressObj.setAddress(address);
        addressObj.setProvince(province);
        addressObj = addressService.save(addressObj);
        return JsonResult.buildSuccessResult(null, AddressRet.address2AddressRet(addressObj));
    }


    @RequestMapping("/user_ext_info/update")
    public JsonResult updateMarryStatus(
            Long userId, IncomeEnum incomeEnum, IncomeRangeEnum incomeRangeEnum,
            OccupationEnum occupationEnum, EducationEnum educationEnum, Boolean hasCar,
            Boolean hasSocialSecurity, Boolean hasHouse, Boolean hasCreditCard, MaritalStatus maritalStatus) {
        if (null == userId) {
            return JsonResult.buildErrorStateResult("用户ID不能为空", null);
        }
        UserExtInfo info = userExtInfoService.findByUserId(userId);
        if (info == null) {
            Timestamp now = new Timestamp(System.currentTimeMillis());
            info = new UserExtInfo();
            info.setUserId(userId);
            info.setCreatedAt(now);
            info.setUpdateAt(now);
        }
        if (incomeEnum != null) {
            info.setIncomeEnum(incomeEnum);
        }
        if (incomeRangeEnum != null) {
            info.setIncomeRangeEnum(incomeRangeEnum);
        }
        if (occupationEnum != null) {
            info.setOccupationEnum(occupationEnum);
        }
        if (educationEnum != null) {
            info.setEducationEnum(educationEnum);
        }
        if (hasCar != null) {
            info.setHasCar(hasCar);
        }
        if (hasSocialSecurity != null) {
            info.setHasCreditCard(hasCreditCard);
        }
        if (hasHouse != null) {
            info.setHasHouse(hasHouse);
        }
        if (maritalStatus != null) {
            info.setMarryStatus(maritalStatus);
        }
        info = userExtInfoService.save(info);
        return JsonResult.buildSuccessResult(null, UserExtInfoRet.getUserExtInfoRet(info));
    }

    @RequestMapping("/user_detail/search_list")
    public JsonResult searchUserDetailList(String name, String phoneNo, String idNo) {
        List<UserDetail> userDetails = userDetailService.searchUserDetailList(name, phoneNo, idNo);
        return JsonResult.buildSuccessResult("success", userDetails);
    }

    @RequestMapping("/user_ext_info/search/user_id")
    public JsonResult searchUserExtInfoByUserId(Long userId) {
        if (userId == null) {
            return JsonResult.buildErrorStateResult("userId不能为空", null);
        }
        UserExtInfo userExtInfo = userExtInfoService.findByUserId(userId);
        if (userExtInfo == null) {
            return JsonResult.buildErrorStateResult(null, null);
        }
        return JsonResult.buildSuccessResult("success", UserExtInfoRet.getUserExtInfoRet(userExtInfo));
    }

}