package cn.quantgroup.xyqb.service.user.impl;

import cn.quantgroup.user.enums.Relation;
import cn.quantgroup.xyqb.entity.Contact;
import cn.quantgroup.xyqb.repository.IContactRepository;
import cn.quantgroup.xyqb.service.user.IContactService;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.cache.annotation.Caching;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.sql.Timestamp;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * Created by Miraculous on 2017/1/3.
 */
@Service
public class ContactServiceImpl implements IContactService {
    private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(ContactServiceImpl.class);
    @Autowired
    private IContactRepository contactRepository;

    @Override
    @Cacheable(value = "contact", key = "'contact:' + #trim + #userId", unless = "#result == null or #result.size() == 0", cacheManager = "cacheManager")
    public List<Contact> findByUserId(Long userId, boolean trim) {
        List<Contact> contacts = contactRepository.findByUserId(userId);
        if (trim) {
            trim(contacts);
        }
        // 按修改时间倒序排列
        if(!CollectionUtils.isEmpty(contacts)){
            Collections.sort(contacts, (o1, o2) -> {
                Date paidAt1 = o1.getUpdateAt();
                Date paidAt2 = o2.getUpdateAt();
                if(paidAt1 == null && paidAt2 == null) {
                    return 0;
                }
                if(paidAt1 == null) {
                    return -1;
                }
                if(paidAt2 == null) {
                    return -1;
                }
                return paidAt1.compareTo(paidAt2);
            });
        }
        return contacts;
    }

    /**
     * 过滤掉非法联系人记录
     *
     * @param contacts - 包含待清除记录的联系人列表
     */
    private void trim(List<Contact> contacts) {
        if (CollectionUtils.isEmpty(contacts)) {
            return;
        }
        contacts.removeIf(contact -> !contact.valid());
    }

    @Override
    public Contact findById(Long id) {
        return contactRepository.findOne(id);
    }

    @Override
    @Caching(evict = {
            @CacheEvict(value = "contact", key = "'contact:true' + #userId", cacheManager = "cacheManager"),
            @CacheEvict(value = "contact", key = "'contact:false' + #userId", cacheManager = "cacheManager")})
    public List<Contact> save(Long userId, List<Contact> contacts) {
        if (userId == null) {
            return null;
        }
        // 合并当前用户列表到更新列表
        mergeContacts(userId, contacts);
        return contactRepository.save(contacts);
    }

    @Override
    @Caching(evict = {
            @CacheEvict(value = "contact", key = "'contact:true' + #contact.userId", cacheManager = "cacheManager"),
            @CacheEvict(value = "contact", key = "'contact:false' + #contact.userId", cacheManager = "cacheManager")})
    public Contact save(Contact contact) {
        return contactRepository.save(contact);
    }

    @Override
    public Contact saveContact(String name, String phoneNo, Relation relation, Contact contact) {
        if (StringUtils.isNotBlank(name)) {
            contact.setName(name);
        }
        if (StringUtils.isNotBlank(phoneNo)) {
            contact.setPhoneNo(phoneNo);
        }
        if (null != relation) {
            contact.setRelation(relation);
        }
        LOGGER.info("修改前联系人信息:{}", contact);
        Timestamp now = new Timestamp(System.currentTimeMillis());
        contact.setUpdateAt(now);
        contact = save(contact);
        return contact;
    }

    /**
     * 合并当前用户列表到更新列表
     *
     * @param userId   - 用户主键
     * @param contacts - 新联系人列表
     */
    private void mergeContacts(Long userId, List<Contact> contacts) {
        Timestamp now = new Timestamp(System.currentTimeMillis());
        for (int i = 0; i < contacts.size(); i++) {
            Contact c = contacts.get(i);
            c.setUserId(userId);
            if(Objects.isNull(c.getRelation())){
                c.setRelation(Relation.OTHER);
            }
            if(Objects.isNull(c.getId())){
                c.setCreatedAt(now);
            }
            c.setUpdateAt(now);
        }
    }
}
