package cn.quantgroup.xyqb.service.user.impl;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserDetail;
import cn.quantgroup.xyqb.entity.UserHashMapping;
import cn.quantgroup.xyqb.model.UserInfo;
import cn.quantgroup.xyqb.repository.IUserHashMappingRepository;
import cn.quantgroup.xyqb.repository.IUserRepository;
import cn.quantgroup.xyqb.service.user.IUserDetailService;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.util.HashUtil;
import cn.quantgroup.xyqb.util.PasswordUtil;
import cn.quantgroup.xyqb.util.ValidationUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.sql.Timestamp;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Created by Miraculous on 15/7/5.
 */
@Service
@Slf4j
public class UserServiceImpl implements IUserService {

    @Autowired
    RedisTemplate<String, String> stringRedisTemplate;
    @Autowired
    private IUserRepository userRepository;

    @Autowired
    private IUserHashMappingRepository userHashMappingRepository;

    @Autowired
    private IUserDetailService userDetailService;


    @Override
    @Cacheable(value = "usercache", key = "'xyqbuser' + #phone", unless = "#result == null", cacheManager = "cacheManager")
    public User findByPhoneInDb(String phone) {
        return userRepository.findByPhoneNo(phone);
    }

    @Override
    public Map<Long, String> findPhoneByIdsInDb(List<Long> userIds) {
        if (CollectionUtils.isEmpty(userIds)) {
            return Maps.newHashMap();
        }
        Map<Long, String> userIdAndPhoneMap = Maps.newHashMap();
        List<User> users = userRepository.findByIdIn(userIds);
        users.forEach(user -> userIdAndPhoneMap.put(user.getId(), user.getPhoneNo()));
        return userIdAndPhoneMap;
    }

    @Override
    public User findByUuidInDb(String uuid) {
        return userRepository.findByUuid(uuid);
    }

    @Override
    public String findUuid(String phoneNo, String idNo) {
        // 优先按手机号查
        if(ValidationUtil.validatePhoneNo(phoneNo)){
            return userRepository.findUuidByPhoneNo(phoneNo);
        }
        // 按身份证号查
        if(StringUtils.isNotBlank(idNo)){
            return userRepository.findUuidByIdNo(idNo);
        }
        return null;
    }

    @Override
    @CacheEvict(value = "usercache", key = "'xyqbuser' + #user.phoneNo", cacheManager = "cacheManager")
    public User saveUser(User user) {
        return userRepository.saveAndFlush(user);
    }

    @Override
    public User findById(Long userId) {
        return userRepository.findById(userId);
    }

    @Override
    public List<User> findByPhones(List<String> phones) {
        return userRepository.findAll((root, query, cb) -> {
            query.where(root.get(Constants.PHONE_NO).as(String.class).in(phones));
            return query.getRestriction();
        });
    }

    @Override
    @Cacheable(value = "usercache", key = "'xyqbuser' + #phone", unless = "#result == null", cacheManager = "cacheManager")
    public User findByPhoneWithCache(String phone) {
        return userRepository.findByPhoneNo(phone);
    }

    @Override
    @Cacheable(value = "usercache", key = "'xyqbuser' + #uuid", unless = "#result == null", cacheManager = "cacheManager")
    public User findByUuidWithCache(String uuid) {
        return userRepository.findByUuid(uuid);
    }

    @Override
    public boolean exist(String phoneNo) {
        return userRepository.findByPhoneNo(phoneNo) != null;
    }

    /**
     * 修改用户密码
     *
     * @param phoneNo
     * @param password
     * @return
     * @date 2017-02-15 修改用户修改密码时，更新updatedAt时间
     */
    @Override
    @CacheEvict(value = "usercache", key = "'xyqbuser' + #phoneNo", cacheManager = "cacheManager")
    public boolean resetPassword(String phoneNo, String password) {

        User user = userRepository.findByPhoneNo(phoneNo);
        if (user == null) {
            throw new RuntimeException("用户[" + phoneNo + "]不存在");
        }
        user.setUpdatedAt(new Timestamp(System.currentTimeMillis()));
        user.setPassword(PasswordUtil.MD5(password.toLowerCase() + Constants.PASSWORD_SALT));
        user = userRepository.save(user);
        stringRedisTemplate.expire("usercache:xyqbuser" + phoneNo, 1L, TimeUnit.MILLISECONDS);
        return StringUtils.equals(PasswordUtil.MD5(password.toLowerCase() + Constants.PASSWORD_SALT), user.getPassword());

    }

    @Override
    public List<User> findRegisterUserByTime(String beginTime, String endTime) {
        return userRepository.findRegisterUserByTime(beginTime, endTime);
    }

    @Override
    @CacheEvict(value = "usercache", key = "'xyqbuser' + #phoneNo", cacheManager = "cacheManager")
    public int forbiddenUser(Boolean enable, String phoneNo) {
        return userRepository.forbiddenUser(enable, phoneNo);
    }


    @Override
    public List<UserInfo> findUserInfosByPhones(List<String> phones) {

        List<User> users = findByPhones(phones);
        if (CollectionUtils.isEmpty(phones)) {
            return Collections.emptyList();
        }

        List<UserDetail> userDetails = userDetailService.findByPhones(phones);

        if (!CollectionUtils.isEmpty(users)) {
            Map<Long, User> userMap = Maps.newHashMapWithExpectedSize(users.size());
            users.forEach(user -> {
                userMap.put(user.getId(), user);
            });

            if (!CollectionUtils.isEmpty(userDetails)) {
                List<UserInfo> userInfos = Lists.newArrayListWithExpectedSize(userDetails.size());
                userDetails.forEach(userDetail -> {
                    UserInfo userInfo = new UserInfo(userMap.get(userDetail.getUserId()), userDetail);
                    userInfos.add(userInfo);
                });
                return userInfos;
            }

        }
        return Collections.emptyList();
    }

    @Override
    public User findByPhoneMd5(String md5Value) {
        md5Value = md5Value.toLowerCase();
        long value = HashUtil.crc32(md5Value);
        UserHashMapping userHashMapping = userHashMappingRepository.findByPhoneNoMd5ShortAndPhoneNoMd5(value, md5Value);
        if (userHashMapping == null) {
            return null;
        }
        Long userId = userHashMapping.getUserId();
        return userRepository.findById(userId);
    }

}
