package cn.quantgroup.xyqb.service.session.impl;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.entity.Merchant;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.model.AuthBean;
import cn.quantgroup.xyqb.model.LoginProperties;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.model.session.SessionValue;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.service.session.aspect.UserBtRegisterFill;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.service.user.impl.UserServiceImpl;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Joiner;
import jdk.nashorn.internal.parser.JSONParser;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Caching;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.SetOperations;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.persistence.Cacheable;
import java.sql.Timestamp;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * Created by 11 on 2016/12/28.
 */
@Slf4j
@Service
public class SessionServiceImpl implements ISessionService {

    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> stringRedisTemplate;
    @Autowired
    private IUserService userService;

    @Override
    public AuthBean createSession(Long channelId, Long createdFrom, String appChannel, User user, Merchant merchant) {
        AuthBean authBean = new AuthBean();
        LoginProperties properties = new LoginProperties();
        properties.setAppChannel(appChannel);
        if (null != merchant) {
            properties.setMerchantName(merchant.getName());
        }
        properties.setChannelId(channelId);
        properties.setCreatedFrom(createdFrom);
        //找到用户
        String sessionId = findSessionIdByUserIdLoginProperties(user.getId(), properties);
        // String sessionId = sessionService.findSessionIdByUserIdAndMerchant(user.getId(), merchant);
        if (org.apache.commons.lang.StringUtils.isNotEmpty(sessionId)) {
            SessionStruct sessionStruct = findSessionBySessionId(sessionId);
            sessionStruct.setAttribute("channelId", String.valueOf(channelId));
            sessionStruct.setAttribute("createdFrom", String.valueOf(createdFrom));
            sessionStruct.setAttribute("appChannel", String.valueOf(appChannel));
            authBean.setPhoneNo(user.getPhoneNo());
            authBean.setToken(sessionStruct.getSid());
            persistSession(sessionStruct.getSid(), sessionStruct.getValues());
            log.info("用户登录成功, loginFrom:{}, phoneNo:{},appChannel:{},channelId:{}", createdFrom, user.getPhoneNo(), appChannel,channelId);
            return authBean;
        }
        SessionStruct sessionStruct = createSessionAndPersist(user, properties);
        authBean.setPhoneNo(user.getPhoneNo());
        authBean.setToken(sessionStruct.getSid());
        log.info("用户登录成功, loginFrom:{}, phoneNo:{},appChannel:{},channelId:{}", createdFrom, user.getPhoneNo(), appChannel,channelId);
        return authBean;
    }

    @Override
    public SessionStruct createSessionAndPersist(User user, LoginProperties properties) {
        SessionStruct sessionStruct;
        //获取sessionid
        String sessionId = findSessionIdByUserIdLoginProperties(user.getId(), properties);
        if (StringUtils.length(sessionId) == 36) {
            sessionStruct = findSessionBySessionId(sessionId);
            if (sessionStruct == null) {
                sessionStruct = newSession(user, properties);
            } else {
                sessionStruct.getValues().setLoginProperties(properties);
            }
            persistSession(sessionStruct.getSid(), sessionStruct.getValues());
        } else {
            sessionStruct = newSession(user, properties);
            persistSession(sessionStruct.getSid(), sessionStruct.getValues());
        }
        return sessionStruct;
    }

    @Override
    public String findSessionIdByUserIdLoginProperties(Long userId, LoginProperties properties) {
        return stringRedisTemplate.opsForValue().get(generateLoginPropertiesKey(userId, properties));
    }

    private String generateLoginPropertiesKey(Long userId, LoginProperties properties) {
        if ("baitiao".equals(properties.getMerchantName())) {
            return Constants.Session.USER_SESSION_ID_CACHE + ":" + String.valueOf(userId) + ":" + properties.getMerchantName() + ":" + properties.getCreatedFrom();
        } else {
            return Constants.Session.USER_SESSION_ID_CACHE + ":" + String.valueOf(userId) + ":" + properties.getMerchantName();
        }
    }

    @Override
    public String findSessionValueBySessionId(String sessionId) {
        String result = stringRedisTemplate.opsForValue().get(Constants.Session.USER_SESSION_CACHE + sessionId);
        return StringUtils.defaultString(result, "");
    }

    @Override
    public SessionStruct newSession(User user, LoginProperties loginProperties) {
        Timestamp now = new Timestamp(System.currentTimeMillis());
        SessionStruct sessionStruct = new SessionStruct();
        SessionValue sessionValue = new SessionValue();
        sessionStruct.setSid(UUID.randomUUID().toString());
        sessionValue.setCreatedAt(now);
        sessionValue.setLastAccessTime(now);
        sessionValue.setUser(user);
        sessionValue.setLoginProperties(loginProperties);
        Map<String, String> values = new HashMap<>();
        sessionValue.setValues(values);
        sessionStruct.setValues(sessionValue);
        return sessionStruct;
    }

    @Override
    @UserBtRegisterFill
    public void persistSession(String token, SessionValue sessionValue) {
        Timestamp current = new Timestamp(System.currentTimeMillis());
        sessionValue.setLastAccessTime(current);
        String json = JSON.toJSONString(sessionValue);
        stringRedisTemplate.opsForValue().set(Constants.Session.USER_SESSION_CACHE + token, json,
                Constants.Session.ONE_DAY, TimeUnit.SECONDS);
        String key = generateLoginPropertiesKey(sessionValue.getUser().getId(), sessionValue.getLoginProperties());
        stringRedisTemplate.opsForValue().set(key, token, Constants.Session.ONE_DAY, TimeUnit.SECONDS);

        setUserIdTokenKeys(sessionValue.getUser().getId(), key);

    }

    /**
     * 设置用户token集合方便注销使用
     *
     * @param userId
     * @param key
     */
    private void setUserIdTokenKeys(long userId, String key) {
        if (0L != userId) {
            try {
                stringRedisTemplate.opsForSet().add(Constants.Session.USER_SESSION_KEY_SET + userId, key);
                stringRedisTemplate.expire(Constants.Session.USER_SESSION_KEY_SET + userId, Constants.Session.ONE_DAY, TimeUnit.SECONDS);
            } catch (Exception e) {
                log.error("存储用户注销件失败，userId:{},Exception:{}", userId, e);
            }

        }
    }


    @Override
    public SessionStruct findSessionBySessionId(String sessionId) {
        String sessionValue = findSessionValueBySessionId(sessionId);
        if (StringUtils.isEmpty(sessionValue)) {
            return null;
        }
        try {
            SessionValue value = JSON.parseObject(sessionValue, SessionValue.class);
            if (null == value) {
                return null;
            }
            SessionStruct struct = new SessionStruct();
            struct.setSid(sessionId);
            struct.setValues(value);
            return struct;
        } catch (Exception ex) {
            return null;
        }

    }

    @Override
    @CacheEvict(value = "userextinfocache", key = "'extinfo' + #userId", cacheManager = "cacheManager")
    public void deleteByUserId(long userId) {
        //1.删除session关联
        Set useIdKeys = stringRedisTemplate.opsForSet().members(Constants.Session.USER_SESSION_KEY_SET + userId);
        if (!CollectionUtils.isEmpty(useIdKeys)) {
            useIdKeys.forEach(key -> {
                log.info("删除用户userId={}的缓存信息", userId);
                stringRedisTemplate.delete(String.valueOf(key));
            });
            //2.删除session缓存健
            stringRedisTemplate.delete(Constants.Session.USER_SESSION_KEY_SET + userId);


        }
        //3.删除用户查询缓存
        User user = userService.findById(userId);
        if (null != user) {
            deleteUserCatch(user);
        }


    }

    /**
     * 删除注销后缓存查询结果
     *
     * @param user
     */
    @Caching(evict = {@CacheEvict(value = "usercache", key = "'xyqbuser' + #user.phoneNo", cacheManager = "cacheManager"),
            @CacheEvict(value = "usercache", key = "'xyqbuser' + #user.uuid", cacheManager = "cacheManager"),
            @CacheEvict(value = "addresscache", key = "'address' + #user.id", cacheManager = "cacheManager"),
            @CacheEvict(value = "contact", key = "'contact' + #user.id", cacheManager = "cacheManager"),
            @CacheEvict(value = "userSpouseCache", key = "'spouse' + #user.id", cacheManager = "cacheManager"),
            @CacheEvict(value = "btRegisterCache", key = "'userId' + #user.id", cacheManager = "cacheManager")})
    private void deleteUserCatch(User user) {

    }
}
