package cn.quantgroup.xyqb.aspect.limit;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.session.XyqbSessionContextHolder;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.Enumeration;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * 免密访问校验切面
 *
 * @author 任文超
 * @version 1.0.0
 * @since 2017-11-21
 */
@Aspect
@Component
public class PasswordFreeAccessValidatorValidateAdvisor {

    private static final Logger LOGGER = LoggerFactory.getLogger(PasswordFreeAccessValidatorValidateAdvisor.class);
    private static final String PHONE_NO = "phoneNo";
    private static final String USER_ID = "userId";

    @Autowired
    @Qualifier("stringRedisTemplate")
    private RedisTemplate<String, String> redisTemplate;

    /**
     * 免密访问校验切面
     */
    @Pointcut("@annotation(cn.quantgroup.xyqb.aspect.limit.PasswordFreeAccessValidator)")
    private void passwordFreeAccess() {}

    /**
     * 执行免密访问校验
     *
     * @throws Throwable
     */
    @Around("passwordFreeAccess()")
    private Object checkToken(ProceedingJoinPoint pjp) throws Throwable {
        boolean valid = tokenValid();
        if (valid) {
            return pjp.proceed();
        }
        return JsonResult.buildErrorStateResult("操作失败，请重新提交请求", "");
    }

    /**
     * 校验免密访问
     * @return True or False
     */
    private boolean tokenValid() {
        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        Set<String> paramKeys = request.getParameterMap().keySet();
        if(!paramKeys.contains(PHONE_NO) && !paramKeys.contains(USER_ID)){
            LOGGER.info("缺少参数信息，请重新请求, paramKeys={}, clientIp={}", paramKeys, request.getRemoteAddr());
            return false;
        }
        // 当前请求的phoneNo/userId
        String phoneNo = request.getParameter(PHONE_NO);
        String userId = request.getParameter(USER_ID);
        if(StringUtils.isBlank(phoneNo) && StringUtils.isBlank(userId)){
            LOGGER.info("缺少参数信息，请重新请求, phoneNo={}, userId={}, clientIp={}", phoneNo, userId, request.getRemoteAddr());
            return false;
        }
        // 当前请求的Token
        String token = request.getHeader(Constants.X_AUTH_TOKEN);
        if (Objects.isNull(token) || token.length() != 36) {
            return false;
        }
        // 当前session
        SessionStruct session = XyqbSessionContextHolder.getXSessionFromRedis(token);
        if (Objects.isNull(session) || Objects.isNull(session.getValues()) || Objects.isNull(session.getValues().getUser())){
            return false;
        }
        // 当前用户
        User user = session.getValues().getUser();
        if(Objects.isNull(user.getId()) && StringUtils.isBlank(user.getPhoneNo())){
            return false;
        }
        // 校对用户信息是否匹配
        boolean valid = Objects.equals(userId, user.getId()) || Objects.equals(phoneNo, user.getPhoneNo());
        if(!valid) {
            LOGGER.info("非法请求, token={}, phoneNo=({},{}), userId=({},{}), clientIp={}", token, phoneNo, user.getPhoneNo(), userId, user.getId(), request.getRemoteAddr());
        }
        return valid;
    }

}
