package cn.quantgroup.xyqb.aspect.captcha;


import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.thirdparty.jcaptcha.AbstractManageableImageCaptchaService;
import com.octo.captcha.service.CaptchaServiceException;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.nio.charset.Charset;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

/**
 * 限次图形验证码校验标记
 * @author 任文超
 * @version 1.0.0
 * @since 2017-11-07
 */
@Aspect
@Component
public class CaptchaFiniteValidateAdvisor {

  private static final Logger LOGGER = LoggerFactory.getLogger(CaptchaFiniteValidateAdvisor.class);
  private static final String SUPER_CAPTCHA_ID = UUID.nameUUIDFromBytes("__QG_APPCLIENT_AGENT__".getBytes(Charset.forName("UTF-8"))).toString();
  private static final String SUPER_CAPTCHA = "__SUPERQG__";
  private static final String IMAGE_CAPTCHA_ID_COUNT = "image:captcha_id:count:";
  /**
   * 图形验证码错误计数器生命周期，与图形验证码生命周期保持一致
   * 参照点：cn.quantgroup.xyqb.config.captcha.RedisCaptchaStore{
   *   DEFAULT_EXPIRED_IN = 120L;
   *   DEFAULT_EXPIRED_TIMEUNIT = TimeUnit.SECONDS;
   * }
    */
  private static final Long SECONDS = 2 * 60L;

  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> redisTemplate;

  @Autowired
  @Qualifier("customCaptchaService")
  private AbstractManageableImageCaptchaService imageCaptchaService;

  /**
   * 自动化测试忽略验证码
   */
  @Value("${xyqb.auth.captcha.autotest.enable:false}")
  private boolean autoTestCaptchaEnabled;

  /**
   * 限次图形验证码切面
   */
  @Pointcut("@annotation(cn.quantgroup.xyqb.aspect.captcha.CaptchaFineteValidator)")
  private void needCaptchaFiniteValidate() {
  }

  /**
   * 在受保护的接口方法执行前, 执行限次图形验证码校验
   *
   * @param pjp
   * @return
   * @throws Throwable
   */
  @Around("needCaptchaFiniteValidate()")
  private Object doCapchaValidate(ProceedingJoinPoint pjp) throws Throwable {
    HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
    String registerFrom = Optional.ofNullable(request.getParameter("registerFrom")).orElse("");
    String captchaId = Optional.ofNullable(request.getParameter("captchaId")).orElse("");
    Object captchaValue = request.getParameter("captchaValue");

    if (shouldSkipCaptchaValidate(registerFrom, captchaId, captchaValue)) {
      LOGGER.info("使用超级图形验证码校验, registerFrom={}, clientIp={}", registerFrom, request.getRemoteAddr());
      return pjp.proceed();
    }
    if (captchaValue != null) {
      String captcha = String.valueOf(captchaValue);
      // 忽略用户输入的大小写
      captcha = StringUtils.lowerCase(captcha);
      // 验证码校验
      Boolean validCaptcha = false;
      Long captchaCount = countCaptchaId(captchaId);
      if(captchaCount == null || captchaCount >= Constants.IMAGE_FINITE_COUNT){
        return JsonResult.buildSuccessResult("验证码失效，请重新获取", "", 2L);
      }
      try {
        validCaptcha = imageCaptchaService.validateResponseForID(Constants.IMAGE_CAPTCHA_KEY + captchaId, captcha);
      } catch (CaptchaServiceException ex) {
        LOGGER.error("验证码校验异常, {}, {}", ex.getMessage(), ex);
      }
      if (validCaptcha) {
        return pjp.proceed();
      }
    }
    return JsonResult.buildSuccessResult("图形验证码不正确", "", 2L);
  }

  private boolean shouldSkipCaptchaValidate(String registerFrom, String captchaId, Object captchaValue) {
    // 如果启用了超级验证码功能, 检查超级验证码, 超级验证码区分大小写
    if (autoTestCaptchaEnabled) {
      return true;
    }
    return StringUtils.equals(SUPER_CAPTCHA_ID, String.valueOf(captchaId)) && StringUtils.equals(SUPER_CAPTCHA, String.valueOf(captchaValue));
  }

  private Long countCaptchaId(String captchaId) {
    String captchaKey = getKey(captchaId);
    if(StringUtils.isBlank(captchaKey)){
      return null;
    }
    if (!redisTemplate.hasKey(captchaKey)) {
      redisTemplate.opsForValue().set(captchaKey, String.valueOf(0), SECONDS, TimeUnit.SECONDS);
    }
    return redisTemplate.opsForValue().increment(captchaKey, 1L);
  }

  private final static String getKey(String captchaId){
    if(StringUtils.isBlank(captchaId)){
      return null;
    }
    return IMAGE_CAPTCHA_ID_COUNT + captchaId;
  }

}
