package cn.quantgroup.xyqb.controller.internal.user.center;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.aspect.forbidden.AccessForbiddenValidator;
import cn.quantgroup.xyqb.aspect.limit.PasswordFreeAccessValidator;
import cn.quantgroup.xyqb.constant.UserConstant;
import cn.quantgroup.xyqb.controller.IBaseController;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserInfoEntity;
import cn.quantgroup.xyqb.model.JsonResult;
import cn.quantgroup.xyqb.model.UserInfoEntityBean;
import cn.quantgroup.xyqb.repository.IUserInfoRepository;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.util.HttpClientUtil;
import cn.quantgroup.xyqb.util.QiNiuYunUtil;
import cn.quantgroup.xyqb.util.TenantUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.*;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Created by 11 on 2017/3/22.
 */
@Slf4j
@RestController
@RequestMapping("/user/center")
public class UserCenterController implements IBaseController {

    @Autowired
    private IUserInfoRepository userInfoRepository;

    @Autowired
    private IUserService userService;
    @Autowired
    private ApplicationEventPublisher applicationEventPublisher;
    @Autowired
    private QiNiuYunUtil qiNiuYunUtil;

    private ExecutorService executorService = Executors.newFixedThreadPool(10);

    private static final java.util.regex.Pattern pattern = java.util.regex.Pattern.compile("(\\d{11})");

    @GetMapping("/attach")
    public JsonResult queryUserAttach(Long userId, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId
    ) {
        if (userId == null) {
            return JsonResult.buildErrorStateResult("用户 ID 不能为空", null);
        }
        //UserAttached userAttached = userCenterService.searchUserAttachedByUserId(userId,tenantId);
        UserInfoEntity userAttached = userInfoRepository.findByUserIdAndTenantId(userId, tenantId);
        return JsonResult.buildSuccessResultGeneric(UserInfoEntityBean.covert(userAttached));
    }

    /**
     * @param userIds
     * @return
     * @yapi http://yapi.quantgroups.com/project/17/interface/api/17291
     */
    @PostMapping("/attach/list")
    public JsonResult queryUserAttachList(@RequestBody List<Long> userIds, @RequestParam(required = false) Integer tenantId) {
        //默认羊小咩租户
        if (tenantId == null) {
            tenantId = getTenantId();
        }
        if (CollectionUtils.isEmpty(userIds)) {
            return JsonResult.buildErrorStateResult("用户 ID 不能为空", null);
        }
        int size = 500;
        if (userIds.size() > size) {
            return JsonResult.buildErrorStateResult("超出最大条数限制" + size, null);
        }

        if (TenantUtil.validationTenantIdIsNullOrZero(tenantId)) {
            tenantId = TenantUtil.TENANT_DEFAULT;
        }
        //List<UserAttached> userAttachedList = userCenterService.searchUserAttachedListByUserId(userIds, tenantId);
        List<UserInfoEntity> userInfoEntityList = userInfoRepository.findByTenantIdAndUserIdIn(tenantId,userIds);
        return JsonResult.buildSuccessResultGeneric(UserInfoEntityBean.covert(userInfoEntityList));
    }


    /**
     * 根据用户手机号查询昵称.
     *
     * @param phoneNo
     * @return
     */
    @RequestMapping("/queryNick")
    public JsonResult queryUserNick(String phoneNo, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isEmpty(phoneNo)) {
            log.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo, tenantId);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }

        /*UserAttached userAttached = userCenterService.searchUserAttachedByUserId(userId,tenantId);
        if (null != userAttached) {
            log.info("获取用户昵称:{}", userAttached.getNick());
            return JsonResult.buildSuccessResult(null, userAttached.getNick());
        }*/

        UserInfoEntity userInfoEntity = userInfoRepository.findByUserIdAndTenantId(userId,tenantId);
        if (null != userInfoEntity) {
            return JsonResult.buildSuccessResult(null, userInfoEntity.getNickName());
        }
        return JsonResult.buildSuccessResult(null, null);
    }

    /**
     * 修改用户的昵称
     *
     * @param phoneNo
     * @param nick
     * @return
     */
    @RequestMapping("/saveNick")
    public JsonResult saveUserNick(String phoneNo, String nick, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isEmpty(phoneNo)) {
            log.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo, tenantId);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        /*UserAttached userAttached = userCenterService.saveUserNick(userId, nick,tenantId);
        return JsonResult.buildSuccessResult(null, userAttached.getNick());*/
        UserInfoEntity userInfoEntity = userInfoRepository.findByUserIdAndTenantId(userId,tenantId);
        if(userInfoEntity ==null){
            userInfoEntity = new UserInfoEntity();
            userInfoEntity.setUserId(userId);
            userInfoEntity.setTenantId(tenantId);
        }
        userInfoEntity.setNickName(nick);
        userInfoRepository.save(userInfoEntity);
        return JsonResult.buildSuccessResult(null, userInfoEntity.getNickName());
    }

    @RequestMapping("/kdsp/saveNick")
    public JsonResult saveUserNickForKdsp(Long userId, String nick, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        /*UserAttached userAttached = userCenterService.saveUserNick(userId, nick,tenantId);
        return JsonResult.buildSuccessResult(null, userAttached.getNick());*/

        UserInfoEntity userInfoEntity = userInfoRepository.findByUserIdAndTenantId(userId,tenantId);
        if(userInfoEntity ==null){
            userInfoEntity = new UserInfoEntity();
            userInfoEntity.setUserId(userId);
            userInfoEntity.setTenantId(tenantId);
        }
        userInfoEntity.setNickName(nick);
        userInfoRepository.save(userInfoEntity);
        return JsonResult.buildSuccessResult(null,  userInfoEntity.getNickName());
    }

    @RequestMapping("/kdsp/save/avatar")
    public JsonResult saveUserAvatarAddrForKdsp(Long userId, String avatarUrl, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isBlank(avatarUrl)) {
            log.error("参数不合法:avatarUrl:{}}", avatarUrl);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        /*UserAttached userAttached = userCenterService.saveUserAvatar(userId, avatarUrl,tenantId);
        if (null == userAttached) {
            log.error("保存用户头像昵称失败.");
            return JsonResult.buildErrorStateResult("信息保存失败,请稍后再试.", null);
        }
        return JsonResult.buildSuccessResult("保存成功", userAttached);*/

        UserInfoEntity userInfoEntity = userInfoRepository.findByUserIdAndTenantId(userId,tenantId);
        if(userInfoEntity ==null){
            userInfoEntity = new UserInfoEntity();
            userInfoEntity.setUserId(userId);
            userInfoEntity.setTenantId(tenantId);
        }
        userInfoEntity.setPhoto(avatarUrl);
        userInfoRepository.save(userInfoEntity);

        return JsonResult.buildSuccessResult("保存成功", UserInfoEntityBean.covert(userInfoEntity));
    }

    /**
     * 上传后调用该接口保存用户头像
     *
     * @param phoneNo   用户手机号
     * @param avatarUrl 上传后生成的头像地址
     * @return
     */
    @RequestMapping("/save/avatar")
    public JsonResult SaveUserAvatarAddr(String phoneNo, String avatarUrl, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isBlank(avatarUrl) || StringUtils.isBlank(phoneNo)) {
            log.error("参数不合法:avatarUrl:{}, phoneNo:{}", avatarUrl, phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo, tenantId);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        if (StringUtils.isBlank(avatarUrl)) {
            avatarUrl = Constants.UserAvatar.AVATAR_DEFAULT;
        }
        /*UserAttached userAttached = userCenterService.saveUserAvatar(userId, avatarUrl,tenantId);
        if (null == userAttached) {
            log.error("保存用户头像昵称失败.");
            return JsonResult.buildErrorStateResult("信息保存失败,请稍后再试.", null);
        }
        return JsonResult.buildSuccessResult("保存成功", userAttached);*/

        UserInfoEntity userInfoEntity = userInfoRepository.findByUserIdAndTenantId(userId,tenantId);
        if(userInfoEntity ==null){
            userInfoEntity = new UserInfoEntity();
            userInfoEntity.setUserId(userId);
            userInfoEntity.setTenantId(tenantId);
        }
        userInfoEntity.setPhoto(avatarUrl);
        userInfoRepository.save(userInfoEntity);

        return JsonResult.buildSuccessResult("保存成功", UserInfoEntityBean.covert(userInfoEntity));
    }



    /**
     * 查询用户是否实名认证.
     *
     * @param phoneNo
     * @return
     */
    @RequestMapping("/query/verified")
    public JsonResult queryVerified(String phoneNo, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isEmpty(phoneNo)) {
            log.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo, tenantId);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        UserInfoEntity userDetail = userInfoRepository.findByUserIdAndTenantId(userId, tenantId);
        if (null != userDetail) {
            return JsonResult.buildSuccessResult(null, userDetail);
        }
        return JsonResult.buildSuccessResult(null, null);
    }


    /**
     * 个人资料信息
     *
     * @param phoneNo
     * @return
     */
    @AccessForbiddenValidator
    @PasswordFreeAccessValidator
    @RequestMapping("/personalData")
    public JsonResult personalData(String phoneNo, @RequestHeader(value = Constants.X_AUTH_TENANT, defaultValue = UserConstant.defaultTenantIdString) Integer tenantId) {
        if (StringUtils.isBlank(phoneNo)) {
            log.error("手机号为空,phoneNo:{}", phoneNo);
            return JsonResult.buildErrorStateResult("参数不合法", null);
        }
        Long userId = queryUserId(phoneNo, tenantId);
        if (null == userId || userId == 0L) {
            return JsonResult.buildErrorStateResult("该用户不存在", null);
        }
        Map<String, Object> result = new HashMap<>();

        return JsonResult.buildSuccessResult(null, result);
    }


    /**
     * 根据手机号查询用户id, nodejs调用接口无法获取到userId.
     * 所以增加该接口来查询用户id
     *
     * @param phoneNo
     * @return
     */
    private Long queryUserId(String phoneNo, Integer tenantId) {
        User user = userService.findByPhoneInDb(phoneNo, tenantId);
        return null == user ? null : user.getId();
    }

    /*@GetMapping("/attach/listLimit/{id}")
    public JsonResult queryUserAttachListLimit(@PathVariable Long id) {
        //默认羊小咩租户
        List<UserAttached> userAttachedList = userCenterService.queryUserAttachListLimit(id);
        return JsonResult.buildSuccessResultGeneric(userAttachedList);
    }*/


    @RequestMapping("/refreshAvatar")
    public JsonResult refreshAvatar(@RequestParam(value = "id",required = false) Long id){
        if(Objects.isNull(id)){
            id=0L;
        }
        List<UserInfoEntity> allUserInfo = userInfoRepository.getAllUserInfo(id);
        while (allUserInfo.size()>0){
            long l = System.currentTimeMillis();
            CountDownLatch countDown = new CountDownLatch(allUserInfo.size());
            for (UserInfoEntity userInfoEntity : allUserInfo) {
                executorService.execute(() -> {
                    try {
                        String avatarUrl = userInfoEntity.getPhoto();
                        if(StringUtils.isEmpty(avatarUrl)){
                            return;
                        }
                        if(StringUtils.isEmpty(extractPhoneNumberFromUrl(avatarUrl))){
                            return;
                        }
                        byte[] avatarBytes = HttpClientUtil.download(avatarUrl);
                        String fileName = qiNiuYunUtil.upload(avatarBytes, ".jpg");
                        userInfoEntity.setPhoto(fileName);
                        userInfoRepository.save(userInfoEntity);
                    } catch (Exception e) {
                        log.error("refeshAvatar.error",e);
                    }finally {
                        countDown.countDown();
                    }
                });
            }
            try {
                countDown.await();
            } catch (InterruptedException e) {
                log.error("refeshAvatar.error",e);
            }
            UserInfoEntity userInfoEntity = allUserInfo.get(allUserInfo.size() - 1);
            log.info("refeshAvatar.执行完的最大的id是:{},执行时间是:{}",userInfoEntity.getId(),System.currentTimeMillis()-l);
            id=userInfoEntity.getId();
            allUserInfo = userInfoRepository.getAllUserInfo(id);
        }
        return JsonResult.buildSuccessResult("刷新成功", null);

    }


    public static String extractPhoneNumberFromUrl(String url) {
        // 使用正则表达式匹配手机号
        java.util.regex.Matcher matcher = pattern.matcher(url);

        if (matcher.find()) {
            return matcher.group(1);
        }

        return null;
    }
}
