package cn.quantgroup.xyqb.service.security.impl;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.entity.User;
import cn.quantgroup.xyqb.entity.UserDetail;
import cn.quantgroup.xyqb.exception.UserNotExistException;
import cn.quantgroup.xyqb.model.AppUserParam;
import cn.quantgroup.xyqb.model.AuthenticationUserDetail;
import cn.quantgroup.xyqb.model.IdCardInfo;
import cn.quantgroup.xyqb.model.IdType;
import cn.quantgroup.xyqb.repository.IUserDetailRepository;
import cn.quantgroup.xyqb.repository.IUserRepository;
import cn.quantgroup.xyqb.service.auth.IIdCardService;
import cn.quantgroup.xyqb.service.sms.ISmsService;
import cn.quantgroup.xyqb.service.user.ILkbUserService;
import cn.quantgroup.xyqb.service.user.IUserService;
import cn.quantgroup.xyqb.util.PasswordUtil;
import cn.quantgroup.xyqb.util.ValidationUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.userdetails.AuthenticationUserDetailsService;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.web.authentication.preauth.PreAuthenticatedAuthenticationToken;
import org.springframework.stereotype.Service;

import java.sql.Timestamp;
import java.util.Random;

/**
 * @author mengfan.feng
 * @time 2015-10-26 16:00
 */
@Service
public class UserDetailsByPhoneService implements AuthenticationUserDetailsService<PreAuthenticatedAuthenticationToken> {

    private static final Logger LOGGER = LoggerFactory.getLogger(UserDetailsByPhoneService.class);

    @Autowired
    private IUserService userService;

    @Autowired
    private ILkbUserService lkbUserService;

    @Autowired
    private ISmsService smsService;
    @Autowired
    private IUserRepository userRepository;
    @Autowired
    private IIdCardService idCardService;
    @Autowired
    private IUserDetailRepository userDetailRepository;



    private static final Random random = new Random();

    @Override
    public UserDetails loadUserDetails(PreAuthenticatedAuthenticationToken token) throws UsernameNotFoundException {
        AppUserParam param = (AppUserParam) token.getPrincipal();
        User user = userService.findByPhone(param.getPhoneNo());

        if (user == null) {
            LOGGER.info("第三方登录用户不存在，开始注册，{}", param);
            user = this.register(param);
        }

        if (user == null) {
            throw new UserNotExistException(String.format("用户未找到, phone:%s", param.getPhoneNo()));
        }

        return new AuthenticationUserDetail(user);
    }

    /**
     * 注册新用户
     *
     * @param param
     * @return
     */
    private User register(AppUserParam param) {
        if (!ValidationUtil.validatePhoneNo(param.getPhoneNo())) {
            return null;
        }

        String randomCode = String.valueOf(random.nextInt(899999) + 100000);

        String uuid = lkbUserService.registerApp(param.getPhoneNo(), randomCode);
        Timestamp currentTime = new Timestamp(System.currentTimeMillis());

        LOGGER.info("第三方登录用户，保存 User");
        User user = new User();
        user.setEnable(true);
        user.setPhoneNo(param.getPhoneNo());
        user.setUpdatedAt(currentTime);
        user.setCreatedAt(currentTime);
        user.setUuid(uuid);
        user.setPassword(PasswordUtil.MD5(randomCode + Constants.PASSWORD_SALT));
        user.setRegisteredFrom(param.getRegisterFrom());

        user = userRepository.save(user);
        if (user == null) {
            throw new RuntimeException("保存用户出错");
        }
        smsService.sendAfterRegister(param.getPhoneNo());

        LOGGER.info("第三方登录用户注册成功, registerFrom:{}, phoneNo:{}, 并且已发送短信通知", param.getRegisterFrom(), param.getPhoneNo());

        if (StringUtils.isNotEmpty(param.getIdNo())) {
            LOGGER.info("第三方登录用户，保存 UserDetail");

            IdCardInfo cardInfo;
            try {
                cardInfo = idCardService.getIdCardInfoWithExceptions(param.getIdNo());
                UserDetail userDetail = new UserDetail();
                userDetail.setIdNo(cardInfo.getIdNo());
                userDetail.setPhoneNo(param.getPhoneNo());
                userDetail.setName(param.getName());
                userDetail.setIdType(IdType.ID_CARD);
                userDetail.setUpdatedAt(currentTime);
                userDetail.setCreatedAt(currentTime);
                userDetail.setUserId(user.getId());
                userDetail.setGender(cardInfo.getGender());

                userDetailRepository.saveAndFlush(userDetail);

                lkbUserService.userUpdate(user.getUuid(), param.getName(), cardInfo.getIdNo());
            } catch (Exception e) {
                LOGGER.error("保存 UserDetail 出现异常", e);
            }
        }

        return user;
    }

}
