package cn.quantgroup.xyqb.util;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;

/**
 * XML <=> Object
 *
 * @author mengfan.feng
 * @time 2015-07-25 18:47
 */
@SuppressWarnings("unchecked")
public class XmlUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(XmlUtil.class);

    /**
     * Disable new
     */
    private XmlUtil() {
    }

    /**
     * XML => Object
     *
     * @param xml
     * @param clazz
     * @param <T>
     * @return
     */
    public static <T> T obj(String xml, Class<T> clazz) {
        if (StringUtils.isEmpty(xml)) {
            return null;
        }

        Reader reader = null;
        try {
            JAXBContext context = JAXBContext.newInstance(clazz);
            Unmarshaller unmarshaller = context.createUnmarshaller();
            reader = new StringReader(xml);
            return (T) unmarshaller.unmarshal(reader);
        } catch (Exception e) {
            LOGGER.error("Parse XML to Object error\nXML:{}\nClass<T>:{}", xml, clazz.getName(), e);
            return null;
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (Exception e) {
                    LOGGER.error("Close Reader error", e);
                }
            }
        }
    }

    /**
     * Object => XML
     *
     * @param obj
     * @param clazz
     * @param <T>
     * @return
     */
    public static <T> String xml(Object obj, Class<T> clazz) {
        return doXml(obj, clazz, "UTF-8", false);
    }

    /**
     * Object => XML
     *
     * @param obj
     * @param clazz
     * @param charset
     * @param <T>
     * @return
     */
    public static <T> String xml(Object obj, Class<T> clazz, String charset) {
        return doXml(obj, clazz, charset, false);
    }

    /**
     * Object => XML pretty
     *
     * @param obj
     * @param clazz
     * @param <T>
     * @return
     */
    public static <T> String xmlPretty(Object obj, Class<T> clazz) {
        return doXml(obj, clazz, "UTF-8", true);
    }

    /**
     * Object => XML pretty
     *
     * @param obj
     * @param clazz
     * @param charset
     * @param <T>
     * @return
     */
    public static <T> String xmlPretty(Object obj, Class<T> clazz, String charset) {
        return doXml(obj, clazz, charset, true);
    }

    /**
     * Object => XML
     *
     * @param obj
     * @param clazz
     * @param pretty
     * @param <T>
     * @return
     */
    private static <T> String doXml(Object obj, Class<T> clazz, String charset, boolean pretty) {
        if (obj == null || clazz == null) {
            return null;
        }

        StringWriter writer = null;
        try {
            JAXBContext context = JAXBContext.newInstance(clazz);
            Marshaller marshaller = context.createMarshaller();
            marshaller.setProperty(Marshaller.JAXB_ENCODING, charset);
            marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, pretty);
            writer = new StringWriter();
            marshaller.marshal(obj, writer);
            return writer.toString();
        } catch (Exception e) {
            LOGGER.error("Parse Object to XML error\nXML:{}\nClass<T>:{}", obj.getClass().getName(), clazz.getName(), e);
            return null;
        } finally {
            if (writer != null) {
                try {
                    writer.close();
                } catch (Exception e) {
                    LOGGER.error("Close Writer error", e);
                }
            }
        }
    }
}
