package cn.quantgroup.xyqb.filter;

import cn.hutool.core.date.DateUtil;
import cn.hutool.extra.servlet.ServletUtil;
import cn.quantgroup.xyqb.util.IpUtil;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.core.StringRedisTemplate;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import static cn.quantgroup.xyqb.controller.api.v2.IpLimitController.IP_LIMIT_STATUS_KEY;

/**
 * @author hyuk
 */
@Slf4j
@RequiredArgsConstructor
@Configuration
public class IpLimitFilter implements Filter {
    
    @Getter
    @Value("${extranet.known.ips:123.56.31.54}")
    private String[] extranetIps;
    
    private final StringRedisTemplate stringRedisTemplate;
    
    // 1. 日 IP 访问次数
    static final String DAY_IP_COUNT_KEY = "xyqb_user:01:ip_limit:day_count:00cs:%s";
    
    // 2. 区间请求次数
    static final String IP_REQ_ZSET_COUNT_KEY = "xyqb_user:01:ip_REQ_ZSET_COUNT:03cs:%s";
    
    // 500ms
    static final long REQUEST_LIMIT_TIME = TimeUnit.MINUTES.toMillis(1L);
    
    private static String getDayIpCountKey() {
        String day = DateUtil.format(DateUtil.date(), "yyyyMMdd");
        return String.format(DAY_IP_COUNT_KEY, day);
    }
    
    private static String getIpReqZSetCountKey(String ip) {
        return String.format(IP_REQ_ZSET_COUNT_KEY, ip);
    }
    
    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
    
    }
    
    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain)
            throws IOException, ServletException {
        HttpServletRequest request = (HttpServletRequest) servletRequest;
        HttpServletResponse response = (HttpServletResponse) servletResponse;
        
        String clientIp = ServletUtil.getClientIP(request);
        // finance-gateway Ip
        if (IpUtil.whiteOf(clientIp) || (StringUtils.isNotEmpty(clientIp) && isInExtranet(clientIp))) {
            // 在 IP 白名单里
            filterChain.doFilter(servletRequest, servletResponse);
            return;
        }
        log.info("[IpLimitFilter]外网调用, clientIp : {}, uri : {}", clientIp, request.getRequestURI());
        String uri = request.getRequestURI();
        if (Objects.equals(uri, "/api/captcha/new")) {
            response.setStatus(HttpServletResponse.SC_FORBIDDEN);
            return;
        }
        filterChain.doFilter(servletRequest, servletResponse);
        return;
        
        
//        String status = stringRedisTemplate.opsForValue().get(IP_LIMIT_STATUS_KEY);
//        if (status == null || Objects.equals(status, "1")) {
//            filterChain.doFilter(servletRequest, servletResponse);
//        } else {
//            response.setStatus(HttpServletResponse.SC_FORBIDDEN);
//        }
//        return;
//        try {
//            // 1. 一天内请求多少次
//            // 2. 1 分钟请求次数超过 5 次
//            String dayIpCountKey = getDayIpCountKey();
//            stringRedisTemplate.opsForHash().increment(dayIpCountKey, clientIp, 1L);
//            stringRedisTemplate.expire(dayIpCountKey, 2L, TimeUnit.DAYS);
//
//            long currentTime = System.currentTimeMillis();
//            String ipReqZSetCountKey = getIpReqZSetCountKey(clientIp);
//            stringRedisTemplate.opsForZSet().add(ipReqZSetCountKey, currentTime + "", currentTime);
//            stringRedisTemplate.expire(ipReqZSetCountKey, 12L, TimeUnit.HOURS);
//            // 1 分钟内请求次数
//            long reqCount = stringRedisTemplate.opsForZSet().count(ipReqZSetCountKey, currentTime - REQUEST_LIMIT_TIME, currentTime);
//            int limit =  5;
//            if (reqCount >= limit) {
//                log.warn("[IpLimitFilter]clientIp一分钟内请求次数超过 {} 次, clientIp : {}, count : {}", limit, clientIp, reqCount);
//            }
//        } catch (Exception e) {
//            log.error("[IpLimitFilter] ip 请求维度统计出现异常, clientIp : {}, uri : {}",  clientIp, request.getRequestURI(), e);
//        } finally {
//            filterChain.doFilter(servletRequest, servletResponse);
//        }
    }
    
    private boolean isInExtranet(String clientIp) {
        for (String extranetIp : extranetIps) {
            if (clientIp.startsWith(extranetIp)) {
                return true;
            }
        }
        return false;
    }
    
    @Override
    public void destroy() {
    
    }
}
