package cn.quantgroup.xyqb.filter;

import cn.quantgroup.xyqb.Constants;
import cn.quantgroup.xyqb.exception.BizException;
import cn.quantgroup.xyqb.exception.BizExceptionEnum;
import cn.quantgroup.xyqb.model.session.SessionStruct;
import cn.quantgroup.xyqb.service.session.ISessionService;
import cn.quantgroup.xyqb.session.XyqbSessionContextHolder;
import cn.quantgroup.xyqb.util.StringUtils;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.HashSet;

/**
 * 用户中心签发的token的拦截器
 */
public class TokenInterceptor implements HandlerInterceptor {
    private final HashSet<String> excludes = new HashSet<>();
    private final ISessionService sessionService;

    public TokenInterceptor(ISessionService sessionService) {
        this.sessionService = sessionService;
        excludes.add("/v2/oauth/login");
        excludes.add("/v2/sms/code");
        excludes.add("/v2/sms/verify");
    }


    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        //所有开放出去的外部接口，都需要验证租户id和注册来源
        String tenantId = request.getHeader(Constants.X_AUTH_TENANT);
        String registered_from = request.getHeader(Constants.X_AUTH_FROM);
        if (StringUtils.isEmpty(tenantId)) {
            throw new BizException(BizExceptionEnum.UN_EXIT_TENANT_ID);
        }
        if (StringUtils.isEmpty(registered_from)) {
            throw new BizException(BizExceptionEnum.UN_EXIT_REGISTERED_FROM);
        }

        //如果不再排查excludes的接口，需要进一步校验token（说明这些接口需要登录）
        SessionStruct sessionStruct = XyqbSessionContextHolder.getXSessionFromRedis();
        if (!excludes.contains(request.getRequestURI())) {
            if (sessionStruct == null) {
                throw new BizException(BizExceptionEnum.UN_AUTHORIZED);
            } else {
                //session续期
                sessionService.persistSession(sessionStruct.getSid(), sessionStruct.getValues(), sessionStruct.getTenantId());
            }
        } else {
            //如果是排除的接口，同时sessionStruct为空，这时候初始化租户和来源字段
            if (sessionStruct == null) {
                XyqbSessionContextHolder.initXSession();
            }
        }

        return true;
    }

    @Override
    public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler, ModelAndView modelAndView) throws Exception {

    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
        XyqbSessionContextHolder.releaseSession();
    }
}
